# Wasm builder is a utility for building a project as a Wasm binary

The Wasm builder is a tool that integrates the process of building the WASM binary of your project into the main
`cargo` build process.

## Project setup

A project that should be compiled as a Wasm binary needs to:

1. Add a `build.rs` file.
2. Add `wasm-builder` as dependency into `build-dependencies`.

The `build.rs` file needs to contain the following code:

```rust
use wasm_builder::WasmBuilder;

fn main() {
    WasmBuilder::new()
        // Tell the builder to build the project (crate) this `build.rs` is part of.
        .with_current_project()
        // Make sure to export the `heap_base` global, this is required by Tetcore
        .export_heap_base()
        // Build the Wasm file so that it imports the memory (need to be provided by at instantiation)
        .import_memory()
        // Build it.
        .build()
}
```

As the final step, you need to add the following to your project:

```rust
include!(concat!(env!("OUT_DIR"), "/wasm_binary.rs"));
```

This will include the generated Wasm binary as two constants `WASM_BINARY` and `WASM_BINARY_BLOATY`.
The former is a compact Wasm binary and the latter is the Wasm binary as being generated by the compiler.
Both variables have `Option<&'static [u8]>` as type.

### Features

Wasm builder supports to enable cargo features while building the Wasm binary. By default it will
enable all features in the wasm build that are enabled for the native build except the
`default` and `std` features. Besides that, wasm builder supports the special `runtime-wasm`
feature. This `runtime-wasm` feature will be enabled by the wasm builder when it compiles the
Wasm binary. If this feature is not present, it will not be enabled.

## Environment variables

By using environment variables, you can configure which Wasm binaries are built and how:

- `SKIP_WASM_BUILD` - Skips building any Wasm binary. This is useful when only native should be recompiled.
                      If this is the first run and there doesn't exist a Wasm binary, this will set both
                      variables to `None`.
- `WASM_BUILD_TYPE` - Sets the build type for building Wasm binaries. Supported values are `release` or `debug`.
                      By default the build type is equal to the build type used by the main build.
- `FORCE_WASM_BUILD` - Can be set to force a Wasm build. On subsequent calls the value of the variable
                       needs to change. As wasm-builder instructs `cargo` to watch for file changes
                       this environment variable should only be required in certain circumstances.
- `WASM_BUILD_RUSTFLAGS` - Extend `RUSTFLAGS` given to `cargo build` while building the wasm binary.
- `WASM_BUILD_NO_COLOR` - Disable color output of the wasm build.
- `WASM_TARGET_DIRECTORY` - Will copy any build Wasm binary to the given directory. The path needs
                           to be absolute.
- `WASM_BUILD_TOOLCHAIN` - The toolchain that should be used to build the Wasm binaries. The
                           format needs to be the same as used by cargo, e.g. `nightly-2020-02-20`.

Each project can be skipped individually by using the environment variable `SKIP_PROJECT_NAME_WASM_BUILD`.
Where `PROJECT_NAME` needs to be replaced by the name of the cargo project, e.g. `node-runtime` will
be `NODE_RUNTIME`.

## Prerequisites:

Wasm builder requires the following prerequisites for building the Wasm binary:

- rust nightly + `wasm32-unknown-unknown` toolchain

If a specific rust nightly is installed with `rustup`, it is important that the wasm target is installed
as well. For example if installing the rust nightly from 20.02.2020 using `rustup install nightly-2020-02-20`,
the wasm target needs to be installed as well `rustup target add wasm32-unknown-unknown --toolchain nightly-2020-02-20`.

License: Apache-2.0
