// TODO: Can we refactor this code in a more declarative manner?

use crate::{Address, Flags, Object, Timeval, TraceProbe, TraceStopReason, TraceType};
use std::ffi::CString;
use std::mem::{size_of, size_of_val};

pub trait WartsSizeable {
    /// Returns the size in bytes of a value in a warts file.
    fn warts_size(&self) -> u16;
}

impl<T: WartsSizeable> WartsSizeable for Option<T> {
    fn warts_size(&self) -> u16 {
        return if let Some(x) = self {
            x.warts_size()
        } else {
            0
        };
    }
}

impl WartsSizeable for u32 {
    fn warts_size(&self) -> u16 {
        size_of::<u32>() as u16
    }
}

impl WartsSizeable for u16 {
    fn warts_size(&self) -> u16 {
        size_of::<u16>() as u16
    }
}

impl WartsSizeable for CString {
    fn warts_size(&self) -> u16 {
        self.to_bytes_with_nul().len() as u16
    }
}

impl WartsSizeable for Flags {
    fn warts_size(&self) -> u16 {
        // TODO.
        8
    }
}

pub struct ListBuilder {
    list_id: u32,
    list_id_human: u32,
    name: CString,
    description: Option<CString>,
    monitor_name: Option<CString>,
}

impl ListBuilder {
    pub fn new() -> Self {
        ListBuilder {
            list_id: 0,
            list_id_human: 0,
            name: Default::default(),
            description: None,
            monitor_name: None,
        }
    }
    pub fn list_id(&mut self, x: u32) -> &mut Self {
        self.list_id = x;
        self
    }
    pub fn list_id_human(&mut self, x: u32) -> &mut Self {
        self.list_id_human = x;
        self
    }
    pub fn name(&mut self, x: &str) -> &mut Self {
        self.name = CString::new(x).unwrap();
        self
    }
    pub fn description(&mut self, x: &str) -> &mut Self {
        // TODO: Set flags appropriately.
        self.description = Some(CString::new(x).unwrap());
        self
    }
    pub fn monitor_name(&mut self, x: &str) -> &mut Self {
        self.monitor_name = Some(CString::new(x).unwrap());
        self
    }
    pub fn build(&mut self) -> Object {
        let flags = Flags::new(0);
        let param_length = 0;
        let length: u32 = (self.list_id.warts_size()
            + self.list_id_human.warts_size()
            + self.name.warts_size()
            + flags.warts_size()
            + param_length) as u32;
        Object::List {
            length,
            list_id: self.list_id,
            list_id_human: self.list_id_human,
            name: self.name.clone(),
            flags,
            param_length: Some(param_length),
            description: self.description.clone(),
            monitor_name: self.monitor_name.clone(),
        }
    }
}

pub struct CycleStartBuilder {
    cycle_id: u32,
    list_id: u32,
    cycle_id_human: u32,
    start_time: u32,
    stop_time: Option<u32>,
    hostname: Option<CString>,
}

impl CycleStartBuilder {
    pub fn new() -> Self {
        CycleStartBuilder {
            cycle_id: 0,
            list_id: 0,
            cycle_id_human: 0,
            start_time: 0,
            stop_time: None,
            hostname: None,
        }
    }
    pub fn cycle_id(&mut self, x: u32) -> &mut Self {
        self.cycle_id = x;
        self
    }
    pub fn list_id(&mut self, x: u32) -> &mut Self {
        self.list_id = x;
        self
    }
    pub fn cycle_id_human(&mut self, x: u32) -> &mut Self {
        self.cycle_id_human = x;
        self
    }
    pub fn start_time(&mut self, x: u32) -> &mut Self {
        self.start_time = x;
        self
    }
    pub fn stop_time(&mut self, x: u32) -> &mut Self {
        self.stop_time = Some(x);
        self
    }
    pub fn hostname(&mut self, x: &str) -> &mut Self {
        self.hostname = Some(CString::new(x).unwrap());
        self
    }
    pub fn build(&mut self) -> Object {
        let flags = Flags::new(0);
        let param_length = 0;
        let length: u32 = (self.cycle_id.warts_size()
            + self.list_id.warts_size()
            + self.cycle_id_human.warts_size()
            + self.start_time.warts_size()
            + self.stop_time.warts_size()
            + self.hostname.warts_size()
            + flags.warts_size()
            + param_length) as u32;
        Object::CycleStart {
            length,
            cycle_id: self.cycle_id,
            list_id: self.list_id,
            cycle_id_human: self.cycle_id_human,
            start_time: self.start_time,
            flags,
            param_length: Some(param_length),
            stop_time: self.stop_time,
            hostname: self.hostname.clone(),
        }
    }
}

pub struct CycleStopBuilder {
    cycle_id: u32,
    stop_time: u32,
}

impl CycleStopBuilder {
    pub fn new() -> Self {
        CycleStopBuilder {
            cycle_id: 0,
            stop_time: 0,
        }
    }
    pub fn cycle_id(&mut self, x: u32) -> &mut Self {
        self.cycle_id = x;
        self
    }
    pub fn stop_time(&mut self, x: u32) -> &mut Self {
        self.stop_time = x;
        self
    }
    pub fn build(&mut self) -> Object {
        let length: u32 = (self.cycle_id.warts_size() + self.stop_time.warts_size()) as u32;
        Object::CycleStop {
            length,
            cycle_id: self.cycle_id,
            stop_time: self.stop_time,
            // TODO: Here the flags must be of length zero.
            flags: Flags::new(0),
        }
    }
}

// pub struct TracerouteBuilder {
//     list_id: Option<u32>,
//     cycle_id: Option<u32>,
//     src_addr_id: Option<u32>,
//     dst_addr_id: Option<u32>,
//     start_time: Option<Timeval>,
//     stop_reason: Option<TraceStopReason>,
//     stop_data: Option<u8>,
//     trace_flags: Option<Flags>,
//     attempts: Option<u8>,
//     hop_limit: Option<u8>,
//     trace_type: Option<TraceType>,
//     probe_size: Option<u16>,
//     src_port: Option<u16>,
//     dst_port: Option<u16>,
//     first_ttl: Option<u8>,
//     ip_tos: Option<u8>,
//     timeout_sec: Option<u8>,
//     allowed_loops: Option<u8>,
//     hops_probed: Option<u16>,
//     gap_limit: Option<u8>,
//     gap_limit_action: Option<u8>,
//     loop_action: Option<u8>,
//     probes_sent: Option<u16>,
//     interval_csec: Option<u8>,
//     confidence_level: Option<u8>,
//     src_addr: Option<Address>,
//     dst_addr: Option<Address>,
//     user_id: Option<u32>,
//     ip_offset: Option<u16>,
//     router_addr: Option<Address>,
//     hops: Vec<TraceProbe>,
// }
//
// impl TracerouteBuilder {
//     pub fn new() -> Self {
//         TracerouteBuilder {
//             list_id: None,
//             cycle_id: None,
//             src_addr_id: None,
//             dst_addr_id: None,
//             start_time: None,
//             stop_reason: None,
//             stop_data: None,
//             trace_flags: None,
//             attempts: None,
//             hop_limit: None,
//             trace_type: None,
//             probe_size: None,
//             src_port: None,
//             dst_port: None,
//             first_ttl: None,
//             ip_tos: None,
//             timeout_sec: None,
//             allowed_loops: None,
//             hops_probed: None,
//             gap_limit: None,
//             gap_limit_action: None,
//             loop_action: None,
//             probes_sent: None,
//             interval_csec: None,
//             confidence_level: None,
//             src_addr: None,
//             dst_addr: None,
//             user_id: None,
//             ip_offset: None,
//             router_addr: None,
//             hops: vec![],
//         }
//     }
//     pub fn build(&mut self) -> Object {
//         let flags = Flags::new(0);
//         let param_length = 0;
//         let length = 0;
//         Object::Traceroute {
//             length,
//             flags,
//             param_length: Some(param_length),
//             list_id: self.list_id,
//             cycle_id: self.cycle_id,
//             src_addr_id: self.src_addr_id,
//             dst_addr_id: self.dst_addr_id,
//             start_time: self.start_time,
//             stop_reason: self.stop_reason,
//             stop_data: self.stop_data,
//             trace_flags: self.trace_flags,
//             attempts: self.attempts,
//             hop_limit: self.hop_limit,
//             trace_type: self.trace_type,
//             probe_size: self.probe_size,
//             src_port: self.src_port,
//             dst_port: self.dst_port,
//             first_ttl: self.first_ttl,
//             ip_tos: self.ip_tos,
//             timeout_sec: self.timeout_sec,
//             allowed_loops: self.allowed_loops,
//             hops_probed: self.hops_probed,
//             gap_limit: self.gap_limit,
//             gap_limit_action: self.gap_limit_action,
//             loop_action: self.loop_action,
//             probes_sent: self.probes_sent,
//             interval_csec: self.interval_csec,
//             confidence_level: self.confidence_level,
//             src_addr: self.src_addr,
//             dst_addr: self.dst_addr,
//             user_id: self.user_id,
//             ip_offset: None,
//             router_addr: None,
//             hop_count: 0,
//             hops: vec![],
//             eof: 0,
//         }
//     }
// }
