mod utils;

use std::{env, process};
use utils::print_error;

use crate::utils::stylization::{style, Style};

const VERSION: Option<&str> = option_env!("CARGO_PKG_VERSION");

fn main() {
    let mut arguments = env::args().skip(1);

    let argument = match arguments.next() {
        Some(argument) => argument,
        None => {
            print_error("this program requires at least one argument to run.");
            process::exit(1)
        }
    };

    let command = match argument.as_str() {
        "help" => Command::Help,
        "version" => Command::Version,
        "run" => Command::Run,
        _ => {
            print_error("this command does not exist.");
            process::exit(2)
        }
    };

    match command {
        Command::Help => {
            let commands = vec![
                CommandEntry::new("help", "this :D"),
                CommandEntry::new("version", "print the version of Walnut that's installed"),
                CommandEntry::new("run <FILENAME>", "execute a Walnut source file"),
            ];

            for command in commands {
                println!(
                    "{} ({})",
                    style(
                        &format!("=> `walnut {}`", command.name),
                        None,
                        Some(Style::Bold)
                    ),
                    command.description
                );
            }
        }
        Command::Version => {
            if let Some(version) = VERSION {
                println!("v{}", version);
            } else {
                println!("unknown version")
            }
        }
        Command::Run => {
            todo!()
        }
    }
}

enum Command {
    Help,
    Version,
    Run,
}

struct CommandEntry<'a> {
    name: &'a str,
    description: &'a str,
}

impl<'a> CommandEntry<'a> {
    fn new(name: &'a str, description: &'a str) -> Self {
        Self { name, description }
    }
}
