use clap::Parser;
use log::{error, info};
use simplelog::*;

use vomit_sync;
mod config;

const VERSION: &'static str = env!("CARGO_PKG_VERSION");
const AUTHORS: &'static str = env!("CARGO_PKG_AUTHORS");

/// Sync a local maildir to a remote IMAP account.
#[derive(Parser, Debug)]
#[clap(name = "vsync", version = VERSION, author = AUTHORS)]
struct Opts {
    /// Sets a custom config file location
    #[clap(short, long)]
    config: Option<String>,
    /// The account to operate on (defaults to first account in config file)
    #[clap(short, long)]
    account: Option<String>,
    /// Check if remote server supports required capabilities
    #[clap(long)]
    check: bool,
    /// Increase verbosity (can be used multiple times)
    #[clap(short, long, parse(from_occurrences))]
    verbose: usize,
    /// Number of threads (and IMAP sessions) to use (default: 3)
    #[clap(short, long, default_value_t = 3)]
    threads: u8,
    /// Force state cache reconciliation (can be slow, use only if prompted)
    #[clap(long)]
    force_reconcile: bool,
}

fn main() {
    let opts: Opts = Opts::parse();
    let level = match opts.verbose {
        0 => LevelFilter::Warn,
        1 => LevelFilter::Info,
        2 => LevelFilter::Debug,
        _ => LevelFilter::Trace,
    };

    TermLogger::init(
        level,
        Config::default(),
        TerminalMode::Mixed,
        ColorChoice::Auto,
    )
    .expect("Failed to initialize logger");

    let cfg = match config::load(opts.config) {
        Ok(c) => c,
        Err(e) => {
            error!("Failed to load config file: {}", e);
            std::process::exit(1);
        }
    };
    let acc = match cfg.for_account(None) {
        Some(a) => a,
        None => {
            error!("Failed to load account configuration: account not found");
            std::process::exit(1);
        }
    };

    let sync_opts = vomit_sync::SyncOptions {
        local: acc.maildir(),
        remote: acc.remote().expect("IMAP server missing in config"),
        user: acc.user().expect("IMAP user missing in config"),
        password: acc.password().expect("IMAP password missing in config"),
        reconcile: opts.force_reconcile,
        threads: opts.threads,
    };

    if opts.check {
        info!("Performing capabilities check for account {}", acc.name());
        match vomit_sync::check_server_capabilities(&sync_opts) {
            Ok(_) => {
                info!("Capability check successful");
                std::process::exit(0);
            }
            Err(e) => {
                error!("Capabilitiy check failed: {}", e);
                std::process::exit(1);
            }
        };
    }

    if let Err(e) = vomit_sync::pull(&sync_opts) {
        error!("Error during pull: {}", e);
        std::process::exit(1);
    }
}
