use std::error::Error;
use std::fmt;
use std::fs;
use std::path::PathBuf;
use std::process::Command;

use shellexpand;
use toml;

const DEFAULT_MAILDIR: &str = "~/.maildir";

#[derive(Debug)]
struct ConfigError {
    msg: String,
}

impl fmt::Display for ConfigError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "error parsing config file: {}", self.msg)
    }
}

impl Error for ConfigError {}

pub struct Account<'a> {
    name: String,
    settings: &'a toml::value::Table,
}

pub struct Config {
    cfg: toml::value::Table,
}

impl Config {
    pub fn for_account(&self, name: Option<String>) -> Option<Account> {
        let name = match &name {
            Some(n) => n,
            None => self.cfg.keys().next().expect("no account found"),
        };

        if let Some(a) = self.cfg.get(name) {
            if let Some(t) = a.as_table() {
                return Some(Account {
                    name: String::from(name),
                    settings: t,
                });
            }
        }
        None
    }
}

impl Account<'_> {
    pub fn maildir(&self) -> String {
        let mut path = DEFAULT_MAILDIR;
        if let Some(l) = self.settings.get("local") {
            if let Some(p) = l.as_str() {
                path = p
            }
        };
        shellexpand::tilde(path).into_owned()
    }

    pub fn name(&self) -> &String {
        &self.name
    }

    pub fn remote(&self) -> Option<String> {
        if let Some(r) = self.settings.get("remote") {
            if let Some(remote) = r.as_str() {
                return Some(String::from(remote));
            }
        }
        return None;
    }

    pub fn user(&self) -> Option<String> {
        if let Some(u) = self.settings.get("user") {
            if let Some(user) = u.as_str() {
                return Some(String::from(user));
            }
        }
        return None;
    }

    pub fn password(&self) -> Option<String> {
        if let Some(p) = self.settings.get("password") {
            if let Some(pw) = p.as_str() {
                return Some(String::from(pw));
            }
        } else if let Some(p) = self.settings.get("pass-cmd") {
            if let Some(cmd) = p.as_str() {
                let out = Command::new("sh").arg("-c").arg(cmd).output();
                match out {
                    Ok(mut output) => {
                        let newline: u8 = 10;
                        if Some(&newline) == output.stdout.last() {
                            _ = output.stdout.pop(); // remove trailing newline
                        }
                        return String::from_utf8(output.stdout).ok();
                    }
                    Err(e) => {
                        eprintln!("Failed to execute pass-cmd: {:?}", e);
                    }
                }
            }
        }
        return None;
    }
}

fn default_cfg() -> String {
    if let Some(path) = dirs::config_dir() {
        let mut cfg = PathBuf::from(path);
        cfg.push("vsync");
        cfg.push("config.toml");
        let oss = cfg.into_os_string();
        if let Some(s) = (&oss).to_str() {
            return String::from(s);
        }
    }
    shellexpand::tilde("~/.vsyncrc").into_owned()
}

pub fn load(path: Option<String>) -> Result<Config, Box<dyn Error>> {
    let filename = match path {
        Some(p) => p,
        None => default_cfg(),
    };

    let contents = fs::read_to_string(filename)?;
    let value = contents.parse::<toml::Value>()?;
    let table: toml::value::Table = value.try_into()?;

    if table.len() < 1 {
        let e = ConfigError {
            msg: String::from("no accounts found"),
        };
        Err(Box::new(e))
    } else {
        Ok(Config { cfg: table })
    }
}
