use std::io::Cursor;

use prost::Message as ProstMesssage;

use crate::utils::{ndarray::Ndarray, vec::Vec3};

/// Protocol buffers generated by `prost.rs`.
pub mod messages {
    include!(concat!(env!("OUT_DIR"), "/protocol.rs"));
}

pub use messages::Message;

pub type MessageType = messages::message::Type;

impl Message {
    /// Create a new protobuf message with the idiomatic Builder pattern.
    pub fn new(r#type: &MessageType) -> MessageBuilder {
        MessageBuilder {
            r#type: r#type.to_owned(),
            ..Default::default()
        }
    }
}

/// Encode message into protocol buffers.
pub fn encode_message(message: &Message) -> Vec<u8> {
    let mut buf = Vec::new();
    buf.reserve(message.encoded_len());
    message.encode(&mut buf).unwrap();
    buf
}

/// Decode protocol buffers into a message struct.
pub fn decode_message(buf: &[u8]) -> Result<Message, prost::DecodeError> {
    Message::decode(&mut Cursor::new(buf))
}

/// Protocol buffer compatible geometry data structure.
#[derive(Debug, Clone)]
pub struct Geometry {
    pub positions: Vec<f32>,
    pub indices: Vec<i32>,
    pub uvs: Vec<f32>,
    pub aos: Vec<i32>,
    pub lights: Vec<i32>,
}

/// Protocol buffer compatible mesh data structure.
#[derive(Debug, Clone)]
pub struct Mesh {
    pub opaque: Option<Geometry>,
    pub transparent: Option<Geometry>,
}

/// Protocol buffer compatible chunk data structure.
#[derive(Debug, Clone, Default)]
pub struct Chunk {
    pub x: i32,
    pub z: i32,
    pub id: String,
    pub mesh: Option<Mesh>,
    pub voxels: Option<Ndarray<u32>>,
    pub lights: Option<Ndarray<u32>>,
    pub height_map: Option<Ndarray<u32>>,
}

/// Protocol buffer compatible peer data structure.
#[derive(Debug, Clone)]
pub struct Peer {
    pub id: String,
    pub name: String,
    pub position: Option<Vec3<f32>>,
    pub direction: Option<Vec3<f32>>,
}

/// Protocol buffer compatible entity data structure.
#[derive(Debug, Clone)]
pub struct Entity {
    pub id: String,
    pub r#type: String,
    pub metadata: Option<String>,
}

/// Builder for a protocol buffer message.
#[derive(Default)]
pub struct MessageBuilder {
    r#type: MessageType,

    json: Option<String>,
    text: Option<String>,
    peer: Option<Peer>,

    peers: Option<Vec<String>>,
    entities: Option<Vec<Entity>>,
    chunks: Option<Vec<Chunk>>,
}

/// Convert a `Vec3` to protocol buffer `Vector3`.
fn vec3_to_vector3(vec3: &Option<Vec3<f32>>) -> Option<messages::Vector3> {
    if let Some(vec3) = vec3 {
        Some(messages::Vector3 {
            x: vec3.0,
            y: vec3.1,
            z: vec3.2,
        })
    } else {
        None
    }
}

impl MessageBuilder {
    /// Configure the json data of the protocol.
    pub fn json(mut self, json: &str) -> Self {
        self.json = Some(json.to_owned());
        self
    }

    /// Configure the text data of the protocol.
    pub fn text(mut self, text: &str) -> Self {
        self.text = Some(text.to_owned());
        self
    }

    /// Configure the peer data of the protocol.
    pub fn peer(mut self, peer: Peer) -> Self {
        self.peer = Some(peer);
        self
    }

    /// Configure the peers data of the protocol.
    pub fn peers(mut self, peers: &[String]) -> Self {
        self.peers = Some(peers.to_vec());
        self
    }

    /// Configure the entities data of the protocol.
    pub fn entities(mut self, entities: &[Entity]) -> Self {
        self.entities = Some(entities.to_vec());
        self
    }

    /// Configure the chunks data of the protocol.
    pub fn chunks(mut self, chunks: &[Chunk]) -> Self {
        self.chunks = Some(chunks.to_vec());
        self
    }

    /// Create a protocol buffer message.
    pub fn build(self) -> Message {
        let mut message = messages::Message {
            r#type: self.r#type as i32,
            ..Default::default()
        };

        message.json = self.json.unwrap_or_default();
        message.text = self.text.unwrap_or_default();
        message.peers = self.peers.unwrap_or_default();

        if let Some(peer) = self.peer {
            let Peer {
                id,
                name,
                direction,
                position,
            } = peer;

            message.peer = Some(messages::Peer {
                id,
                name,
                direction: vec3_to_vector3(&direction),
                position: vec3_to_vector3(&position),
            });
        }

        if let Some(entities) = self.entities {
            message.entities = entities
                .into_iter()
                .map(|entity| messages::Entity {
                    id: entity.id,
                    r#type: entity.r#type,
                    metadata: entity.metadata.unwrap_or_default(),
                })
                .collect();
        }

        if let Some(chunks) = self.chunks {
            message.chunks = chunks
                .into_iter()
                .map(|chunk| messages::Chunk {
                    id: chunk.id,
                    mesh: if let Some(mesh) = chunk.mesh {
                        let opaque = mesh.opaque.as_ref();
                        let transparent = mesh.transparent.as_ref();

                        Some(messages::Mesh {
                            opaque: opaque.map(|opaque| messages::Geometry {
                                aos: opaque.aos.to_owned(),
                                indices: opaque.indices.to_owned(),
                                positions: opaque.positions.to_owned(),
                                lights: opaque.lights.to_owned(),
                                uvs: opaque.uvs.to_owned(),
                            }),
                            transparent: transparent.map(|transparent| messages::Geometry {
                                aos: transparent.aos.to_owned(),
                                indices: transparent.indices.to_owned(),
                                positions: transparent.positions.to_owned(),
                                lights: transparent.lights.to_owned(),
                                uvs: transparent.uvs.to_owned(),
                            }),
                        })
                    } else {
                        None
                    },
                    lights: chunk.lights.unwrap_or_default().data,
                    voxels: chunk.voxels.unwrap_or_default().data,
                    height_map: chunk.height_map.unwrap_or_default().data,
                    x: chunk.x,
                    z: chunk.z,
                })
                .collect();
        }

        message
    }
}
