/*
 * // Copyright (c) 2021 Feng Yang
 * //
 * // I am making my contributions/submissions to this project solely in my
 * // personal capacity and am not conveying any rights to any intellectual
 * // property of any third parties.
 */

use std::ops::{Index, IndexMut, AddAssign, SubAssign, MulAssign, DivAssign, Add, Sub, Mul, Div};
use std::fmt::{Debug, Formatter, Result};
use crate::usize2::USize2;

///
/// # 3-D size class.
///
/// This class defines simple 3-D size data.
///
/// - tparam T - Type of the element
///
pub struct USize3 {
    /// X (or the first) component of the size.
    pub x: usize,

    /// Y (or the second) component of the size.
    pub y: usize,

    /// Z (or the third) component of the size.
    pub z: usize,
}

/// # Constructors
impl USize3 {
    /// Constructs default size (0, 0, 0).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec = USize3::new_default();
    /// assert_eq!(0, vec.x);
    /// assert_eq!(0, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn new_default() -> USize3 {
        return USize3 {
            x: 0,
            y: 0,
            z: 0,
        };
    }

    /// Constructs size with given parameters **x_**, **y_**, and **z_**.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec2 = USize3::new(5, 3, 8);
    /// assert_eq!(5, vec2.x);
    /// assert_eq!(3, vec2.y);
    /// assert_eq!(8, vec2.z);
    /// ```
    pub fn new(x_: usize, y_: usize, z_: usize) -> USize3 {
        return USize3 {
            x: x_,
            y: y_,
            z: z_,
        };
    }

    /// Constructs size with a 2-D size and a scalar.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec3 = USize2::new(4, 7);
    /// let vec4=  USize3::new_vec(vec3, 9);
    /// assert_eq!(4, vec4.x);
    /// assert_eq!(7, vec4.y);
    /// assert_eq!(9, vec4.z);
    /// ```
    pub fn new_vec(v: USize2, z_: usize) -> USize3 {
        return USize3 {
            x: v.x,
            y: v.y,
            z: z_,
        };
    }

    /// Constructs size with initializer list.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec5 = USize3::new_lst([7, 6, 1]);
    /// assert_eq!(7, vec5.x);
    /// assert_eq!(6, vec5.y);
    /// assert_eq!(1, vec5.z);
    /// ```
    pub fn new_lst(lst: [usize; 3]) -> USize3 {
        return USize3 {
            x: lst[0],
            y: lst[1],
            z: lst[2],
        };
    }
}

/// # Basic setters
impl USize3 {
    /// Set all x, y, and z components to **s**.
    pub fn set_scalar(&mut self, s: usize) {
        self.x = s;
        self.y = s;
        self.z = s;
    }

    /// Set x, y, and z components with given parameters.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new_default();
    /// vec.set_scalar3(4, 2, 8);
    /// assert_eq!(4, vec.x);
    /// assert_eq!(2, vec.y);
    /// assert_eq!(8, vec.z);
    /// ```
    pub fn set_scalar3(&mut self, x: usize, y: usize, z: usize) {
        self.x = x;
        self.y = y;
        self.z = z;
    }

    /// Set x, y, and z components with given **pt.x**, **pt.y**, and **z**.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize3::new_default();
    /// vec.set_vec(USize2::new(1, 3), 10);
    /// assert_eq!(1, vec.x);
    /// assert_eq!(3, vec.y);
    /// assert_eq!(10, vec.z);
    /// ```
    pub fn set_vec(&mut self, pt: USize2, z: usize) {
        self.x = pt.x;
        self.y = pt.y;
        self.z = z;
    }

    /// Set x, y, and z components with given initializer list.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new_default();
    /// let lst = [0, 5, 6];
    /// vec.set_lst(lst);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(5, vec.y);
    /// assert_eq!(6, vec.z);
    /// ```
    pub fn set_lst(&mut self, lst: [usize; 3]) {
        self.x = lst[0];
        self.y = lst[1];
        self.z = lst[2];
    }

    /// Set x, y, and z with other size **v**.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new_default();
    /// vec.set_self(USize3::new(9, 8, 2));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(8, vec.y);
    /// assert_eq!(2, vec.z);
    /// ```
    pub fn set_self(&mut self, v: USize3) {
        self.x = v.x;
        self.y = v.y;
        self.z = v.z;
    }

    /// Set all x, y, and z to zero.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.set_zero();
    /// assert_eq!(0, vec.x);
    /// assert_eq!(0, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn set_zero(&mut self) {
        self.x = 0;
        self.y = 0;
        self.z = 0;
    }
}

/// # Binary operations: new instance = self (+) v
impl USize3 {
    /// Computes self + (v, v, v).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec = vec.add_scalar(4);
    /// assert_eq!(7, vec.x);
    /// assert_eq!(13, vec.y);
    /// assert_eq!(8, vec.z);
    /// ```
    pub fn add_scalar(&self, v: usize) -> USize3 {
        return USize3::new(self.x + v, self.y + v, self.z + v);
    }

    /// Computes self + (v.x, v.y, v.z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize3::new(2, 1, 5));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(14, vec.y);
    /// assert_eq!(13, vec.z);
    /// ```
    pub fn add_vec(&self, v: USize3) -> USize3 {
        return USize3::new(self.x + v.x, self.y + v.y, self.z + v.z);
    }

    /// Computes self - (v, v, v).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize3::new(2, 1, 5));
    /// vec = vec.sub_scalar(8);
    /// assert_eq!(1, vec.x);
    /// assert_eq!(6, vec.y);
    /// assert_eq!(5, vec.z);
    /// ```
    pub fn sub_scalar(&self, v: usize) -> USize3 {
        return USize3::new(self.x - v, self.y - v, self.z - v);
    }

    /// Computes self - (v.x, v.y, v.z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize3::new(2, 1, 5));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize3::new(1, 3, 5));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn sub_vec(&self, v: USize3) -> USize3 {
        return USize3::new(self.x - v.x, self.y - v.y, self.z - v.z);
    }

    /// Computes self * (v, v, v).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize3::new(2, 1, 5));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize3::new(1, 3, 5));
    /// vec = vec.mul_scalar(2);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(6, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn mul_scalar(&self, v: usize) -> USize3 {
        return USize3::new(self.x * v, self.y * v, self.z * v);
    }

    /// Computes self * (v.x, v.y, v.z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize3::new(2, 1, 5));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize3::new(1, 3, 5));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(USize3::new(3, 2, 0));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(12, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn mul_vec(&self, v: USize3) -> USize3 {
        return USize3::new(self.x * v.x, self.y * v.y, self.z * v.z);
    }

    /// Computes self / (v, v, v).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize3::new(2, 1, 5));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize3::new(1, 3, 5));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(USize3::new(3, 2, 0));
    /// vec = vec.div_scalar(4);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn div_scalar(&self, v: usize) -> USize3 {
        return USize3::new(self.x / v, self.y / v, self.z / v);
    }

    /// Computes self / (v.x, v.y, v.z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize3::new(2, 1, 5));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize3::new(1, 3, 5));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(USize3::new(3, 2, 0));
    /// vec = vec.div_scalar(4);
    /// vec = vec.div_vec(USize3::new(1, 1, 1));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn div_vec(&self, v: USize3) -> USize3 {
        return USize3::new(self.x / v.x, self.y / v.y, self.z / v.z);
    }
}

/// # Binary operations: new instance = v (+) self
impl USize3 {
    /// Computes (v, v, v) - self.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(5, 14, 13);
    /// vec = vec.rsub_scalar(14);
    /// assert_eq!(9, vec.x);
    /// assert_eq!(0, vec.y);
    /// assert_eq!(1, vec.z);
    /// ```
    pub fn rsub_scalar(&self, v: usize) -> USize3 {
        return USize3::new(v - self.x, v - self.y, v - self.z);
    }

    /// Computes (v.x, v.y, v.z) - self.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(5, 14, 13);
    /// vec = vec.rsub_scalar(14);
    /// vec = vec.rsub_vec(USize3::new(9, 0, 1));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(0, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn rsub_vec(&self, v: USize3) -> USize3 {
        return USize3::new(v.x - self.x, v.y - self.y, v.z - self.z);
    }

    /// Computes (v, v, v) / self.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(12, 9, 8);
    /// vec = vec.rdiv_scalar(36);
    /// assert_eq!(3, vec.x);
    /// assert_eq!(4, vec.y);
    /// assert_eq!(4, vec.z);
    /// ```
    pub fn rdiv_scalar(&self, v: usize) -> USize3 {
        return USize3::new(v / self.x, v / self.y, v / self.z);
    }

    /// Computes (v.x, v.y, v.z) / self.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(12, 9, 8);
    /// vec = vec.rdiv_scalar(36);
    /// vec = vec.rdiv_vec(USize3::new(3, 16, 18));
    /// assert_eq!(1, vec.x);
    /// assert_eq!(4, vec.y);
    /// assert_eq!(4, vec.z);
    /// ```
    pub fn rdiv_vec(&self, v: USize3) -> USize3 {
        return USize3::new(v.x / self.x, v.y / self.y, v.z / self.z);
    }
}

/// # Augmented operators: self (+)= v
impl USize3 {
    /// Computes self += (v, v, v).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.iadd_scalar(4);
    /// assert_eq!(7, vec.x);
    /// assert_eq!(13, vec.y);
    /// assert_eq!(8, vec.z);
    /// ```
    pub fn iadd_scalar(&mut self, v: usize) {
        self.x = self.x + v;
        self.y = self.y + v;
        self.z = self.z + v;
    }

    /// Computes self += (v.x, v.y, v.z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize3::new(2, 1, 5));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(14, vec.y);
    /// assert_eq!(13, vec.z);
    /// ```
    pub fn iadd_vec(&mut self, v: USize3) {
        self.x = self.x + v.x;
        self.y = self.y + v.y;
        self.z = self.z + v.z;
    }

    /// Computes self -= (v, v, v).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize3::new(2, 1, 5));
    /// vec.isub_scalar(8);
    /// assert_eq!(1, vec.x);
    /// assert_eq!(6, vec.y);
    /// assert_eq!(5, vec.z);
    /// ```
    pub fn isub_scalar(&mut self, v: usize) {
        self.x = self.x - v;
        self.y = self.y - v;
        self.z = self.z - v;
    }

    /// Computes self -= (v.x, v.y, v.z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize3::new(2, 1, 5));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize3::new(1, 3, 5));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn isub_vec(&mut self, v: USize3) {
        self.x = self.x - v.x;
        self.y = self.y - v.y;
        self.z = self.z - v.z;
    }

    /// Computes self *= (v, v, v).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize3::new(2, 1, 5));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize3::new(1, 3, 5));
    /// vec.imul_scalar(2);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(6, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn imul_scalar(&mut self, v: usize) {
        self.x = self.x * v;
        self.y = self.y * v;
        self.z = self.z * v;
    }

    /// Computes self *= (v.x, v.y, v.z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize3::new(2, 1, 5));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize3::new(1, 3, 5));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(USize3::new(3, 2, 0));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(12, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn imul_vec(&mut self, v: USize3) {
        self.x = self.x * v.x;
        self.y = self.y * v.y;
        self.z = self.z * v.z;
    }

    /// Computes self /= (v, v, v).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize3::new(2, 1, 5));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize3::new(1, 3, 5));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(USize3::new(3, 2, 0));
    /// vec.idiv_scalar(4);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn idiv_scalar(&mut self, v: usize) {
        self.x = self.x / v;
        self.y = self.y / v;
        self.z = self.z / v;
    }

    /// Computes self /= (v.x, v.y, v.z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(3, 9, 4);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize3::new(2, 1, 5));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize3::new(1, 3, 5));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(USize3::new(3, 2, 0));
    /// vec.idiv_scalar(4);
    /// vec.idiv_vec(USize3::new(3, 1, 1));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// assert_eq!(0, vec.z);
    /// ```
    pub fn idiv_vec(&mut self, v: USize3) {
        self.x = self.x / v.x;
        self.y = self.y / v.y;
        self.z = self.z / v.z;
    }
}

/// # Basic getters
impl USize3 {
    /// Returns const reference to the **i** -th element of the size.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec = USize3::new(8, 9, 1);
    /// assert_eq!(8, *vec.at(0));
    /// assert_eq!(9, *vec.at(1));
    /// assert_eq!(1, *vec.at(2));
    /// ```
    pub fn at(&self, i: usize) -> &usize {
        match i {
            0 => return &self.x,
            1 => return &self.y,
            2 => return &self.z,
            _ => { panic!() }
        }
    }

    /// Returns reference to the **i** -th element of the size.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let mut vec = USize3::new(8, 9, 1);
    /// *vec.at_mut(0) = 7;
    /// *vec.at_mut(1) = 6;
    /// *vec.at_mut(2) = 5;
    /// assert_eq!(7, vec.x);
    /// assert_eq!(6, vec.y);
    /// assert_eq!(5, vec.z);
    /// ```
    pub fn at_mut(&mut self, i: usize) -> &mut usize {
        match i {
            0 => return &mut self.x,
            1 => return &mut self.y,
            2 => return &mut self.z,
            _ => { panic!() }
        }
    }

    /// Returns the sum of all the components (i.e. x + y + z).
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec = USize3::new(3, 7, 1);
    /// let sum = vec.sum();
    /// assert_eq!(11, sum);
    /// ```
    pub fn sum(&self) -> usize {
        return self.x + self.y + self.z;
    }

    /// Returns the minimum value among x, y, and z.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec = USize3::new(3, 7, 1);
    /// let min = vec.min();
    /// assert_eq!(1, min);
    /// ```
    pub fn min(&self) -> usize {
        return usize::min(usize::min(self.x, self.y), self.z);
    }

    /// Returns the maximum value among x, y, and z.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec = USize3::new(3, 7, 1);
    /// let max = vec.max();
    /// assert_eq!(7, max);
    /// ```
    pub fn max(&self) -> usize {
        return usize::max(usize::max(self.x, self.y), self.z);
    }

    /// Returns the index of the dominant axis.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec = USize3::new(3, 7, 1);
    /// let dominant_axis = vec.dominant_axis();
    /// assert_eq!(1, dominant_axis);
    /// ```
    pub fn dominant_axis(&self) -> usize {
        return match self.x > self.y {
            true => match self.x > self.z {
                true => 0,
                false => 2
            }
            false => match self.y > self.z {
                true => 1,
                false => 2
            }
        };
    }

    /// Returns the index of the subminant axis.
    /// ```
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec = USize3::new(3, 7, 1);
    /// let subminant_axis = vec.subminant_axis();
    /// assert_eq!(2, subminant_axis);
    /// ```
    pub fn subminant_axis(&self) -> usize {
        return match self.x < self.y {
            true => match self.x < self.z {
                true => 0,
                false => 2
            }
            false => match self.y < self.z {
                true => 1,
                false => 2
            }
        };
    }

    /// Returns true if **other** is the same as self size.
    pub fn is_equal(&self, other: &USize3) -> bool {
        return self.x == other.x && self.y == other.y && self.z == other.z;
    }
}

/// Copy constructor.
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let vec5 = USize3::new_lst([7, 6, 1]);
/// let vec6 = vec5.clone();
/// assert_eq!(7, vec6.x);
/// assert_eq!(6, vec6.y);
/// (1, vec6.z);
/// ```
impl Clone for USize3 {
    fn clone(&self) -> Self {
        return USize3 {
            x: self.x,
            y: self.y,
            z: self.z,
        };
    }
}

/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let vec = USize3::new(5, 1, 0);
/// let mut vec2 = USize3::new(3, 3, 3);
/// vec2 = vec;
/// assert_eq!(5, vec2.x);
/// assert_eq!(1, vec2.y);
/// assert_eq!(0, vec2.z);
/// ```
impl Copy for USize3 {}

/// # Operators
/// Returns const reference to the **i** -th element of the size.
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let vec = USize3::new(8, 9, 1);
/// assert_eq!(8, vec[0]);
/// assert_eq!(9, vec[1]);
/// assert_eq!(1, vec[2]);
/// ```
impl Index<usize> for USize3 {
    type Output = usize;
    fn index(&self, index: usize) -> &Self::Output {
        return self.at(index);
    }
}

/// Returns reference to the **i** -th element of the size.
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(8, 9, 1);
/// vec[0] = 7;
/// vec[1] = 6;
/// vec[2] = 5;
/// assert_eq!(7, vec.x);
/// assert_eq!(6, vec.y);
/// assert_eq!(5, vec.z);
/// ```
impl IndexMut<usize> for USize3 {
    fn index_mut(&mut self, index: usize) -> &mut Self::Output {
        return self.at_mut(index);
    }
}

/// Computes self += (v, v, v)
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 2);
/// vec += 4;
/// assert_eq!(7, vec.x);
/// assert_eq!(13, vec.y);
/// assert_eq!(6, vec.z);
/// ```
impl AddAssign<usize> for USize3 {
    fn add_assign(&mut self, rhs: usize) {
        self.iadd_scalar(rhs);
    }
}

/// Computes self += (v.x, v.y, v.z)
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 2);
/// vec += 4;
/// vec += USize3::new(2, 1, 5);
/// assert_eq!(9, vec.x);
/// assert_eq!(14, vec.y);
/// assert_eq!(11, vec.z);
/// ```
impl AddAssign for USize3 {
    fn add_assign(&mut self, rhs: Self) {
        self.iadd_vec(rhs);
    }
}

/// Computes self -= (v, v, v)
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 2);
/// vec += 4;
/// vec += USize3::new(2, 1, 5);
/// vec -= 8;
/// assert_eq!(1, vec.x);
/// assert_eq!(6, vec.y);
/// assert_eq!(3, vec.z);
/// ```
impl SubAssign<usize> for USize3 {
    fn sub_assign(&mut self, rhs: usize) {
        self.isub_scalar(rhs);
    }
}

/// Computes self -= (v.x, v.y, v.z)
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 2);
/// vec += 4;
/// vec += USize3::new(2, 1, 5);
/// vec -= 8;
/// vec -= USize3::new(1, 3, 3);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl SubAssign for USize3 {
    fn sub_assign(&mut self, rhs: Self) {
        self.isub_vec(rhs);
    }
}

/// Computes self *= (v, v, v)
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 2);
/// vec += 4;
/// vec += USize3::new(2, 1, 5);
/// vec -= 8;
/// vec -= USize3::new(1, 3, 3);
/// vec *= 2;
/// assert_eq!(0, vec.x);
/// assert_eq!(6, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl MulAssign<usize> for USize3 {
    fn mul_assign(&mut self, rhs: usize) {
        self.imul_scalar(rhs);
    }
}

/// Computes self *= (v.x, v.y, v.z)
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 2);
/// vec += 4;
/// vec += USize3::new(2, 1, 5);
/// vec -= 8;
/// vec -= USize3::new(1, 3, 3);
/// vec *= 2;
/// vec *= USize3::new(3, 2, 0);
/// assert_eq!(0, vec.x);
/// assert_eq!(12, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl MulAssign for USize3 {
    fn mul_assign(&mut self, rhs: Self) {
        self.imul_vec(rhs);
    }
}

/// Computes self /= (v, v, v)
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 2);
/// vec += 4;
/// vec += USize3::new(2, 1, 5);
/// vec -= 8;
/// vec -= USize3::new(1, 3, 3);
/// vec *= 2;
/// vec *= USize3::new(3, 2, 0);
/// vec /= 4;
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl DivAssign<usize> for USize3 {
    fn div_assign(&mut self, rhs: usize) {
        self.idiv_scalar(rhs);
    }
}

/// Computes self /= (v.x, v.y, v.z)
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 2);
/// vec += 4;
/// vec += USize3::new(2, 1, 5);
/// vec -= 8;
/// vec -= USize3::new(1, 3, 3);
/// vec *= 2;
/// vec *= USize3::new(3, 2, 0);
/// vec /= 4;
/// vec /= USize3::new(3, 1, 2);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl DivAssign for USize3 {
    fn div_assign(&mut self, rhs: Self) {
        self.idiv_vec(rhs);
    }
}

/// Returns true if **other** is the same as self size.
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new_default();
/// let vec2 = USize3::new(3, 7, 4);
/// let vec3 = USize3::new(3, 5, 4);
/// let vec4 = USize3::new(5, 1, 2);
/// vec = vec2;
/// assert_eq!(vec == vec2, true);
/// assert_eq!(vec == vec3, false);
/// assert_eq!(vec != vec2, false);
/// assert_eq!(vec != vec3, true);
/// assert_eq!(vec != vec4, true);
/// ```
impl PartialEq for USize3 {
    fn eq(&self, other: &Self) -> bool {
        return self.is_equal(other);
    }
}

impl Eq for USize3 {}

/// Computes (a, a, a) + (b.x, b.y, b.z).
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 4);
/// vec = vec + 4;
/// assert_eq!(7, vec.x);
/// assert_eq!(13, vec.y);
/// assert_eq!(8, vec.z);
/// ```
impl Add<usize> for USize3 {
    type Output = USize3;
    fn add(self, rhs: usize) -> Self::Output {
        return self.add_scalar(rhs);
    }
}

/// Computes (a.x, a.y, a.z) + (b.x, b.y, b.z).
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 4);
/// vec = vec + 4;
/// vec = vec + USize3::new(2, 1, 5);
/// assert_eq!(9, vec.x);
/// assert_eq!(14, vec.y);
/// assert_eq!(13, vec.z);
/// ```
impl Add for USize3 {
    type Output = USize3;
    fn add(self, rhs: Self) -> Self::Output {
        return self.add_vec(rhs);
    }
}

/// Computes (a.x, a.y, a.z) - (b, b, b).
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 4);
/// vec = vec + 4;
/// vec = vec + USize3::new(2, 1, 5);
/// vec = vec - 8;
/// assert_eq!(1, vec.x);
/// assert_eq!(6, vec.y);
/// assert_eq!(5, vec.z);
/// ```
impl Sub<usize> for USize3 {
    type Output = USize3;
    fn sub(self, rhs: usize) -> Self::Output {
        return self.sub_scalar(rhs);
    }
}

/// Computes (a.x, a.y, a.z) - (b.x, b.y, b.z).
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 4);
/// vec = vec + 4;
/// vec = vec + USize3::new(2, 1, 5);
/// vec = vec - 8;
/// vec = vec - USize3::new(1, 3, 5);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl Sub for USize3 {
    type Output = USize3;
    fn sub(self, rhs: Self) -> Self::Output {
        return self.sub_vec(rhs);
    }
}

/// Computes (a.x, a.y, a.z) * (b, b, b).
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 4);
/// vec = vec + 4;
/// vec = vec + USize3::new(2, 1, 5);
/// vec = vec - 8;
/// vec = vec - USize3::new(1, 3, 5);
/// vec = vec * 2;
/// assert_eq!(0, vec.x);
/// assert_eq!(6, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl Mul<usize> for USize3 {
    type Output = USize3;
    fn mul(self, rhs: usize) -> Self::Output {
        return self.mul_scalar(rhs);
    }
}

/// Computes (a.x, a.y, a.z) * (b.x, b.y, b.z).
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 4);
/// vec = vec + 4;
/// vec = vec + USize3::new(2, 1, 5);
/// vec = vec - 8;
/// vec = vec - USize3::new(1, 3, 5);
/// vec = vec * 2;
/// vec = vec * USize3::new(3, 2, 0);
/// assert_eq!(0, vec.x);
/// assert_eq!(12, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl Mul for USize3 {
    type Output = USize3;
    fn mul(self, rhs: Self) -> Self::Output {
        return self.mul_vec(rhs);
    }
}

/// Computes (a.x, a.y, a.z) / (b, b, b).
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 4);
/// vec = vec + 4;
/// vec = vec + USize3::new(2, 1, 5);
/// vec = vec - 8;
/// vec = vec - USize3::new(1, 3, 5);
/// vec = vec * 2;
/// vec = vec * USize3::new(3, 2, 0);
/// vec = vec / 4;
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl Div<usize> for USize3 {
    type Output = USize3;
    fn div(self, rhs: usize) -> Self::Output {
        return self.div_scalar(rhs);
    }
}

/// Computes (a.x, a.y, a.z) / (b.x, b.y, b.z).
/// ```
/// use vox_geometry_rust::usize3::USize3;
/// let mut vec = USize3::new(3, 9, 4);
/// vec = vec + 4;
/// vec = vec + USize3::new(2, 1, 5);
/// vec = vec - 8;
/// vec = vec - USize3::new(1, 3, 5);
/// vec = vec * 2;
/// vec = vec * USize3::new(3, 2, 0);
/// vec = vec / 4;
/// vec = vec / USize3::new(3, 1, 1);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// assert_eq!(0, vec.z);
/// ```
impl Div for USize3 {
    type Output = USize3;
    fn div(self, rhs: Self) -> Self::Output {
        return self.div_vec(rhs);
    }
}

impl Debug for USize3 {
    /// # Example
    /// ```
    ///
    /// use vox_geometry_rust::usize3::USize3;
    /// let vec = USize3::new(10, 20, 30);
    /// assert_eq!(format!("{:?}", vec), "(10, 20, 30)");
    ///
    /// assert_eq!(format!("{:#?}", vec), "(
    ///     10,
    ///     20,
    ///     30,
    /// )");
    /// ```
    fn fmt(&self, f: &mut Formatter<'_>) -> Result {
        f.debug_tuple("")
            .field(&self.x)
            .field(&self.y)
            .field(&self.z)
            .finish()
    }
}

/// Returns element-wise min size.
/// ```
/// use vox_geometry_rust::usize3::{USize3, min};
/// let vec = USize3::new(5, 1, 0);
/// let vec2 = USize3::new(3, 3, 3);
/// let min_size = min(&vec, &vec2);
/// assert_eq!(min_size == USize3::new(3, 1, 0), true);
/// ```
pub fn min(a: &USize3, b: &USize3) -> USize3 {
    return USize3::new(usize::min(a.x, b.x),
                       usize::min(a.y, b.y),
                       usize::min(a.z, b.z));
}

/// Returns element-wise max size.
/// ```
/// use vox_geometry_rust::usize3::{USize3, max};
/// let vec = USize3::new(5, 1, 0);
/// let vec2 = USize3::new(3, 3, 3);
/// let max_size = max(&vec, &vec2);
/// assert_eq!(max_size == USize3::new(5, 3, 3), true);
/// ```
pub fn max(a: &USize3, b: &USize3) -> USize3 {
    return USize3::new(usize::max(a.x, b.x),
                       usize::max(a.y, b.y),
                       usize::max(a.z, b.z));
}

/// Returns element-wise clamped size.
/// ```
/// use vox_geometry_rust::usize3::{USize3, clamp};
/// let vec = USize3::new(2, 4, 1);
/// let low = USize3::new(3, 1, 0);
/// let high = USize3::new(5, 2, 3);
/// let clamped_vec = clamp(&vec, &low, &high);
/// assert_eq!(clamped_vec == USize3::new(3, 2, 1), true);
/// ```
pub fn clamp(v: &USize3, low: &USize3, high: &USize3) -> USize3 {
    return USize3::new(usize::clamp(v.x, low.x, high.x),
                       usize::clamp(v.y, low.y, high.y),
                       usize::clamp(v.z, low.z, high.z));
}