/*
 * // Copyright (c) 2021 Feng Yang
 * //
 * // I am making my contributions/submissions to this project solely in my
 * // personal capacity and am not conveying any rights to any intellectual
 * // property of any third parties.
 */

use std::ops::{Index, IndexMut, AddAssign, SubAssign, MulAssign, DivAssign, Add, Sub, Mul, Div};
use std::fmt::{Debug, Formatter, Result};

///
/// # 2-D size class.
///
/// This class defines simple 2-D size data.
///
/// - tparam T - Type of the element
///
pub struct USize2 {
    /// X (or the first) component of the size.
    pub x: usize,

    /// Y (or the second) component of the size.
    pub y: usize,
}

/// # Constructors
impl USize2 {
    /// Constructs default size (0, 0).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec = USize2::new_default();
    /// assert_eq!(0, vec.x);
    /// assert_eq!(0, vec.y);
    /// ```
    pub fn new_default() -> USize2 {
        return USize2 {
            x: 0,
            y: 0,
        };
    }

    /// Constructs size with given parameters \p x_ and \p y_.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec2 = USize2::new(5, 3);
    /// assert_eq!(5, vec2.x);
    /// assert_eq!(3, vec2.y);
    /// ```
    pub fn new(x_: usize, y_: usize) -> USize2 {
        return USize2 {
            x: x_,
            y: y_,
        };
    }

    /// Constructs size with initializer list.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec5 = USize2::new_lst([7, 6]);
    /// assert_eq!(7, vec5.x);
    /// assert_eq!(6, vec5.y);
    /// ```
    pub fn new_lst(lst: [usize; 2]) -> USize2 {
        return USize2 {
            x: lst[0],
            y: lst[1],
        };
    }
}

/// # Basic setters
impl USize2 {
    /// Set both x and y components to **s**.
    pub fn set_scalar(&mut self, s: usize) {
        self.x = s;
        self.y = s;
    }

    /// Set x and y components with given parameters.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new_default();
    /// vec.set_scalar2(4, 2);
    /// assert_eq!(4, vec.x);
    /// assert_eq!(2, vec.y);
    /// ```
    pub fn set_scalar2(&mut self, x: usize, y: usize) {
        self.x = x;
        self.y = y;
    }

    /// Set x and y components with given initializer list.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new_default();
    /// let lst = [0, 5];
    /// vec.set_lst(lst);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(5, vec.y);
    /// ```
    pub fn set_lst(&mut self, lst: [usize; 2]) {
        self.x = lst[0];
        self.y = lst[1];
    }

    /// Set x and y with other size **pt**.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new_default();
    /// vec.set_self(USize2::new(9, 8));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(8, vec.y);
    /// ```
    pub fn set_self(&mut self, pt: USize2) {
        self.x = pt.x;
        self.y = pt.y;
    }

    /// Set both x and y to zero.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.set_zero();
    /// assert_eq!(0, vec.x);
    /// assert_eq!(0, vec.y);
    /// ```
    pub fn set_zero(&mut self) {
        self.x = 0;
        self.y = 0;
    }
}

/// # Binary operations: new instance = this (+) v
impl USize2 {
    /// Computes self + (v, v).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// assert_eq!(7, vec.x);
    /// assert_eq!(13, vec.y);
    /// ```
    pub fn add_scalar(&self, v: usize) -> USize2 {
        return USize2::new(self.x + v, self.y + v);
    }

    /// Computes self + (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize2::new(2, 1));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(14, vec.y);
    /// ```
    pub fn add_vec(&self, v: USize2) -> USize2 {
        return USize2::new(self.x + v.x, self.y + v.y);
    }

    /// Computes self - (v, v).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// assert_eq!(1, vec.x);
    /// assert_eq!(6, vec.y);
    /// ```
    pub fn sub_scalar(&self, v: usize) -> USize2 {
        return USize2::new(self.x - v, self.y - v);
    }

    /// Computes self - (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize2::new(1, 3));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn sub_vec(&self, v: USize2) -> USize2 {
        return USize2::new(self.x - v.x, self.y - v.y);
    }

    /// Computes self * (v, v).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize2::new(1, 3));
    /// vec = vec.mul_scalar(2);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(6, vec.y);
    /// ```
    pub fn mul_scalar(&self, v: usize) -> USize2 {
        return USize2::new(self.x * v, self.y * v);
    }

    /// Computes self * (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize2::new(1, 3));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(USize2::new(3, 2));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(12, vec.y);
    /// ```
    pub fn mul_vec(&self, v: USize2) -> USize2 {
        return USize2::new(self.x * v.x, self.y * v.y);
    }

    /// Computes self / (v, v).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize2::new(1, 3));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(USize2::new(3, 2));
    /// vec = vec.div_scalar(4);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn div_scalar(&self, v: usize) -> USize2 {
        return USize2::new(self.x / v, self.y / v);
    }

    /// Computes self / (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(USize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(USize2::new(1, 3));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(USize2::new(3, 2));
    /// vec = vec.div_scalar(4);
    /// vec = vec.div_vec(USize2::new(3, 1));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn div_vec(&self, v: USize2) -> USize2 {
        return USize2::new(self.x / v.x, self.y / v.y);
    }
}

/// # Binary operations: new instance = v (+) this
impl USize2 {
    /// Computes (v, v) - self.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.rsub_scalar(9);
    /// assert_eq!(6, vec.x);
    /// assert_eq!(0, vec.y);
    /// ```
    pub fn rsub_scalar(&self, v: usize) -> USize2 {
        return USize2::new(v - self.x, v - self.y);
    }

    /// Computes (v.x, v.y) - self.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec = vec.rsub_scalar(9);
    /// vec = vec.rsub_vec(USize2::new(6, 3));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn rsub_vec(&self, v: USize2) -> USize2 {
        return USize2::new(v.x - self.x, v.y - self.y);
    }

    /// Computes (v, v) / self.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(4, 3);
    /// vec = vec.rdiv_scalar(12);
    /// assert_eq!(3, vec.x);
    /// assert_eq!(vec.y, 4);
    /// ```
    pub fn rdiv_scalar(&self, v: usize) -> USize2 {
        return USize2::new(v / self.x, v / self.y);
    }

    /// Computes (v.x, v.y) / self.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(4, 3);
    /// vec = vec.rdiv_scalar(12);
    /// vec = vec.rdiv_vec(USize2::new(3, 16));
    /// assert_eq!(1, vec.x);
    /// assert_eq!(4, vec.y);
    /// ```
    pub fn rdiv_vec(&self, v: USize2) -> USize2 {
        return USize2::new(v.x / self.x, v.y / self.y);
    }
}

/// # Augmented operations: usizehis (+)= v
impl USize2 {
    /// Computes self += (v, v).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// assert_eq!(7, vec.x);
    /// assert_eq!(vec.y, 13);
    /// ```
    pub fn iadd_scalar(&mut self, v: usize) {
        self.x = usize::add(self.x, v);
        self.y = usize::add(self.y, v);
    }

    /// Computes self += (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize2::new(2, 1));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(vec.y, 14);
    /// ```
    pub fn iadd_vec(&mut self, v: USize2) {
        self.x = usize::add(self.x, v.x);
        self.y = usize::add(self.y, v.y);
    }

    /// Computes self -= (v, v).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// assert_eq!(1, vec.x);
    /// assert_eq!(6, vec.y);
    /// ```
    pub fn isub_scalar(&mut self, v: usize) {
        self.x = usize::sub(self.x, v);
        self.y = usize::sub(self.y, v);
    }

    /// Computes self -= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize2::new(1, 3));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn isub_vec(&mut self, v: USize2) {
        self.x = usize::sub(self.x, v.x);
        self.y = usize::sub(self.y, v.y);
    }

    /// Computes self *= (v, v).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize2::new(1, 3));
    /// vec.imul_scalar(2);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(6, vec.y);
    /// ```
    pub fn imul_scalar(&mut self, v: usize) {
        self.x = usize::mul(self.x, v);
        self.y = usize::mul(self.y, v);
    }

    /// Computes self *= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize2::new(1, 3));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(USize2::new(3, 2));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(12, vec.y);
    /// ```
    pub fn imul_vec(&mut self, v: USize2) {
        self.x = usize::mul(self.x, v.x);
        self.y = usize::mul(self.y, v.y);
    }

    /// Computes self /= (v, v).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize2::new(1, 3));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(USize2::new(3, 2));
    /// vec.idiv_scalar(4);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn idiv_scalar(&mut self, v: usize) {
        self.x = usize::div(self.x, v);
        self.y = usize::div(self.y, v);
    }

    /// Computes self /= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut vec = USize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(USize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(USize2::new(1, 3));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(USize2::new(3, 2));
    /// vec.idiv_scalar(4);
    /// vec.idiv_vec(USize2::new(3, 1));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn idiv_vec(&mut self, v: USize2) {
        self.x = usize::div(self.x, v.x);
        self.y = usize::div(self.y, v.y);
    }
}

/// # Basic getters
impl USize2 {
    /// Returns const reference to the **i** -th element of the size.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec = USize2::new(8, 9);
    /// assert_eq!(*vec.at(0), 8);
    /// assert_eq!(*vec.at(1), 9);
    /// ```
    pub fn at(&self, i: usize) -> &usize {
        match i {
            0 => return &self.x,
            1 => return &self.y,
            _ => { panic!() }
        }
    }

    /// Returns reference to the **i** -th element of the size.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let mut a = USize2::new_default();
    /// *a.at_mut(0) =  10;
    /// *a.at_mut(1) =  20;
    /// assert_eq!(*a.at(0), 10);
    /// assert_eq!(*a.at(1), 20);
    /// ```
    pub fn at_mut(&mut self, i: usize) -> &mut usize {
        match i {
            0 => return &mut self.x,
            1 => return &mut self.y,
            _ => { panic!() }
        }
    }

    /// Returns the sum of all the components (i.e. x + y).
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec = USize2::new(3, 7);
    /// let sum = vec.sum();
    /// assert_eq!(sum, 10);
    /// ```
    pub fn sum(&self) -> usize {
        return self.x + self.y;
    }

    /// Returns the minimum value among x and y.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec = USize2::new(3, 7);
    /// let min = vec.min();
    /// assert_eq!(min, 3);
    /// ```
    pub fn min(&self) -> usize {
        return self.x.min(self.y);
    }

    /// Returns the maximum value among x and y.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec = USize2::new(3, 7);
    /// let max = vec.max();
    /// assert_eq!(max, 7);
    /// ```
    pub fn max(&self) -> usize {
        return self.x.max(self.y);
    }

    /// Returns the index of the dominant axis.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec = USize2::new(3, 7);
    /// let dominant_axis = vec.dominant_axis();
    /// assert_eq!(dominant_axis, 1);
    /// ```
    pub fn dominant_axis(&self) -> usize {
        match self.x > self.y {
            true => 0,
            false => 1
        }
    }

    /// Returns the index of the subminant axis.
    /// ```
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec = USize2::new(3, 7);
    /// let subminant_axis = vec.subminant_axis();
    /// assert_eq!(subminant_axis, 0);
    /// ```
    pub fn subminant_axis(&self) -> usize {
        match self.x < self.y {
            true => 0,
            false => 1
        }
    }

    /// Returns true if **other** is the same as self size.
    pub fn is_equal(&self, other: &USize2) -> bool {
        return self.x == other.x && self.y == other.y;
    }
}

/// Copy constructor.
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec5 = USize2::new_lst([7, 6]);
/// let mut vec6 = vec5.clone();
/// vec6.x = 10;
/// assert_eq!(10, vec6.x);
/// assert_eq!(7, vec5.x);
/// ```
impl Clone for USize2 {
    fn clone(&self) -> Self {
        return USize2 {
            x: self.x,
            y: self.y,
        };
    }
}

///
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let vec = USize2::new(5, 1);
/// let mut vec2 = USize2::new(3, 3);
/// vec2 = vec;
/// assert_eq!(5, vec2.x);
/// assert_eq!(vec2.y, 1);
/// ```
impl Copy for USize2 {}

/// # Operators
/// Returns const reference to the **i** -th element of the size.
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let vec = USize2::new(8, 9);
/// assert_eq!(vec[0], 8);
/// assert_eq!(vec[1], 9);
/// ```
impl Index<usize> for USize2 {
    type Output = usize;
    fn index(&self, index: usize) -> &Self::Output {
        return self.at(index);
    }
}

/// Returns reference to the **i** -th element of the size.
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(8, 9);
/// vec[0] = 7;
/// vec[1] = 6;
/// assert_eq!(7, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl IndexMut<usize> for USize2 {
    fn index_mut(&mut self, index: usize) -> &mut Self::Output {
        return self.at_mut(index);
    }
}

/// Computes self += (v, v)
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec += 4;
/// assert_eq!(7, vec.x);
/// assert_eq!(vec.y, 13);
/// ```
impl AddAssign<usize> for USize2 {
    fn add_assign(&mut self, rhs: usize) {
        self.iadd_scalar(rhs);
    }
}

/// Computes self += (v.x, v.y)
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec += 4;
/// vec += USize2::new(2, 1);
/// assert_eq!(9, vec.x);
/// assert_eq!(vec.y, 14);
/// ```
impl AddAssign for USize2 {
    fn add_assign(&mut self, rhs: Self) {
        self.iadd_vec(rhs);
    }
}

/// Computes self -= (v, v)
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec += 4;
/// vec += USize2::new(2, 1);
/// vec -= 8;
/// assert_eq!(1, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl SubAssign<usize> for USize2 {
    fn sub_assign(&mut self, rhs: usize) {
        self.isub_scalar(rhs);
    }
}

/// Computes self -= (v.x, v.y)
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec += 4;
/// vec += USize2::new(2, 1);
/// vec -= 8;
/// vec -= USize2::new(1, 3);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl SubAssign for USize2 {
    fn sub_assign(&mut self, rhs: Self) {
        self.isub_vec(rhs);
    }
}

/// Computes self *= (v, v)
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec += 4;
/// vec += USize2::new(2, 1);
/// vec -= 8;
/// vec -= USize2::new(1, 3);
/// vec *= 2;
/// assert_eq!(0, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl MulAssign<usize> for USize2 {
    fn mul_assign(&mut self, rhs: usize) {
        self.imul_scalar(rhs);
    }
}

/// Computes self *= (v.x, v.y)
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec += 4;
/// vec += USize2::new(2, 1);
/// vec -= 8;
/// vec -= USize2::new(1, 3);
/// vec *= 2;
/// vec *= USize2::new(3, 2);
/// assert_eq!(0, vec.x);
/// assert_eq!(12, vec.y);
/// ```
impl MulAssign for USize2 {
    fn mul_assign(&mut self, rhs: Self) {
        self.imul_vec(rhs);
    }
}

/// Computes self /= (v, v)
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec += 4;
/// vec += USize2::new(2, 1);
/// vec -= 8;
/// vec -= USize2::new(1, 3);
/// vec *= 2;
/// vec *= USize2::new(3, 2);
/// vec /= 4;
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl DivAssign<usize> for USize2 {
    fn div_assign(&mut self, rhs: usize) {
        self.idiv_scalar(rhs);
    }
}

/// Computes self /= (v.x, v.y)
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec += 4;
/// vec += USize2::new(2, 1);
/// vec -= 8;
/// vec -= USize2::new(1, 3);
/// vec *= 2;
/// vec *= USize2::new(3, 2);
/// vec /= 4;
/// vec /= USize2::new(3, 1);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl DivAssign for USize2 {
    fn div_assign(&mut self, rhs: Self) {
        self.idiv_vec(rhs);
    }
}

/// Returns true if **other** is the same as self size.
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new_default();
/// let vec2 = USize2::new(3, 7);
/// let vec3 = USize2::new(3, 5);
/// let vec4 = USize2::new(5, 1);
/// vec = vec2;
/// assert_eq!(vec == vec2, true);
/// assert_eq!(vec == vec3, false);
/// assert_eq!(vec != vec2, false);
/// assert_eq!(vec != vec3, true);
/// assert_eq!(vec != vec4, true);
/// ```
impl PartialEq for USize2 {
    fn eq(&self, other: &Self) -> bool {
        return self.is_equal(other);
    }
}

impl Eq for USize2 {}

/// Computes (a, a) + (b.x, b.y).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec = vec + 4;
/// assert_eq!(7, vec.x);
/// assert_eq!(vec.y, 13);
/// ```
impl Add<usize> for USize2 {
    type Output = USize2;
    fn add(self, rhs: usize) -> Self::Output {
        return self.add_scalar(rhs);
    }
}

/// Computes (a.x, a.y) + (b.x, b.y).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + USize2::new(2, 1);
/// assert_eq!(9, vec.x);
/// assert_eq!(vec.y, 14);
/// ```
impl Add for USize2 {
    type Output = USize2;
    fn add(self, rhs: Self) -> Self::Output {
        return self.add_vec(rhs);
    }
}

/// Computes (a.x, a.y) - (b, b).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + USize2::new(2, 1);
/// vec = vec - 8;
/// assert_eq!(1, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl Sub<usize> for USize2 {
    type Output = USize2;
    fn sub(self, rhs: usize) -> Self::Output {
        return self.sub_scalar(rhs);
    }
}

/// Computes (a.x, a.y) - (b.x, b.y).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + USize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - USize2::new(1, 3);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl Sub for USize2 {
    type Output = USize2;
    fn sub(self, rhs: Self) -> Self::Output {
        return self.sub_vec(rhs);
    }
}

/// Computes (a.x, a.y) * (b, b).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + USize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - USize2::new(1, 3);
/// vec = vec * 2;
/// assert_eq!(0, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl Mul<usize> for USize2 {
    type Output = USize2;
    fn mul(self, rhs: usize) -> Self::Output {
        return self.mul_scalar(rhs);
    }
}

/// Computes (a.x, a.y) * (b.x, b.y).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + USize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - USize2::new(1, 3);
/// vec = vec * 2;
/// vec = vec * USize2::new(3, 2);
/// assert_eq!(0, vec.x);
/// assert_eq!(12, vec.y);
/// ```
impl Mul for USize2 {
    type Output = USize2;
    fn mul(self, rhs: Self) -> Self::Output {
        return self.mul_vec(rhs);
    }
}

/// Computes (a.x, a.y) / (b, b).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + USize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - USize2::new(1, 3);
/// vec = vec * 2;
/// vec = vec * USize2::new(3, 2);
/// vec = vec / 4;
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl Div<usize> for USize2 {
    type Output = USize2;
    fn div(self, rhs: usize) -> Self::Output {
        return self.div_scalar(rhs);
    }
}

/// Computes (a.x, a.y) / (b.x, b.y).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// let mut vec = USize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + USize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - USize2::new(1, 3);
/// vec = vec * 2;
/// vec = vec * USize2::new(3, 2);
/// vec = vec / 4;
/// vec = vec / USize2::new(3, 1);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl Div for USize2 {
    type Output = USize2;
    fn div(self, rhs: Self) -> Self::Output {
        return self.div_vec(rhs);
    }
}

impl Debug for USize2 {
    /// # Example
    /// ```
    ///
    /// use vox_geometry_rust::usize2::USize2;
    /// let vec = USize2::new(10, 20);
    /// assert_eq!(format!("{:?}", vec), "(10, 20)");
    ///
    /// assert_eq!(format!("{:#?}", vec), "(
    ///     10,
    ///     20,
    /// )");
    /// ```
    fn fmt(&self, f: &mut Formatter<'_>) -> Result {
        f.debug_tuple("")
            .field(&self.x)
            .field(&self.y)
            .finish()
    }
}

/// # utility
/// Returns element-wise min size: (min(a.x, b.x), min(a.y, b.y)).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// use vox_geometry_rust::usize2::min;
/// let vec = USize2::new(5, 1);
/// let vec2 = USize2::new(3, 3);
/// let min_size = min(&vec, &vec2);
/// assert_eq!(USize2::new(3, 1), min_size);
/// ```
pub fn min(a: &USize2, b: &USize2) -> USize2 {
    return USize2::new(usize::min(a.x, b.x), usize::min(a.y, b.y));
}

/// Returns element-wise max size: (max(a.x, b.x), max(a.y, b.y)).
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// use vox_geometry_rust::usize2::max;
/// let vec = USize2::new(5, 1);
/// let vec2 = USize2::new(3, 3);
/// let min_size = max(&vec, &vec2);
/// assert_eq!(USize2::new(5, 3), min_size);
/// ```
pub fn max(a: &USize2, b: &USize2) -> USize2 {
    return USize2::new(usize::max(a.x, b.x), usize::max(a.y, b.y));
}

/// Returns element-wise clamped size.
/// ```
/// use vox_geometry_rust::usize2::USize2;
/// use vox_geometry_rust::usize2::clamp;
/// let vec = USize2::new(2, 4);
/// let low = USize2::new(3, 1);
/// let high = USize2::new(5, 2);
/// let clamped_vec = clamp(&vec, &low, &high);
/// assert_eq!(USize2::new(3, 2), clamped_vec);
/// ```
pub fn clamp(v: &USize2, low: &USize2, high: &USize2) -> USize2 {
    return USize2::new(usize::clamp(v.x, low.x, high.x),
                       usize::clamp(v.y, low.y, high.y));
}