/*
 * // Copyright (c) 2021 Feng Yang
 * //
 * // I am making my contributions/submissions to this project solely in my
 * // personal capacity and am not conveying any rights to any intellectual
 * // property of any third parties.
 */

use num::Float;
use crate::vector2::Vector2;

///
/// #      Class for 2-D ray.
///
/// - tparam     T     The value type.
///
pub struct Ray2<T: Float> {
    /// The origin of the ray.
    pub origin: Vector2<T>,

    /// The direction of the ray.
    pub direction: Vector2<T>,
}

/// Float-type 2-D ray.
pub type Ray2F = Ray2<f32>;

/// Double-type 2-D ray.
pub type Ray2D = Ray2<f64>;

impl<T: Float> Ray2<T> {
    /// Constructs an empty ray that points (1, 0) from (0, 0).
    /// ```
    /// use vox_geometry_rust::ray2::Ray2D;
    /// use vox_geometry_rust::vector2::Vector2D;
    /// let ray = Ray2D::new_default();
    /// assert_eq!(Vector2D::new_default(), ray.origin);
    /// assert_eq!(Vector2D::new(1.0, 0.0), ray.direction);
    /// ```
    pub fn new_default() -> Ray2<T> {
        return Ray2 {
            origin: Vector2::new_default(),
            direction: Vector2::new(T::one(), T::zero()),
        };
    }

    /// Constructs a ray with given origin and direction.
    /// ```
    /// use vox_geometry_rust::ray2::Ray2D;
    /// use vox_geometry_rust::vector2::Vector2D;
    /// let ray2 = Ray2D::new(Vector2D::new_lst([1.0, 2.0]), Vector2D::new_lst([3.0, 4.0]));
    /// assert_eq!(Vector2D::new(1.0, 2.0), ray2.origin);
    /// assert_eq!(Vector2D::new(3.0, 4.0).normalized(), ray2.direction);
    /// ```
    pub fn new(new_origin: Vector2<T>, new_direction: Vector2<T>) -> Ray2<T> {
        return Ray2 {
            origin: new_origin,
            direction: new_direction.normalized(),
        };
    }

    /// Returns a point on the ray at distance **t**.
    /// ```
    /// use vox_geometry_rust::ray2::Ray2D;
    /// use vox_geometry_rust::vector2::Vector2D;
    /// let ray = Ray2D::new_default();
    /// assert_eq!(Vector2D::new(4.5, 0.0), ray.point_at(4.5));
    /// ```
    pub fn point_at(&self, t: T) -> Vector2<T> {
        return self.origin + self.direction * t;
    }
}