/*
 * // Copyright (c) 2021 Feng Yang
 * //
 * // I am making my contributions/submissions to this project solely in my
 * // personal capacity and am not conveying any rights to any intellectual
 * // property of any third parties.
 */

use num::Float;
use std::ops::{Index, IndexMut, AddAssign, SubAssign, MulAssign, DivAssign, Add, Neg, Sub, Mul, Div};
use std::fmt::{Debug, Formatter, Result};

///
/// # 2-D point class.
///
/// This class defines simple 2-D point data.
///
/// - tparam T - Type of the element
///
pub struct Point2<T: Float> {
    /// X (or the first) component of the point.
    pub x: T,

    /// Y (or the second) component of the point.
    pub y: T,
}

/// Float-type 2D point.
pub type Point2F = Point2<f32>;
/// Double-type 2D point.
pub type Point2D = Point2<f64>;

/// # Constructors
impl<T: Float> Point2<T> {
    /// Constructs default point (0, 0).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new_default();
    /// assert_eq!(0.0, vec.x);
    /// assert_eq!(0.0, vec.y);
    /// ```
    pub fn new_default() -> Point2<T> {
        return Point2 {
            x: T::zero(),
            y: T::zero(),
        };
    }

    /// Constructs point with given parameters \p x_ and \p y_.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec2 = Point2F::new(5.0, 3.0);
    /// assert_eq!(5.0, vec2.x);
    /// assert_eq!(3.0, vec2.y);
    /// ```
    pub fn new(x_: T, y_: T) -> Point2<T> {
        return Point2 {
            x: x_,
            y: y_,
        };
    }

    /// Constructs point with initializer list.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec5 = Point2F::new_lst([7.0, 6.0]);
    /// assert_eq!(7.0, vec5.x);
    /// assert_eq!(6.0, vec5.y);
    /// ```
    pub fn new_lst(lst: [T; 2]) -> Point2<T> {
        return Point2 {
            x: lst[0],
            y: lst[1],
        };
    }
}

/// # Basic setters
impl<T: Float> Point2<T> {
    /// Set both x and y components to **s**.
    pub fn set_scalar(&mut self, s: T) {
        self.x = s;
        self.y = s;
    }

    /// Set x and y components with given parameters.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new_default();
    /// vec.set_scalar2(4.0, 2.0);
    /// assert_eq!(4.0, vec.x);
    /// assert_eq!(2.0, vec.y);
    /// ```
    pub fn set_scalar2(&mut self, x: T, y: T) {
        self.x = x;
        self.y = y;
    }

    /// Set x and y components with given initializer list.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new_default();
    /// let lst = [0.0, 5.0];
    /// vec.set_lst(lst);
    /// assert_eq!(0.0, vec.x);
    /// assert_eq!(5.0, vec.y);
    /// ```
    pub fn set_lst(&mut self, lst: [T; 2]) {
        self.x = lst[0];
        self.y = lst[1];
    }

    /// Set x and y with other point **pt**.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new_default();
    /// vec.set_self(Point2F::new(9.0, 8.0));
    /// assert_eq!(9.0, vec.x);
    /// assert_eq!(8.0, vec.y);
    /// ```
    pub fn set_self(&mut self, pt: Point2<T>) {
        self.x = pt.x;
        self.y = pt.y;
    }

    /// Set both x and y to zero.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.set_zero();
    /// assert_eq!(0.0, vec.x);
    /// assert_eq!(0.0, vec.y);
    /// ```
    pub fn set_zero(&mut self) {
        self.x = T::zero();
        self.y = T::zero();
    }
}

/// # Binary operations: new instance = this (+) v
impl<T: Float> Point2<T> {
    /// Computes self + (v, v).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.add_scalar(4.0);
    /// assert_eq!(7.0, vec.x);
    /// assert_eq!(13.0, vec.y);
    /// ```
    pub fn add_scalar(&self, v: T) -> Point2<T> {
        return Point2::new(self.x + v, self.y + v);
    }

    /// Computes self + (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.add_scalar(4.0);
    /// vec = vec.add_vec(Point2F::new(-2.0, 1.0));
    /// assert_eq!(5.0, vec.x);
    /// assert_eq!(14.0, vec.y);
    /// ```
    pub fn add_vec(&self, v: Point2<T>) -> Point2<T> {
        return Point2::new(self.x + v.x, self.y + v.y);
    }

    /// Computes self - (v, v).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.add_scalar(4.0);
    /// vec = vec.add_vec(Point2F::new(-2.0, 1.0));
    /// vec = vec.sub_scalar(8.0);
    /// assert_eq!(-3.0, vec.x);
    /// assert_eq!(6.0, vec.y);
    /// ```
    pub fn sub_scalar(&self, v: T) -> Point2<T> {
        return Point2::new(self.x - v, self.y - v);
    }

    /// Computes self - (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.add_scalar(4.0);
    /// vec = vec.add_vec(Point2F::new(-2.0, 1.0));
    /// vec = vec.sub_scalar(8.0);
    /// vec = vec.sub_vec(Point2F::new(-5.0, 3.0));
    /// assert_eq!(2.0, vec.x);
    /// assert_eq!(3.0, vec.y);
    /// ```
    pub fn sub_vec(&self, v: Point2<T>) -> Point2<T> {
        return Point2::new(self.x - v.x, self.y - v.y);
    }

    /// Computes self * (v, v).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.add_scalar(4.0);
    /// vec = vec.add_vec(Point2F::new(-2.0, 1.0));
    /// vec = vec.sub_scalar(8.0);
    /// vec = vec.sub_vec(Point2F::new(-5.0, 3.0));
    /// vec = vec.mul_scalar(2.0);
    /// assert_eq!(4.0, vec.x);
    /// assert_eq!(6.0, vec.y);
    /// ```
    pub fn mul_scalar(&self, v: T) -> Point2<T> {
        return Point2::new(self.x * v, self.y * v);
    }

    /// Computes self * (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.add_scalar(4.0);
    /// vec = vec.add_vec(Point2F::new(-2.0, 1.0));
    /// vec = vec.sub_scalar(8.0);
    /// vec = vec.sub_vec(Point2F::new(-5.0, 3.0));
    /// vec = vec.mul_scalar(2.0);
    /// vec = vec.mul_vec(Point2F::new(3.0, -2.0));
    /// assert_eq!(12.0, vec.x);
    /// assert_eq!(-12.0, vec.y);
    /// ```
    pub fn mul_vec(&self, v: Point2<T>) -> Point2<T> {
        return Point2::new(self.x * v.x, self.y * v.y);
    }

    /// Computes self / (v, v).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.add_scalar(4.0);
    /// vec = vec.add_vec(Point2F::new(-2.0, 1.0));
    /// vec = vec.sub_scalar(8.0);
    /// vec = vec.sub_vec(Point2F::new(-5.0, 3.0));
    /// vec = vec.mul_scalar(2.0);
    /// vec = vec.mul_vec(Point2F::new(3.0, -2.0));
    /// vec = vec.div_scalar(4.0);
    /// assert_eq!(3.0, vec.x);
    /// assert_eq!(-3.0, vec.y);
    /// ```
    pub fn div_scalar(&self, v: T) -> Point2<T> {
        return Point2::new(self.x / v, self.y / v);
    }

    /// Computes self / (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.add_scalar(4.0);
    /// vec = vec.add_vec(Point2F::new(-2.0, 1.0));
    /// vec = vec.sub_scalar(8.0);
    /// vec = vec.sub_vec(Point2F::new(-5.0, 3.0));
    /// vec = vec.mul_scalar(2.0);
    /// vec = vec.mul_vec(Point2F::new(3.0, -2.0));
    /// vec = vec.div_scalar(4.0);
    /// vec = vec.div_vec(Point2F::new(3.0, -1.0));
    /// assert_eq!(1.0, vec.x);
    /// assert_eq!(3.0, vec.y);
    /// ```
    pub fn div_vec(&self, v: Point2<T>) -> Point2<T> {
        return Point2::new(self.x / v.x, self.y / v.y);
    }
}

/// # Binary operations: new instance = v (+) this
impl<T: Float> Point2<T> {
    /// Computes (v, v) - self.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.rsub_scalar(8.0);
    /// assert_eq!(5.0, vec.x);
    /// assert_eq!(-1.0, vec.y);
    /// ```
    pub fn rsub_scalar(&self, v: T) -> Point2<T> {
        return Point2::new(v - self.x, v - self.y);
    }

    /// Computes (v.x, v.y) - self.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec = vec.rsub_scalar(8.0);
    /// vec = vec.rsub_vec(Point2F::new(-5.0, 3.0));
    /// assert_eq!(-10.0, vec.x);
    /// assert_eq!(4.0, vec.y);
    /// ```
    pub fn rsub_vec(&self, v: Point2<T>) -> Point2<T> {
        return Point2::new(v.x - self.x, v.y - self.y);
    }

    /// Computes (v, v) / self.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(-4.0, -3.0);
    /// vec = vec.rdiv_scalar(12.0);
    /// assert_eq!(-3.0, vec.x);
    /// assert_eq!(vec.y, -4.0);
    /// ```
    pub fn rdiv_scalar(&self, v: T) -> Point2<T> {
        return Point2::new(v / self.x, v / self.y);
    }

    /// Computes (v.x, v.y) / self.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(-4.0, -3.0);
    /// vec = vec.rdiv_scalar(12.0);
    /// vec = vec.rdiv_vec(Point2F::new(3.0, -16.0));
    /// assert_eq!(-1.0, vec.x);
    /// assert_eq!(4.0, vec.y);
    /// ```
    pub fn rdiv_vec(&self, v: Point2<T>) -> Point2<T> {
        return Point2::new(v.x / self.x, v.y / self.y);
    }
}

/// # Augmented operations: this (+)= v
impl<T: Float> Point2<T> {
    /// Computes self += (v, v).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.iadd_scalar(4.0);
    /// assert_eq!(7.0, vec.x);
    /// assert_eq!(vec.y, 13.0);
    /// ```
    pub fn iadd_scalar(&mut self, v: T) {
        self.x = T::add(self.x, v);
        self.y = T::add(self.y, v);
    }

    /// Computes self += (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.iadd_scalar(4.0);
    /// vec.iadd_vec(Point2F::new(-2.0, 1.0));
    /// assert_eq!(5.0, vec.x);
    /// assert_eq!(vec.y, 14.0);
    /// ```
    pub fn iadd_vec(&mut self, v: Point2<T>) {
        self.x = T::add(self.x, v.x);
        self.y = T::add(self.y, v.y);
    }

    /// Computes self -= (v, v).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.iadd_scalar(4.0);
    /// vec.iadd_vec(Point2F::new(-2.0, 1.0));
    /// vec.isub_scalar(8.0);
    /// assert_eq!(-3.0, vec.x);
    /// assert_eq!(6.0, vec.y);
    /// ```
    pub fn isub_scalar(&mut self, v: T) {
        self.x = T::sub(self.x, v);
        self.y = T::sub(self.y, v);
    }

    /// Computes self -= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.iadd_scalar(4.0);
    /// vec.iadd_vec(Point2F::new(-2.0, 1.0));
    /// vec.isub_scalar(8.0);
    /// vec.isub_vec(Point2F::new(-5.0, 3.0));
    /// assert_eq!(2.0, vec.x);
    /// assert_eq!(3.0, vec.y);
    /// ```
    pub fn isub_vec(&mut self, v: Point2<T>) {
        self.x = T::sub(self.x, v.x);
        self.y = T::sub(self.y, v.y);
    }

    /// Computes self *= (v, v).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.iadd_scalar(4.0);
    /// vec.iadd_vec(Point2F::new(-2.0, 1.0));
    /// vec.isub_scalar(8.0);
    /// vec.isub_vec(Point2F::new(-5.0, 3.0));
    /// vec.imul_scalar(2.0);
    /// assert_eq!(4.0, vec.x);
    /// assert_eq!(6.0, vec.y);
    /// ```
    pub fn imul_scalar(&mut self, v: T) {
        self.x = T::mul(self.x, v);
        self.y = T::mul(self.y, v);
    }

    /// Computes self *= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.iadd_scalar(4.0);
    /// vec.iadd_vec(Point2F::new(-2.0, 1.0));
    /// vec.isub_scalar(8.0);
    /// vec.isub_vec(Point2F::new(-5.0, 3.0));
    /// vec.imul_scalar(2.0);
    /// vec.imul_vec(Point2F::new(3.0, -2.0));
    /// assert_eq!(12.0, vec.x);
    /// assert_eq!(-12.0, vec.y);
    /// ```
    pub fn imul_vec(&mut self, v: Point2<T>) {
        self.x = T::mul(self.x, v.x);
        self.y = T::mul(self.y, v.y);
    }

    /// Computes self /= (v, v).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.iadd_scalar(4.0);
    /// vec.iadd_vec(Point2F::new(-2.0, 1.0));
    /// vec.isub_scalar(8.0);
    /// vec.isub_vec(Point2F::new(-5.0, 3.0));
    /// vec.imul_scalar(2.0);
    /// vec.imul_vec(Point2F::new(3.0, -2.0));
    /// vec.idiv_scalar(4.0);
    /// assert_eq!(3.0, vec.x);
    /// assert_eq!(-3.0, vec.y);
    /// ```
    pub fn idiv_scalar(&mut self, v: T) {
        self.x = T::div(self.x, v);
        self.y = T::div(self.y, v);
    }

    /// Computes self /= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut vec = Point2F::new(3.0, 9.0);
    /// vec.iadd_scalar(4.0);
    /// vec.iadd_vec(Point2F::new(-2.0, 1.0));
    /// vec.isub_scalar(8.0);
    /// vec.isub_vec(Point2F::new(-5.0, 3.0));
    /// vec.imul_scalar(2.0);
    /// vec.imul_vec(Point2F::new(3.0, -2.0));
    /// vec.idiv_scalar(4.0);
    /// vec.idiv_vec(Point2F::new(3.0, -1.0));
    /// assert_eq!(1.0, vec.x);
    /// assert_eq!(3.0, vec.y);
    /// ```
    pub fn idiv_vec(&mut self, v: Point2<T>) {
        self.x = T::div(self.x, v.x);
        self.y = T::div(self.y, v.y);
    }
}

/// # Basic getters
impl<T: Float> Point2<T> {
    /// Returns const reference to the **i** -th element of the point.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(8.0, 9.0);
    /// assert_eq!(*vec.at(0), 8.0);
    /// assert_eq!(*vec.at(1), 9.0);
    /// ```
    pub fn at(&self, i: usize) -> &T {
        match i {
            0 => return &self.x,
            1 => return &self.y,
            _ => { panic!() }
        }
    }

    /// Returns reference to the **i** -th element of the point.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let mut a = Point2F::new_default();
    /// *a.at_mut(0) =  10.0_f32;
    /// *a.at_mut(1) =  20.0_f32;
    /// assert_eq!(*a.at(0), 10.0);
    /// assert_eq!(*a.at(1), 20.0);
    /// ```
    pub fn at_mut(&mut self, i: usize) -> &mut T {
        match i {
            0 => return &mut self.x,
            1 => return &mut self.y,
            _ => { panic!() }
        }
    }

    /// Returns the sum of all the components (i.e. x + y).
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(3.0, 7.0);
    /// let sum = vec.sum();
    /// assert_eq!(sum, 10.0);
    /// ```
    pub fn sum(&self) -> T {
        return self.x + self.y;
    }

    /// Returns the minimum value among x and y.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(3.0, 7.0);
    /// let min = vec.min();
    /// assert_eq!(min, 3.0);
    /// ```
    pub fn min(&self) -> T {
        return self.x.min(self.y);
    }

    /// Returns the maximum value among x and y.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(3.0, 7.0);
    /// let max = vec.max();
    /// assert_eq!(max, 7.0);
    /// ```
    pub fn max(&self) -> T {
        return self.x.max(self.y);
    }

    /// Returns the absolute minimum value among x and y.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(-3.0, -7.0);
    /// let absmin = vec.absmin();
    /// assert_eq!(absmin, -3.0);
    /// ```
    pub fn absmin(&self) -> T {
        return crate::math_utils::absmin(self.x, self.y);
    }

    /// Returns the absolute maximum value among x and y.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(-3.0, -7.0);
    /// let absmax = vec.absmax();
    /// assert_eq!(absmax, -7.0);
    /// ```
    pub fn absmax(&self) -> T {
        return crate::math_utils::absmax(self.x, self.y);
    }

    /// Returns the index of the dominant axis.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(3.0, 7.0);
    /// let dominant_axis = vec.dominant_axis();
    /// assert_eq!(dominant_axis, 1);
    /// ```
    pub fn dominant_axis(&self) -> usize {
        match self.x.abs() > self.y.abs() {
            true => 0,
            false => 1
        }
    }

    /// Returns the index of the subminant axis.
    /// ```
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(3.0, 7.0);
    /// let subminant_axis = vec.subminant_axis();
    /// assert_eq!(subminant_axis, 0);
    /// ```
    pub fn subminant_axis(&self) -> usize {
        match self.x.abs() < self.y.abs() {
            true => 0,
            false => 1
        }
    }

    /// Returns true if **other** is the same as self point.
    pub fn is_equal(&self, other: &Point2<T>) -> bool {
        return self.x == other.x && self.y == other.y;
    }
}

/// Copy constructor.
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec5 = Point2F::new_lst([7.0, 6.0]);
/// let mut vec6 = vec5.clone();
/// vec6.x = 10.0;
/// assert_eq!(10.0, vec6.x);
/// assert_eq!(7.0, vec5.x);
/// ```
impl<T: Float> Clone for Point2<T> {
    fn clone(&self) -> Self {
        return Point2 {
            x: self.x,
            y: self.y,
        };
    }
}

///
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let vec = Point2F::new(5.0, 1.0);
/// let mut vec2 = Point2F::new(3.0, 3.0);
/// vec2 = vec;
/// assert_eq!(5.0, vec2.x);
/// assert_eq!(vec2.y, 1.0);
/// ```
impl<T: Float> Copy for Point2<T> {}

/// # Operators
/// Returns const reference to the **i** -th element of the point.
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let vec = Point2F::new(8.0, 9.0);
/// assert_eq!(vec[0], 8.0);
/// assert_eq!(vec[1], 9.0);
/// ```
impl<T: Float> Index<usize> for Point2<T> {
    type Output = T;
    fn index(&self, index: usize) -> &Self::Output {
        return self.at(index);
    }
}

/// Returns reference to the **i** -th element of the point.
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(8.0, 9.0);
/// vec[0] = 7.0;
/// vec[1] = 6.0;
/// assert_eq!(7.0, vec.x);
/// assert_eq!(6.0, vec.y);
/// ```
impl<T: Float> IndexMut<usize> for Point2<T> {
    fn index_mut(&mut self, index: usize) -> &mut Self::Output {
        return self.at_mut(index);
    }
}

/// Computes self += (v, v)
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec += 4.0;
/// assert_eq!(7.0, vec.x);
/// assert_eq!(vec.y, 13.0);
/// ```
impl<T: Float> AddAssign<T> for Point2<T> {
    fn add_assign(&mut self, rhs: T) {
        self.iadd_scalar(rhs);
    }
}

/// Computes self += (v.x, v.y)
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec += 4.0;
/// vec += Point2F::new(-2.0, 1.0);
/// assert_eq!(5.0, vec.x);
/// assert_eq!(vec.y, 14.0);
/// ```
impl<T: Float> AddAssign for Point2<T> {
    fn add_assign(&mut self, rhs: Self) {
        self.iadd_vec(rhs);
    }
}

/// Computes self -= (v, v)
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec += 4.0;
/// vec += Point2F::new(-2.0, 1.0);
/// vec -= 8.0;
/// assert_eq!(-3.0, vec.x);
/// assert_eq!(6.0, vec.y);
/// ```
impl<T: Float> SubAssign<T> for Point2<T> {
    fn sub_assign(&mut self, rhs: T) {
        self.isub_scalar(rhs);
    }
}

/// Computes self -= (v.x, v.y)
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec += 4.0;
/// vec += Point2F::new(-2.0, 1.0);
/// vec -= 8.0;
/// vec -= Point2F::new(-5.0, 3.0);
/// assert_eq!(2.0, vec.x);
/// assert_eq!(3.0, vec.y);
/// ```
impl<T: Float> SubAssign for Point2<T> {
    fn sub_assign(&mut self, rhs: Self) {
        self.isub_vec(rhs);
    }
}

/// Computes self *= (v, v)
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec += 4.0;
/// vec += Point2F::new(-2.0, 1.0);
/// vec -= 8.0;
/// vec -= Point2F::new(-5.0, 3.0);
/// vec *= 2.0;
/// assert_eq!(4.0, vec.x);
/// assert_eq!(6.0, vec.y);
/// ```
impl<T: Float> MulAssign<T> for Point2<T> {
    fn mul_assign(&mut self, rhs: T) {
        self.imul_scalar(rhs);
    }
}

/// Computes self *= (v.x, v.y)
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec += 4.0;
/// vec += Point2F::new(-2.0, 1.0);
/// vec -= 8.0;
/// vec -= Point2F::new(-5.0, 3.0);
/// vec *= 2.0;
/// vec *= Point2F::new(3.0, -2.0);
/// assert_eq!(12.0, vec.x);
/// assert_eq!(-12.0, vec.y);
/// ```
impl<T: Float> MulAssign for Point2<T> {
    fn mul_assign(&mut self, rhs: Self) {
        self.imul_vec(rhs);
    }
}

/// Computes self /= (v, v)
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec += 4.0;
/// vec += Point2F::new(-2.0, 1.0);
/// vec -= 8.0;
/// vec -= Point2F::new(-5.0, 3.0);
/// vec *= 2.0;
/// vec *= Point2F::new(3.0, -2.0);
/// vec /= 4.0;
/// assert_eq!(3.0, vec.x);
/// assert_eq!(-3.0, vec.y);
/// ```
impl<T: Float> DivAssign<T> for Point2<T> {
    fn div_assign(&mut self, rhs: T) {
        self.idiv_scalar(rhs);
    }
}

/// Computes self /= (v.x, v.y)
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec += 4.0;
/// vec += Point2F::new(-2.0, 1.0);
/// vec -= 8.0;
/// vec -= Point2F::new(-5.0, 3.0);
/// vec *= 2.0;
/// vec *= Point2F::new(3.0, -2.0);
/// vec /= 4.0;
/// vec /= Point2F::new(3.0, -1.0);
/// assert_eq!(1.0, vec.x);
/// assert_eq!(3.0, vec.y);
/// ```
impl<T: Float> DivAssign for Point2<T> {
    fn div_assign(&mut self, rhs: Self) {
        self.idiv_vec(rhs);
    }
}

/// Returns true if **other** is the same as self point.
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new_default();
/// let vec2 = Point2F::new(3.0, 7.0);
/// let vec3 = Point2F::new(3.0, 5.0);
/// let vec4 = Point2F::new(5.0, 1.0);
/// vec = vec2;
/// assert_eq!(vec == vec2, true);
/// assert_eq!(vec == vec3, false);
/// assert_eq!(vec != vec2, false);
/// assert_eq!(vec != vec3, true);
/// assert_eq!(vec != vec4, true);
/// ```
impl<T: Float> PartialEq for Point2<T> {
    fn eq(&self, other: &Self) -> bool {
        return self.is_equal(other);
    }
}

impl<T: Float> Eq for Point2<T> {}

impl<T: Float> Neg for Point2<T> {
    type Output = Point2<T>;
    /// Negative sign operator.
    fn neg(self) -> Self::Output {
        return Point2::new(-self.x, -self.y);
    }
}

/// Computes (a, a) + (b.x, b.y).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec = vec + 4.0;
/// assert_eq!(7.0, vec.x);
/// assert_eq!(vec.y, 13.0);
/// ```
impl<T: Float> Add<T> for Point2<T> {
    type Output = Point2<T>;
    fn add(self, rhs: T) -> Self::Output {
        return self.add_scalar(rhs);
    }
}

/// Computes (a.x, a.y) + (b.x, b.y).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec = vec + 4.0;
/// vec = vec + Point2F::new(-2.0, 1.0);
/// assert_eq!(5.0, vec.x);
/// assert_eq!(vec.y, 14.0);
/// ```
impl<T: Float> Add for Point2<T> {
    type Output = Point2<T>;
    fn add(self, rhs: Self) -> Self::Output {
        return self.add_vec(rhs);
    }
}

/// Computes (a.x, a.y) - (b, b).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec = vec + 4.0;
/// vec = vec + Point2F::new(-2.0, 1.0);
/// vec = vec - 8.0;
/// assert_eq!(-3.0, vec.x);
/// assert_eq!(6.0, vec.y);
/// ```
impl<T: Float> Sub<T> for Point2<T> {
    type Output = Point2<T>;
    fn sub(self, rhs: T) -> Self::Output {
        return self.sub_scalar(rhs);
    }
}

/// Computes (a.x, a.y) - (b.x, b.y).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec = vec + 4.0;
/// vec = vec + Point2F::new(-2.0, 1.0);
/// vec = vec - 8.0;
/// vec = vec - Point2F::new(-5.0, 3.0);
/// assert_eq!(2.0, vec.x);
/// assert_eq!(3.0, vec.y);
/// ```
impl<T: Float> Sub for Point2<T> {
    type Output = Point2<T>;
    fn sub(self, rhs: Self) -> Self::Output {
        return self.sub_vec(rhs);
    }
}

/// Computes (a.x, a.y) * (b, b).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec = vec + 4.0;
/// vec = vec + Point2F::new(-2.0, 1.0);
/// vec = vec - 8.0;
/// vec = vec - Point2F::new(-5.0, 3.0);
/// vec = vec * 2.0;
/// assert_eq!(4.0, vec.x);
/// assert_eq!(6.0, vec.y);
/// ```
impl<T: Float> Mul<T> for Point2<T> {
    type Output = Point2<T>;
    fn mul(self, rhs: T) -> Self::Output {
        return self.mul_scalar(rhs);
    }
}

/// Computes (a.x, a.y) * (b.x, b.y).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec = vec + 4.0;
/// vec = vec + Point2F::new(-2.0, 1.0);
/// vec = vec - 8.0;
/// vec = vec - Point2F::new(-5.0, 3.0);
/// vec = vec * 2.0;
/// vec = vec * Point2F::new(3.0, -2.0);
/// assert_eq!(12.0, vec.x);
/// assert_eq!(-12.0, vec.y);
/// ```
impl<T: Float> Mul for Point2<T> {
    type Output = Point2<T>;
    fn mul(self, rhs: Self) -> Self::Output {
        return self.mul_vec(rhs);
    }
}

/// Computes (a.x, a.y) / (b, b).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec = vec + 4.0;
/// vec = vec + Point2F::new(-2.0, 1.0);
/// vec = vec - 8.0;
/// vec = vec - Point2F::new(-5.0, 3.0);
/// vec = vec * 2.0;
/// vec = vec * Point2F::new(3.0, -2.0);
/// vec = vec / 4.0;
/// assert_eq!(3.0, vec.x);
/// assert_eq!(-3.0, vec.y);
/// ```
impl<T: Float> Div<T> for Point2<T> {
    type Output = Point2<T>;
    fn div(self, rhs: T) -> Self::Output {
        return self.div_scalar(rhs);
    }
}

/// Computes (a.x, a.y) / (b.x, b.y).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// let mut vec = Point2F::new(3.0, 9.0);
/// vec = vec + 4.0;
/// vec = vec + Point2F::new(-2.0, 1.0);
/// vec = vec - 8.0;
/// vec = vec - Point2F::new(-5.0, 3.0);
/// vec = vec * 2.0;
/// vec = vec * Point2F::new(3.0, -2.0);
/// vec = vec / 4.0;
/// vec = vec / Point2F::new(3.0, -1.0);
/// assert_eq!(1.0, vec.x);
/// assert_eq!(3.0, vec.y);
/// ```
impl<T: Float> Div for Point2<T> {
    type Output = Point2<T>;
    fn div(self, rhs: Self) -> Self::Output {
        return self.div_vec(rhs);
    }
}

impl<T: Float + Debug> Debug for Point2<T> {
    /// # Example
    /// ```
    ///
    /// use vox_geometry_rust::point2::Point2F;
    /// let vec = Point2F::new(10.0, 20.0);
    /// assert_eq!(format!("{:?}", vec), "(10.0, 20.0)");
    ///
    /// assert_eq!(format!("{:#?}", vec), "(
    ///     10.0,
    ///     20.0,
    /// )");
    /// ```
    fn fmt(&self, f: &mut Formatter<'_>) -> Result {
        f.debug_tuple("")
            .field(&self.x)
            .field(&self.y)
            .finish()
    }
}

/// # utility
/// Returns element-wise min point: (min(a.x, b.x), min(a.y, b.y)).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// use vox_geometry_rust::point2::min;
/// let vec = Point2F::new(5.0, 1.0);
/// let vec2 = Point2F::new(3.0, 3.0);
/// let min_point = min(&vec, &vec2);
/// assert_eq!(Point2F::new(3.0, 1.0), min_point);
/// ```
pub fn min<T: Float>(a: &Point2<T>, b: &Point2<T>) -> Point2<T> {
    return Point2::new(T::min(a.x, b.x), T::min(a.y, b.y));
}

/// Returns element-wise max point: (max(a.x, b.x), max(a.y, b.y)).
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// use vox_geometry_rust::point2::max;
/// let vec = Point2F::new(5.0, 1.0);
/// let vec2 = Point2F::new(3.0, 3.0);
/// let min_point = max(&vec, &vec2);
/// assert_eq!(Point2F::new(5.0, 3.0), min_point);
/// ```
pub fn max<T: Float>(a: &Point2<T>, b: &Point2<T>) -> Point2<T> {
    return Point2::new(T::max(a.x, b.x), T::max(a.y, b.y));
}

/// Returns element-wise clamped point.
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// use vox_geometry_rust::point2::clamp;
/// let vec = Point2F::new(2.0, 4.0);
/// let low = Point2F::new(3.0, -1.0);
/// let high = Point2F::new(5.0, 2.0);
/// let clamped_vec = clamp(&vec, &low, &high);
/// assert_eq!(Point2F::new(3.0, 2.0), clamped_vec);
/// ```
pub fn clamp<T: Float>(v: &Point2<T>, low: &Point2<T>, high: &Point2<T>) -> Point2<T> {
    return Point2::new(crate::math_utils::clamp(v.x, low.x, high.x),
                       crate::math_utils::clamp(v.y, low.y, high.y));
}

/// Returns element-wise ceiled point.
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// use vox_geometry_rust::point2::ceil;
/// let vec = Point2F::new(2.2, 4.7);
/// let ceil_vec = ceil(&vec);
/// assert_eq!(Point2F::new(3.0, 5.0), ceil_vec);
/// ```
pub fn ceil<T: Float>(a: &Point2<T>) -> Point2<T> {
    return Point2::new((a.x).ceil(), (a.y).ceil());
}

/// Returns element-wise floored point.
/// ```
/// use vox_geometry_rust::point2::Point2F;
/// use vox_geometry_rust::point2::floor;
/// let vec = Point2F::new(2.2, 4.7);
/// let floor_vec = floor(&vec);
/// assert_eq!(Point2F::new(2.0, 4.0), floor_vec);
/// ```
pub fn floor<T: Float>(a: &Point2<T>) -> Point2<T> {
    return Point2::new((a.x).floor(), (a.y).floor());
}