/*
 * // Copyright (c) 2021 Feng Yang
 * //
 * // I am making my contributions/submissions to this project solely in my
 * // personal capacity and am not conveying any rights to any intellectual
 * // property of any third parties.
 */

use std::ops::{Index, IndexMut, AddAssign, SubAssign, MulAssign, DivAssign, Add, Sub, Mul, Div};
use std::fmt::{Debug, Formatter, Result};

///
/// # 2-D size class.
///
/// This class defines simple 2-D size data.
///
/// - tparam T - Type of the element
///
pub struct ISize2 {
    /// X (or the first) component of the size.
    pub x: isize,

    /// Y (or the second) component of the size.
    pub y: isize,
}

/// # Constructors
impl ISize2 {
    /// Constructs default size (0, 0).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec = ISize2::new_default();
    /// assert_eq!(0, vec.x);
    /// assert_eq!(0, vec.y);
    /// ```
    pub fn new_default() -> ISize2 {
        return ISize2 {
            x: 0,
            y: 0,
        };
    }

    /// Constructs size with given parameters \p x_ and \p y_.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec2 = ISize2::new(5, 3);
    /// assert_eq!(5, vec2.x);
    /// assert_eq!(3, vec2.y);
    /// ```
    pub fn new(x_: isize, y_: isize) -> ISize2 {
        return ISize2 {
            x: x_,
            y: y_,
        };
    }

    /// Constructs size with initializer list.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec5 = ISize2::new_lst([7, 6]);
    /// assert_eq!(7, vec5.x);
    /// assert_eq!(6, vec5.y);
    /// ```
    pub fn new_lst(lst: [isize; 2]) -> ISize2 {
        return ISize2 {
            x: lst[0],
            y: lst[1],
        };
    }
}

/// # Basic setters
impl ISize2 {
    /// Set both x and y components to **s**.
    pub fn set_scalar(&mut self, s: isize) {
        self.x = s;
        self.y = s;
    }

    /// Set x and y components with given parameters.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new_default();
    /// vec.set_scalar2(4, 2);
    /// assert_eq!(4, vec.x);
    /// assert_eq!(2, vec.y);
    /// ```
    pub fn set_scalar2(&mut self, x: isize, y: isize) {
        self.x = x;
        self.y = y;
    }

    /// Set x and y components with given initializer list.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new_default();
    /// let lst = [0, 5];
    /// vec.set_lst(lst);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(5, vec.y);
    /// ```
    pub fn set_lst(&mut self, lst: [isize; 2]) {
        self.x = lst[0];
        self.y = lst[1];
    }

    /// Set x and y with other size **pt**.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new_default();
    /// vec.set_self(ISize2::new(9, 8));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(8, vec.y);
    /// ```
    pub fn set_self(&mut self, pt: ISize2) {
        self.x = pt.x;
        self.y = pt.y;
    }

    /// Set both x and y to zero.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.set_zero();
    /// assert_eq!(0, vec.x);
    /// assert_eq!(0, vec.y);
    /// ```
    pub fn set_zero(&mut self) {
        self.x = 0;
        self.y = 0;
    }
}

/// # Binary operations: new instance = this (+) v
impl ISize2 {
    /// Computes self + (v, v).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// assert_eq!(7, vec.x);
    /// assert_eq!(13, vec.y);
    /// ```
    pub fn add_scalar(&self, v: isize) -> ISize2 {
        return ISize2::new(self.x + v, self.y + v);
    }

    /// Computes self + (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(ISize2::new(2, 1));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(14, vec.y);
    /// ```
    pub fn add_vec(&self, v: ISize2) -> ISize2 {
        return ISize2::new(self.x + v.x, self.y + v.y);
    }

    /// Computes self - (v, v).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(ISize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// assert_eq!(1, vec.x);
    /// assert_eq!(6, vec.y);
    /// ```
    pub fn sub_scalar(&self, v: isize) -> ISize2 {
        return ISize2::new(self.x - v, self.y - v);
    }

    /// Computes self - (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(ISize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(ISize2::new(1, 3));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn sub_vec(&self, v: ISize2) -> ISize2 {
        return ISize2::new(self.x - v.x, self.y - v.y);
    }

    /// Computes self * (v, v).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(ISize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(ISize2::new(1, 3));
    /// vec = vec.mul_scalar(2);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(6, vec.y);
    /// ```
    pub fn mul_scalar(&self, v: isize) -> ISize2 {
        return ISize2::new(self.x * v, self.y * v);
    }

    /// Computes self * (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(ISize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(ISize2::new(1, 3));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(ISize2::new(3, 2));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(12, vec.y);
    /// ```
    pub fn mul_vec(&self, v: ISize2) -> ISize2 {
        return ISize2::new(self.x * v.x, self.y * v.y);
    }

    /// Computes self / (v, v).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(ISize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(ISize2::new(1, 3));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(ISize2::new(3, 2));
    /// vec = vec.div_scalar(4);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn div_scalar(&self, v: isize) -> ISize2 {
        return ISize2::new(self.x / v, self.y / v);
    }

    /// Computes self / (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.add_scalar(4);
    /// vec = vec.add_vec(ISize2::new(2, 1));
    /// vec = vec.sub_scalar(8);
    /// vec = vec.sub_vec(ISize2::new(1, 3));
    /// vec = vec.mul_scalar(2);
    /// vec = vec.mul_vec(ISize2::new(3, 2));
    /// vec = vec.div_scalar(4);
    /// vec = vec.div_vec(ISize2::new(3, 1));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn div_vec(&self, v: ISize2) -> ISize2 {
        return ISize2::new(self.x / v.x, self.y / v.y);
    }
}

/// # Binary operations: new instance = v (+) this
impl ISize2 {
    /// Computes (v, v) - self.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.rsub_scalar(9);
    /// assert_eq!(6, vec.x);
    /// assert_eq!(0, vec.y);
    /// ```
    pub fn rsub_scalar(&self, v: isize) -> ISize2 {
        return ISize2::new(v - self.x, v - self.y);
    }

    /// Computes (v.x, v.y) - self.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec = vec.rsub_scalar(9);
    /// vec = vec.rsub_vec(ISize2::new(6, 3));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn rsub_vec(&self, v: ISize2) -> ISize2 {
        return ISize2::new(v.x - self.x, v.y - self.y);
    }

    /// Computes (v, v) / self.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(4, 3);
    /// vec = vec.rdiv_scalar(12);
    /// assert_eq!(3, vec.x);
    /// assert_eq!(vec.y, 4);
    /// ```
    pub fn rdiv_scalar(&self, v: isize) -> ISize2 {
        return ISize2::new(v / self.x, v / self.y);
    }

    /// Computes (v.x, v.y) / self.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(4, 3);
    /// vec = vec.rdiv_scalar(12);
    /// vec = vec.rdiv_vec(ISize2::new(3, 16));
    /// assert_eq!(1, vec.x);
    /// assert_eq!(4, vec.y);
    /// ```
    pub fn rdiv_vec(&self, v: ISize2) -> ISize2 {
        return ISize2::new(v.x / self.x, v.y / self.y);
    }
}

/// # Augmented operations: isizehis (+)= v
impl ISize2 {
    /// Computes self += (v, v).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// assert_eq!(7, vec.x);
    /// assert_eq!(vec.y, 13);
    /// ```
    pub fn iadd_scalar(&mut self, v: isize) {
        self.x = isize::add(self.x, v);
        self.y = isize::add(self.y, v);
    }

    /// Computes self += (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(ISize2::new(2, 1));
    /// assert_eq!(9, vec.x);
    /// assert_eq!(vec.y, 14);
    /// ```
    pub fn iadd_vec(&mut self, v: ISize2) {
        self.x = isize::add(self.x, v.x);
        self.y = isize::add(self.y, v.y);
    }

    /// Computes self -= (v, v).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(ISize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// assert_eq!(1, vec.x);
    /// assert_eq!(6, vec.y);
    /// ```
    pub fn isub_scalar(&mut self, v: isize) {
        self.x = isize::sub(self.x, v);
        self.y = isize::sub(self.y, v);
    }

    /// Computes self -= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(ISize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(ISize2::new(1, 3));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn isub_vec(&mut self, v: ISize2) {
        self.x = isize::sub(self.x, v.x);
        self.y = isize::sub(self.y, v.y);
    }

    /// Computes self *= (v, v).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(ISize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(ISize2::new(1, 3));
    /// vec.imul_scalar(2);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(6, vec.y);
    /// ```
    pub fn imul_scalar(&mut self, v: isize) {
        self.x = isize::mul(self.x, v);
        self.y = isize::mul(self.y, v);
    }

    /// Computes self *= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(ISize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(ISize2::new(1, 3));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(ISize2::new(3, 2));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(12, vec.y);
    /// ```
    pub fn imul_vec(&mut self, v: ISize2) {
        self.x = isize::mul(self.x, v.x);
        self.y = isize::mul(self.y, v.y);
    }

    /// Computes self /= (v, v).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(ISize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(ISize2::new(1, 3));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(ISize2::new(3, 2));
    /// vec.idiv_scalar(4);
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn idiv_scalar(&mut self, v: isize) {
        self.x = isize::div(self.x, v);
        self.y = isize::div(self.y, v);
    }

    /// Computes self /= (v.x, v.y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut vec = ISize2::new(3, 9);
    /// vec.iadd_scalar(4);
    /// vec.iadd_vec(ISize2::new(2, 1));
    /// vec.isub_scalar(8);
    /// vec.isub_vec(ISize2::new(1, 3));
    /// vec.imul_scalar(2);
    /// vec.imul_vec(ISize2::new(3, 2));
    /// vec.idiv_scalar(4);
    /// vec.idiv_vec(ISize2::new(3, 1));
    /// assert_eq!(0, vec.x);
    /// assert_eq!(3, vec.y);
    /// ```
    pub fn idiv_vec(&mut self, v: ISize2) {
        self.x = isize::div(self.x, v.x);
        self.y = isize::div(self.y, v.y);
    }
}

/// # Basic getters
impl ISize2 {
    /// Returns const reference to the **i** -th element of the size.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec = ISize2::new(8, 9);
    /// assert_eq!(*vec.at(0), 8);
    /// assert_eq!(*vec.at(1), 9);
    /// ```
    pub fn at(&self, i: isize) -> &isize {
        match i {
            0 => return &self.x,
            1 => return &self.y,
            _ => { panic!() }
        }
    }

    /// Returns reference to the **i** -th element of the size.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let mut a = ISize2::new_default();
    /// *a.at_mut(0) =  10;
    /// *a.at_mut(1) =  20;
    /// assert_eq!(*a.at(0), 10);
    /// assert_eq!(*a.at(1), 20);
    /// ```
    pub fn at_mut(&mut self, i: isize) -> &mut isize {
        match i {
            0 => return &mut self.x,
            1 => return &mut self.y,
            _ => { panic!() }
        }
    }

    /// Returns the sum of all the components (i.e. x + y).
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec = ISize2::new(3, 7);
    /// let sum = vec.sum();
    /// assert_eq!(sum, 10);
    /// ```
    pub fn sum(&self) -> isize {
        return self.x + self.y;
    }

    /// Returns the minimum value among x and y.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec = ISize2::new(3, 7);
    /// let min = vec.min();
    /// assert_eq!(min, 3);
    /// ```
    pub fn min(&self) -> isize {
        return self.x.min(self.y);
    }

    /// Returns the maximum value among x and y.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec = ISize2::new(3, 7);
    /// let max = vec.max();
    /// assert_eq!(max, 7);
    /// ```
    pub fn max(&self) -> isize {
        return self.x.max(self.y);
    }

    /// Returns the index of the dominant axis.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec = ISize2::new(3, 7);
    /// let dominant_axis = vec.dominant_axis();
    /// assert_eq!(dominant_axis, 1);
    /// ```
    pub fn dominant_axis(&self) -> isize {
        match self.x > self.y {
            true => 0,
            false => 1
        }
    }

    /// Returns the index of the subminant axis.
    /// ```
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec = ISize2::new(3, 7);
    /// let subminant_axis = vec.subminant_axis();
    /// assert_eq!(subminant_axis, 0);
    /// ```
    pub fn subminant_axis(&self) -> isize {
        match self.x < self.y {
            true => 0,
            false => 1
        }
    }

    /// Returns true if **other** is the same as self size.
    pub fn is_equal(&self, other: &ISize2) -> bool {
        return self.x == other.x && self.y == other.y;
    }
}

/// Copy constructor.
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec5 = ISize2::new_lst([7, 6]);
/// let mut vec6 = vec5.clone();
/// vec6.x = 10;
/// assert_eq!(10, vec6.x);
/// assert_eq!(7, vec5.x);
/// ```
impl Clone for ISize2 {
    fn clone(&self) -> Self {
        return ISize2 {
            x: self.x,
            y: self.y,
        };
    }
}

///
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let vec = ISize2::new(5, 1);
/// let mut vec2 = ISize2::new(3, 3);
/// vec2 = vec;
/// assert_eq!(5, vec2.x);
/// assert_eq!(vec2.y, 1);
/// ```
impl Copy for ISize2 {}

/// # Operators
/// Returns const reference to the **i** -th element of the size.
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let vec = ISize2::new(8, 9);
/// assert_eq!(vec[0], 8);
/// assert_eq!(vec[1], 9);
/// ```
impl Index<isize> for ISize2 {
    type Output = isize;
    fn index(&self, index: isize) -> &Self::Output {
        return self.at(index);
    }
}

/// Returns reference to the **i** -th element of the size.
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(8, 9);
/// vec[0] = 7;
/// vec[1] = 6;
/// assert_eq!(7, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl IndexMut<isize> for ISize2 {
    fn index_mut(&mut self, index: isize) -> &mut Self::Output {
        return self.at_mut(index);
    }
}

/// Computes self += (v, v)
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec += 4;
/// assert_eq!(7, vec.x);
/// assert_eq!(vec.y, 13);
/// ```
impl AddAssign<isize> for ISize2 {
    fn add_assign(&mut self, rhs: isize) {
        self.iadd_scalar(rhs);
    }
}

/// Computes self += (v.x, v.y)
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec += 4;
/// vec += ISize2::new(2, 1);
/// assert_eq!(9, vec.x);
/// assert_eq!(vec.y, 14);
/// ```
impl AddAssign for ISize2 {
    fn add_assign(&mut self, rhs: Self) {
        self.iadd_vec(rhs);
    }
}

/// Computes self -= (v, v)
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec += 4;
/// vec += ISize2::new(2, 1);
/// vec -= 8;
/// assert_eq!(1, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl SubAssign<isize> for ISize2 {
    fn sub_assign(&mut self, rhs: isize) {
        self.isub_scalar(rhs);
    }
}

/// Computes self -= (v.x, v.y)
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec += 4;
/// vec += ISize2::new(2, 1);
/// vec -= 8;
/// vec -= ISize2::new(1, 3);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl SubAssign for ISize2 {
    fn sub_assign(&mut self, rhs: Self) {
        self.isub_vec(rhs);
    }
}

/// Computes self *= (v, v)
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec += 4;
/// vec += ISize2::new(2, 1);
/// vec -= 8;
/// vec -= ISize2::new(1, 3);
/// vec *= 2;
/// assert_eq!(0, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl MulAssign<isize> for ISize2 {
    fn mul_assign(&mut self, rhs: isize) {
        self.imul_scalar(rhs);
    }
}

/// Computes self *= (v.x, v.y)
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec += 4;
/// vec += ISize2::new(2, 1);
/// vec -= 8;
/// vec -= ISize2::new(1, 3);
/// vec *= 2;
/// vec *= ISize2::new(3, 2);
/// assert_eq!(0, vec.x);
/// assert_eq!(12, vec.y);
/// ```
impl MulAssign for ISize2 {
    fn mul_assign(&mut self, rhs: Self) {
        self.imul_vec(rhs);
    }
}

/// Computes self /= (v, v)
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec += 4;
/// vec += ISize2::new(2, 1);
/// vec -= 8;
/// vec -= ISize2::new(1, 3);
/// vec *= 2;
/// vec *= ISize2::new(3, 2);
/// vec /= 4;
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl DivAssign<isize> for ISize2 {
    fn div_assign(&mut self, rhs: isize) {
        self.idiv_scalar(rhs);
    }
}

/// Computes self /= (v.x, v.y)
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec += 4;
/// vec += ISize2::new(2, 1);
/// vec -= 8;
/// vec -= ISize2::new(1, 3);
/// vec *= 2;
/// vec *= ISize2::new(3, 2);
/// vec /= 4;
/// vec /= ISize2::new(3, 1);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl DivAssign for ISize2 {
    fn div_assign(&mut self, rhs: Self) {
        self.idiv_vec(rhs);
    }
}

/// Returns true if **other** is the same as self size.
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new_default();
/// let vec2 = ISize2::new(3, 7);
/// let vec3 = ISize2::new(3, 5);
/// let vec4 = ISize2::new(5, 1);
/// vec = vec2;
/// assert_eq!(vec == vec2, true);
/// assert_eq!(vec == vec3, false);
/// assert_eq!(vec != vec2, false);
/// assert_eq!(vec != vec3, true);
/// assert_eq!(vec != vec4, true);
/// ```
impl PartialEq for ISize2 {
    fn eq(&self, other: &Self) -> bool {
        return self.is_equal(other);
    }
}

impl Eq for ISize2 {}

/// Computes (a, a) + (b.x, b.y).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec = vec + 4;
/// assert_eq!(7, vec.x);
/// assert_eq!(vec.y, 13);
/// ```
impl Add<isize> for ISize2 {
    type Output = ISize2;
    fn add(self, rhs: isize) -> Self::Output {
        return self.add_scalar(rhs);
    }
}

/// Computes (a.x, a.y) + (b.x, b.y).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + ISize2::new(2, 1);
/// assert_eq!(9, vec.x);
/// assert_eq!(vec.y, 14);
/// ```
impl Add for ISize2 {
    type Output = ISize2;
    fn add(self, rhs: Self) -> Self::Output {
        return self.add_vec(rhs);
    }
}

/// Computes (a.x, a.y) - (b, b).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + ISize2::new(2, 1);
/// vec = vec - 8;
/// assert_eq!(1, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl Sub<isize> for ISize2 {
    type Output = ISize2;
    fn sub(self, rhs: isize) -> Self::Output {
        return self.sub_scalar(rhs);
    }
}

/// Computes (a.x, a.y) - (b.x, b.y).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + ISize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - ISize2::new(1, 3);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl Sub for ISize2 {
    type Output = ISize2;
    fn sub(self, rhs: Self) -> Self::Output {
        return self.sub_vec(rhs);
    }
}

/// Computes (a.x, a.y) * (b, b).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + ISize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - ISize2::new(1, 3);
/// vec = vec * 2;
/// assert_eq!(0, vec.x);
/// assert_eq!(6, vec.y);
/// ```
impl Mul<isize> for ISize2 {
    type Output = ISize2;
    fn mul(self, rhs: isize) -> Self::Output {
        return self.mul_scalar(rhs);
    }
}

/// Computes (a.x, a.y) * (b.x, b.y).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + ISize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - ISize2::new(1, 3);
/// vec = vec * 2;
/// vec = vec * ISize2::new(3, 2);
/// assert_eq!(0, vec.x);
/// assert_eq!(12, vec.y);
/// ```
impl Mul for ISize2 {
    type Output = ISize2;
    fn mul(self, rhs: Self) -> Self::Output {
        return self.mul_vec(rhs);
    }
}

/// Computes (a.x, a.y) / (b, b).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + ISize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - ISize2::new(1, 3);
/// vec = vec * 2;
/// vec = vec * ISize2::new(3, 2);
/// vec = vec / 4;
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl Div<isize> for ISize2 {
    type Output = ISize2;
    fn div(self, rhs: isize) -> Self::Output {
        return self.div_scalar(rhs);
    }
}

/// Computes (a.x, a.y) / (b.x, b.y).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// let mut vec = ISize2::new(3, 9);
/// vec = vec + 4;
/// vec = vec + ISize2::new(2, 1);
/// vec = vec - 8;
/// vec = vec - ISize2::new(1, 3);
/// vec = vec * 2;
/// vec = vec * ISize2::new(3, 2);
/// vec = vec / 4;
/// vec = vec / ISize2::new(3, 1);
/// assert_eq!(0, vec.x);
/// assert_eq!(3, vec.y);
/// ```
impl Div for ISize2 {
    type Output = ISize2;
    fn div(self, rhs: Self) -> Self::Output {
        return self.div_vec(rhs);
    }
}

impl Debug for ISize2 {
    /// # Example
    /// ```
    ///
    /// use vox_geometry_rust::isize2::ISize2;
    /// let vec = ISize2::new(10, 20);
    /// assert_eq!(format!("{:?}", vec), "(10, 20)");
    ///
    /// assert_eq!(format!("{:#?}", vec), "(
    ///     10,
    ///     20,
    /// )");
    /// ```
    fn fmt(&self, f: &mut Formatter<'_>) -> Result {
        f.debug_tuple("")
            .field(&self.x)
            .field(&self.y)
            .finish()
    }
}

/// # utility
/// Returns element-wise min size: (min(a.x, b.x), min(a.y, b.y)).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// use vox_geometry_rust::isize2::min;
/// let vec = ISize2::new(5, 1);
/// let vec2 = ISize2::new(3, 3);
/// let min_size = min(&vec, &vec2);
/// assert_eq!(ISize2::new(3, 1), min_size);
/// ```
pub fn min(a: &ISize2, b: &ISize2) -> ISize2 {
    return ISize2::new(isize::min(a.x, b.x), isize::min(a.y, b.y));
}

/// Returns element-wise max size: (max(a.x, b.x), max(a.y, b.y)).
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// use vox_geometry_rust::isize2::max;
/// let vec = ISize2::new(5, 1);
/// let vec2 = ISize2::new(3, 3);
/// let min_size = max(&vec, &vec2);
/// assert_eq!(ISize2::new(5, 3), min_size);
/// ```
pub fn max(a: &ISize2, b: &ISize2) -> ISize2 {
    return ISize2::new(isize::max(a.x, b.x), isize::max(a.y, b.y));
}

/// Returns element-wise clamped size.
/// ```
/// use vox_geometry_rust::isize2::ISize2;
/// use vox_geometry_rust::isize2::clamp;
/// let vec = ISize2::new(2, 4);
/// let low = ISize2::new(3, 1);
/// let high = ISize2::new(5, 2);
/// let clamped_vec = clamp(&vec, &low, &high);
/// assert_eq!(ISize2::new(3, 2), clamped_vec);
/// ```
pub fn clamp(v: &ISize2, low: &ISize2, high: &ISize2) -> ISize2 {
    return ISize2::new(isize::clamp(v.x, low.x, high.x),
                       isize::clamp(v.y, low.y, high.y));
}