/*
 * // Copyright (c) 2021 Feng Yang
 * //
 * // I am making my contributions/submissions to this project solely in my
 * // personal capacity and am not conveying any rights to any intellectual
 * // property of any third parties.
 */

use crate::surface3::*;
use crate::bvh3::Bvh3;
use crate::transform3::Transform3;
use crate::vector3::Vector3D;
use crate::bounding_box3::BoundingBox3D;
use crate::ray3::Ray3D;
use crate::nearest_neighbor_query_engine3::NearestNeighborQueryEngine3;
use crate::intersection_query_engine3::IntersectionQueryEngine3;
use std::cell::RefCell;
use std::rc::Rc;

///
/// # 3-D surface set.
///
/// This class represents 3-D surface set which extends Surface3 by overriding
/// surface-related queries. This is class can hold a collection of other
/// surface instances.
///
pub struct SurfaceSet3 {
    _surfaces: Vec<Surface3Ptr>,
    _unbounded_surfaces: Vec<Surface3Ptr>,
    _bvh: RefCell<Bvh3<Surface3Ptr>>,
    _bvh_invalidated: RefCell<bool>,

    /// Local-to-world transform.
    pub transform: Transform3,

    /// Flips normal.
    pub is_normal_flipped: bool,
}

impl SurfaceSet3 {
    /// Constructs an empty surface set.
    /// ```
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// let sset1 = SurfaceSet3::new_default();
    /// assert_eq!(0, sset1.number_of_surfaces());
    /// ```
    pub fn new_default() -> SurfaceSet3 {
        return SurfaceSet3 {
            _surfaces: vec![],
            _unbounded_surfaces: vec![],
            _bvh: RefCell::new(Bvh3::new()),
            _bvh_invalidated: RefCell::new(true),
            transform: Transform3::new_default(),
            is_normal_flipped: false,
        };
    }

    /// Constructs with a list of other surfaces.
    /// ```
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::vector3::Vector3D;
    /// use vox_geometry_rust::transform3::Transform3;
    /// use vox_geometry_rust::quaternion::QuaternionD;
    /// let sph1 = Sphere3::builder().with_radius(1.0).with_center(Vector3D::new_default()).make_shared();
    /// let sph2 = Sphere3::builder().with_radius(0.5).with_center(Vector3D::new(0.0, 3.0, 2.0)).make_shared();
    /// let sph3 = Sphere3::builder().with_radius(0.25).with_center(Vector3D::new(-2.0, 0.0, 0.0)).make_shared();
    /// let sset3 = SurfaceSet3::new(vec![sph1, sph2, sph3], Some(Transform3::new_default()), Some(false));
    ///
    /// assert_eq!(3, sset3.number_of_surfaces());
    /// assert_eq!(Vector3D::new_default(), sset3.transform.translation());
    /// assert_eq!(QuaternionD::new_default(), sset3.transform.orientation());
    /// ```
    pub fn new(others: Vec<Surface3Ptr>,
               transform: Option<Transform3>,
               is_normal_flipped: Option<bool>) -> SurfaceSet3 {
        let mut unbounded_surfaces: Vec<Surface3Ptr> = vec![];
        for surface in &others {
            if !surface.borrow().is_bounded() {
                unbounded_surfaces.push(surface.clone());
            }
        }

        return SurfaceSet3 {
            _surfaces: others.clone(),
            _unbounded_surfaces: unbounded_surfaces,
            _bvh: RefCell::new(Bvh3::new()),
            _bvh_invalidated: RefCell::new(true),
            transform: transform.unwrap_or(Transform3::new_default()),
            is_normal_flipped: is_normal_flipped.unwrap_or(false),
        };
    }

    /// Returns builder fox SurfaceSet3.
    pub fn builder() -> Builder {
        return Builder::new();
    }

    /// Returns the number of surfaces.
    pub fn number_of_surfaces(&self) -> usize {
        return self._surfaces.len();
    }

    /// Returns the i-th surface.
    pub fn surface_at(&self, i: usize) -> Surface3Ptr {
        return self._surfaces[i].clone();
    }

    /// Adds a surface instance.
    /// ```
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::vector3::Vector3D;
    /// let mut sset1 = SurfaceSet3::new_default();
    /// assert_eq!(0, sset1.number_of_surfaces());
    ///
    /// let sph1 = Sphere3::builder().with_radius(1.0).with_center(Vector3D::new_default()).make_shared();
    /// let sph2 = Sphere3::builder().with_radius(0.5).with_center(Vector3D::new(0.0, 3.0, 2.0)).make_shared();
    /// let sph3 = Sphere3::builder().with_radius(0.25).with_center(Vector3D::new(-2.0, 0.0, 0.0)).make_shared();
    ///
    /// sset1.add_surface(sph1);
    /// sset1.add_surface(sph2);
    /// sset1.add_surface(sph3);
    ///
    /// assert_eq!(3, sset1.number_of_surfaces());
    /// ```
    pub fn add_surface(&mut self, surface: Surface3Ptr) {
        self._surfaces.push(surface.clone());
        if !surface.borrow().is_bounded() {
            self._unbounded_surfaces.push(surface.clone());
        }
        self.invalidate_bvh();
    }

    /// must call it manually in Rust!
    pub fn build_bvh(&self) {
        if *self._bvh_invalidated.borrow() {
            let mut surfs: Vec<Surface3Ptr> = Vec::new();
            let mut bounds: Vec<BoundingBox3D> = Vec::new();
            for i in 0..self._surfaces.len() {
                if self._surfaces[i].borrow().is_bounded() {
                    surfs.push(self._surfaces[i].clone());
                    bounds.push(self._surfaces[i].borrow().bounding_box());
                }
            }
            self._bvh.borrow_mut().build(&surfs, &bounds);
            self._bvh_invalidated.replace(false);
        }
    }

    fn invalidate_bvh(&self) {
        self._bvh_invalidated.replace(true);
    }
}

impl Surface3 for SurfaceSet3 {
    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::vector3::Vector3D;
    /// use vox_geometry_rust::unit_tests_utils::*;
    /// let mut sset1 = SurfaceSet3::new_default();
    /// // Test empty set
    /// let empty_point = sset1.closest_point(&Vector3D::new(1.0, 2.0, 3.0));
    /// assert_eq!(f64::MAX, empty_point.x);
    /// assert_eq!(f64::MAX, empty_point.y);
    /// assert_eq!(f64::MAX, empty_point.z);
    ///
    /// let num_samples = get_number_of_sample_points3();
    ///
    /// // Use first half of the samples as the centers of the spheres
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset1.add_surface(sph);
    /// }
    ///
    /// let brute_force_search = |pt:&Vector3D| {
    ///     let mut min_dist3 = f64::MAX;
    ///     let mut result = Vector3D::new_default();
    ///     for i in 0..num_samples/2 {
    ///         let local_result = sset1.surface_at(i).borrow().closest_point(pt);
    ///         let local_dist3 = pt.distance_squared_to(local_result);
    ///         if local_dist3 < min_dist3 {
    ///             min_dist3 = local_dist3;
    ///             result = local_result;
    ///         }
    ///     }
    ///     return result;
    /// };
    ///
    /// // Use second half of the samples as the query points
    /// for i in num_samples/2..num_samples {
    ///     let actual = sset1.closest_point(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     let expected = brute_force_search(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     assert_eq!(expected, actual);
    /// }
    ///
    /// // Now with translation instead of center
    /// let mut sset3 = SurfaceSet3::new_default();
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_default())
    ///                        .with_translation(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset3.add_surface(sph);
    /// }
    ///
    /// for i in num_samples/2..num_samples {
    ///     let actual = sset3.closest_point(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     let expected = brute_force_search(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     assert_eq!(expected, actual);
    /// }
    /// ```
    fn closest_point_local(&self, other_point: &Vector3D) -> Vector3D {
        self.build_bvh();

        let mut distance_func = |surface: &Surface3Ptr, pt: &Vector3D| {
            return surface.borrow().closest_distance(pt);
        };

        let mut result = Vector3D::new(f64::MAX, f64::MAX, f64::MAX);
        let query_result = self._bvh.borrow().nearest(other_point, &mut distance_func);
        match query_result.item {
            None => {}
            Some(ptr) => {
                result = ptr.borrow().closest_point(other_point);
            }
        }

        let mut min_dist = query_result.distance;
        for surface in &self._unbounded_surfaces {
            let pt = surface.borrow().closest_point(other_point);
            let dist = pt.distance_to(*other_point);
            if dist < min_dist {
                min_dist = dist;
                result = surface.borrow().closest_point(other_point);
            }
        }

        return result;
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    /// use vox_geometry_rust::bounding_box3::BoundingBox3D;
    /// use vox_geometry_rust::unit_tests_utils::*;
    /// let mut sset1 = SurfaceSet3::new_default();
    ///
    /// let num_samples = get_number_of_sample_points3();
    ///
    ///     // Use first half of the samples as the centers of the spheres
    /// let mut answer = BoundingBox3D::new_default();
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset1.add_surface(sph.clone());
    ///
    ///     answer.merge_box(&sph.borrow().bounding_box());
    /// }
    /// assert_eq!(answer.lower_corner.is_similar(&sset1.bounding_box().lower_corner, Some(1e-9)), true);
    /// assert_eq!(answer.upper_corner.is_similar(&sset1.bounding_box().upper_corner, Some(1e-9)), true);
    ///
    /// // Now with translation instead of center
    /// let mut sset3 = SurfaceSet3::new_default();
    /// let mut debug = BoundingBox3D::new_default();
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_default())
    ///                        .with_translation(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset3.add_surface(sph.clone());
    ///
    ///     debug.merge_box(&sph.borrow().bounding_box());
    /// }
    /// assert_eq!(answer.lower_corner.is_similar(&debug.lower_corner, Some(1e-9)), true);
    /// assert_eq!(answer.upper_corner.is_similar(&debug.upper_corner, Some(1e-9)), true);
    ///
    /// assert_eq!(answer.lower_corner.is_similar(&sset3.bounding_box().lower_corner, Some(1e-9)), true);
    /// assert_eq!(answer.upper_corner.is_similar(&sset3.bounding_box().upper_corner, Some(1e-9)), true);
    /// ```
    fn bounding_box_local(&self) -> BoundingBox3D {
        self.build_bvh();

        return self._bvh.borrow().bounding_box();
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    /// use vox_geometry_rust::ray3::Ray3D;
    /// use vox_geometry_rust::unit_tests_utils::*;
    /// let mut sset1 = SurfaceSet3::new_default();
    ///
    /// let num_samples = get_number_of_sample_points3();
    ///
    /// // Use first half of the samples as the centers of the spheres
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset1.add_surface(sph);
    /// }
    /// let brute_force_test = |ray:&Ray3D| {
    ///     let mut result = SurfaceRayIntersection3::new();
    ///     for i in 0..num_samples/2 {
    ///         let local_result = sset1.surface_at(i).borrow().closest_intersection(ray);
    ///         if local_result.distance < result.distance {
    ///             result = local_result;
    ///         }
    ///     }
    ///     return result;
    /// };
    ///
    /// // Use second half of the samples as the query points
    /// for i in num_samples/2..num_samples {
    ///     let ray = Ray3D::new(Vector3D::new_lst(get_sample_points3()[i]), Vector3D::new_lst(get_sample_dirs3()[i]));
    ///     let actual = sset1.closest_intersection(&ray);
    ///     let expected = brute_force_test(&ray);
    ///     assert_eq!(expected.distance, actual.distance);
    ///     assert_eq!(expected.point, actual.point);
    ///     assert_eq!(expected.normal, actual.normal);
    ///     assert_eq!(expected.is_intersecting, actual.is_intersecting);
    /// }
    ///
    /// // Now with translation instead of center
    /// let mut sset3 = SurfaceSet3::new_default();
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_default())
    ///                        .with_translation(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///         sset3.add_surface(sph);
    /// }
    /// for i in num_samples/2..num_samples {
    ///     let ray = Ray3D::new(Vector3D::new_lst(get_sample_points3()[i]), Vector3D::new_lst(get_sample_dirs3()[i]));
    ///     let actual = sset3.closest_intersection(&ray);
    ///     let expected = brute_force_test(&ray);
    ///     assert_eq!(expected.distance, actual.distance);
    ///     assert_eq!(expected.point, actual.point);
    ///     assert_eq!(expected.normal, actual.normal);
    ///     assert_eq!(expected.is_intersecting, actual.is_intersecting);
    /// }
    /// ```
    fn closest_intersection_local(&self, ray: &Ray3D) -> SurfaceRayIntersection3 {
        self.build_bvh();

        let mut test_func = |surface: &Surface3Ptr, ray: &Ray3D| {
            let result = surface.borrow().closest_intersection(ray);
            return result.distance;
        };

        let query_result = self._bvh.borrow().closest_intersection(ray, &mut test_func);
        let mut result = SurfaceRayIntersection3::new();
        result.distance = query_result.distance;
        match query_result.item {
            None => {
                result.is_intersecting = false;
            }
            Some(ptr) => {
                result.is_intersecting = true;
                result.point = ray.point_at(query_result.distance);
                result.normal = ptr.borrow().closest_normal(&result.point);
            }
        }

        for surface in &self._unbounded_surfaces {
            let local_result = surface.borrow().closest_intersection(ray);
            if local_result.distance < result.distance {
                result = local_result;
            }
        }

        return result;
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    /// use vox_geometry_rust::unit_tests_utils::*;
    /// let mut sset1 = SurfaceSet3::new_default();
    ///
    /// let num_samples = get_number_of_sample_points3();
    ///
    /// // Use first half of the samples as the centers of the spheres
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset1.add_surface(sph);
    /// }
    ///
    /// let brute_force_search = |pt:&Vector3D| {
    ///     let mut min_dist3 = f64::MAX;
    ///     let mut result = Vector3D::new_default();
    ///     for i in 0..num_samples/2 {
    ///         let local_result = sset1.surface_at(i).borrow().closest_normal(pt);
    ///         let closest_pt = sset1.surface_at(i).borrow().closest_point(pt);
    ///         let local_dist3 = pt.distance_squared_to(closest_pt);
    ///         if local_dist3 < min_dist3 {
    ///             min_dist3 = local_dist3;
    ///             result = local_result;
    ///         }
    ///     }
    ///     return result;
    /// };
    ///
    /// // Use second half of the samples as the query points
    /// for i in num_samples/2..num_samples {
    ///     let actual = sset1.closest_normal(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     let expected = brute_force_search(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     assert_eq!(expected, actual);
    /// }
    ///
    /// // Now with translation instead of center
    /// let mut sset3 = SurfaceSet3::new_default();
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_default())
    ///                        .with_translation(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset3.add_surface(sph);
    /// }
    ///
    /// for i in num_samples/2..num_samples {
    ///     let actual = sset3.closest_normal(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     let expected = brute_force_search(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     assert_eq!(expected, actual);
    /// }
    /// ```
    fn closest_normal_local(&self, other_point: &Vector3D) -> Vector3D {
        self.build_bvh();

        let mut distance_func = |surface: &Surface3Ptr, pt: &Vector3D| {
            return surface.borrow().closest_distance(pt);
        };

        let mut result = Vector3D::new(1.0, 0.0, 0.0);
        let query_result = self._bvh.borrow().nearest(other_point, &mut distance_func);
        match query_result.item {
            None => {}
            Some(ptr) => {
                result = ptr.borrow().closest_normal(other_point);
            }
        }

        let mut min_dist = query_result.distance;
        for surface in &self._unbounded_surfaces {
            let pt = surface.borrow().closest_point(other_point);
            let dist = pt.distance_to(*other_point);
            if dist < min_dist {
                min_dist = dist;
                result = surface.borrow().closest_normal(other_point);
            }
        }

        return result;
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    /// use vox_geometry_rust::ray3::Ray3D;
    /// use vox_geometry_rust::unit_tests_utils::*;
    /// let mut sset1 = SurfaceSet3::new_default();
    ///
    /// let num_samples = get_number_of_sample_points3();
    ///
    ///  // Use first half of the samples as the centers of the spheres
    ///  for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset1.add_surface(sph);
    /// }
    /// let brute_force_test = |ray:&Ray3D| {
    ///     for i in 0..num_samples/2 {
    ///         if sset1.surface_at(i).borrow().intersects(ray) {
    ///             return true;
    ///         }
    ///     }
    ///     return false;
    /// };
    ///
    /// // Use second half of the samples as the query points
    /// for i in num_samples/2..num_samples {
    ///     let ray = Ray3D::new(Vector3D::new_lst(get_sample_points3()[i]), Vector3D::new_lst(get_sample_dirs3()[i]));
    ///     let actual = sset1.intersects(&ray);
    ///     let expected = brute_force_test(&ray);
    ///     assert_eq!(expected, actual);
    /// }
    ///
    /// // Now with translation instead of center
    /// let mut sset3 = SurfaceSet3::new_default();
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_default())
    ///                        .with_translation(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset3.add_surface(sph);
    /// }
    /// for i in num_samples/2..num_samples {
    ///     let ray = Ray3D::new(Vector3D::new_lst(get_sample_points3()[i]), Vector3D::new_lst(get_sample_dirs3()[i]));
    ///     let actual = sset3.intersects(&ray);
    ///     let expected = brute_force_test(&ray);
    ///     assert_eq!(expected, actual);
    /// }
    /// ```
    fn intersects_local(&self, ray_local: &Ray3D) -> bool {
        self.build_bvh();

        let mut test_func = |surface: &Surface3Ptr, ray: &Ray3D| {
            return surface.borrow().intersects(ray);
        };

        let mut result = self._bvh.borrow().intersects_ray(ray_local, &mut test_func);
        for surface in &self._unbounded_surfaces {
            result |= surface.borrow().intersects(ray_local);
        }

        return result;
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    /// use vox_geometry_rust::unit_tests_utils::*;
    /// let mut sset1 = SurfaceSet3::new_default();
    ///
    /// let num_samples = get_number_of_sample_points3();
    ///
    /// // Use first half of the samples as the centers of the spheres
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///     sset1.add_surface(sph);
    /// }
    /// let brute_force_search = |pt:&Vector3D| {
    ///     let mut min_dist = f64::MAX;
    ///     for i in 0..num_samples/2 {
    ///         let local_dist = sset1.surface_at(i).borrow().closest_distance(pt);
    ///         if local_dist < min_dist {
    ///             min_dist = local_dist;
    ///         }
    ///     }
    ///     return min_dist;
    /// };
    ///
    /// // Use second half of the samples as the query points
    /// for i in num_samples/2..num_samples {
    ///     let actual = sset1.closest_distance(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     let expected = brute_force_search(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     assert_eq!(expected, actual);
    /// }
    ///
    /// // Now with translation instead of center
    /// let mut sset3 = SurfaceSet3::new_default();
    /// for i in 0..num_samples/2 {
    ///     let sph = Sphere3::builder()
    ///                        .with_radius(0.01)
    ///                        .with_center(Vector3D::new_default())
    ///                        .with_translation(Vector3D::new_lst(get_sample_points3()[i]))
    ///                        .make_shared();
    ///         sset3.add_surface(sph);
    /// }
    /// for i in num_samples/2..num_samples {
    ///     let actual = sset3.closest_distance(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     let expected = brute_force_search(&Vector3D::new_lst(get_sample_points3()[i]));
    ///     assert_eq!(expected, actual);
    /// }
    /// ```
    fn closest_distance_local(&self, other_point: &Vector3D) -> f64 {
        self.build_bvh();

        let mut distance_func = |surface: &Surface3Ptr, pt: &Vector3D| {
            return surface.borrow().closest_distance(pt);
        };

        let query_result = self._bvh.borrow().nearest(other_point, &mut distance_func);

        let mut min_dist = query_result.distance;
        for surface in &self._unbounded_surfaces {
            let pt = surface.borrow().closest_point(other_point);
            let dist = pt.distance_to(*other_point);
            if dist < min_dist {
                min_dist = dist;
            }
        }

        return min_dist;
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::plane3::*;
    /// use vox_geometry_rust::bounding_box3::*;
    /// use vox_geometry_rust::transform3::*;
    /// use vox_geometry_rust::quaternion::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    ///
    /// let domain = BoundingBox3D::new(Vector3D::new_default(), Vector3D::new(1.0, 2.0, 1.0));
    /// let offset = Vector3D::new(1.0, 2.0, 3.0);
    ///
    /// let plane = Plane3::builder()
    ///                      .with_normal(Vector3D::new(0.0, 1.0, 0.0))
    ///                      .with_point(Vector3D::new(0.0, 0.25 * domain.height(), 0.0))
    ///                      .make_shared();
    ///
    /// let sphere = Sphere3::builder()
    ///                       .with_center(domain.mid_point())
    ///                       .with_radius(0.15 * domain.width())
    ///                       .make_shared();
    ///
    /// let surface_set = SurfaceSet3::builder()
    ///                           .with_surfaces(vec![plane, sphere])
    ///                           .with_transform(Transform3::new(offset, QuaternionD::new_default()))
    ///                           .make_shared();
    ///
    /// assert_eq!(surface_set.borrow().is_inside(&(Vector3D::new(0.5, 0.25, 0.5) + offset)), true);
    /// assert_eq!(surface_set.borrow().is_inside(&(Vector3D::new(0.5, 1.0, 0.5) + offset)), true);
    /// assert_eq!(surface_set.borrow().is_inside(&(Vector3D::new(0.5, 1.5, 0.5) + offset)), false);
    /// ```
    fn is_inside_local(&self, other_point_local: &Vector3D) -> bool {
        for surface in &self._surfaces {
            if surface.borrow().is_inside(other_point_local) {
                return true;
            }
        }

        return false;
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::plane3::*;
    /// use vox_geometry_rust::bounding_box3::*;
    /// use vox_geometry_rust::transform3::*;
    /// use vox_geometry_rust::quaternion::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    ///
    /// let sphere =
    ///         Sphere3::builder().with_center(Vector3D::new(-1.0, 1.0, 2.0)).with_radius(0.5).make_shared();
    ///
    /// let surface_set = SurfaceSet3::builder()
    ///                           .with_surfaces(vec![sphere.clone()])
    ///                           .with_transform(Transform3::new(Vector3D::new(1.0, 2.0, -1.0), QuaternionD::new_default()))
    ///                           .make_shared();
    ///
    /// let bbox1 = surface_set.borrow().bounding_box();
    /// assert_eq!(BoundingBox3D::new(Vector3D::new(-0.5, 2.5, 0.5), Vector3D::new(0.5, 3.5, 1.5)).upper_corner, bbox1.upper_corner);
    /// assert_eq!(BoundingBox3D::new(Vector3D::new(-0.5, 2.5, 0.5), Vector3D::new(0.5, 3.5, 1.5)).lower_corner, bbox1.lower_corner);
    ///
    /// surface_set.borrow_mut().transform = Transform3::new(Vector3D::new(3.0, -4.0, 7.0), QuaternionD::new_default());
    /// surface_set.borrow_mut().update_query_engine();
    /// let bbox3 = surface_set.borrow().bounding_box();
    /// assert_eq!(BoundingBox3D::new(Vector3D::new(1.5, -3.5, 8.5), Vector3D::new(2.5, -2.5, 9.5)).upper_corner, bbox3.upper_corner);
    /// assert_eq!(BoundingBox3D::new(Vector3D::new(1.5, -3.5, 8.5), Vector3D::new(2.5, -2.5, 9.5)).lower_corner, bbox3.lower_corner);
    ///
    /// sphere.borrow_mut().transform = Transform3::new(Vector3D::new(-6.0, 9.0, 2.0), QuaternionD::new_default());
    /// surface_set.borrow_mut().update_query_engine();
    /// let bbox3 = surface_set.borrow().bounding_box();
    /// assert_eq!(BoundingBox3D::new(Vector3D::new(-4.5, 5.5, 10.5), Vector3D::new(-3.5, 6.5, 11.5)).upper_corner, bbox3.upper_corner);
    /// assert_eq!(BoundingBox3D::new(Vector3D::new(-4.5, 5.5, 10.5), Vector3D::new(-3.5, 6.5, 11.5)).lower_corner, bbox3.lower_corner);
    ///
    /// // Plane is unbounded. Total bbox should ignore it.
    /// let plane = Plane3::builder().with_normal(Vector3D::new(1.0, 0.0, 0.0)).make_shared();
    /// surface_set.borrow_mut().add_surface(plane);
    /// surface_set.borrow_mut().update_query_engine();
    /// let bbox4 = surface_set.borrow().bounding_box();
    /// assert_eq!(BoundingBox3D::new(Vector3D::new(-4.5, 5.5, 10.5), Vector3D::new(-3.5, 6.5, 11.5)).upper_corner, bbox4.upper_corner);
    /// assert_eq!(BoundingBox3D::new(Vector3D::new(-4.5, 5.5, 10.5), Vector3D::new(-3.5, 6.5, 11.5)).lower_corner, bbox4.lower_corner);
    /// ```
    fn update_query_engine(&self) {
        self.invalidate_bvh();
        self.build_bvh();
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::plane3::*;
    /// use vox_geometry_rust::bounding_box3::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    /// use vox_geometry_rust::unit_tests_utils::*;
    /// let mut sset1 = SurfaceSet3::new_default();
    ///
    /// let domain = BoundingBox3D::new(Vector3D::new_default(), Vector3D::new(1.0, 2.0, 1.0));
    ///
    /// let plane = Plane3::builder()
    ///                      .with_normal(Vector3D::new(0.0, 1.0, 0.0))
    ///                      .with_point(Vector3D::new(0.0, 0.25 * domain.height(), 0.0))
    ///                      .make_shared();
    ///
    /// let sphere = Sphere3::builder()
    ///                       .with_center(domain.mid_point())
    ///                       .with_radius(0.15 * domain.width())
    ///                       .make_shared();
    ///
    /// let surface_set = SurfaceSet3::builder().with_surfaces(vec![plane, sphere]).make_shared();
    /// assert_eq!(surface_set.borrow().is_bounded(), false);
    ///
    /// let cp = surface_set.borrow().closest_point(&Vector3D::new(0.5, 0.4, 0.5));
    /// let answer = Vector3D::new(0.5, 0.5, 0.5);
    ///
    /// assert_eq!(answer.is_similar(&cp, Some(1.0e-9)), true);
    /// ```
    fn is_bounded(&self) -> bool {
        // All surfaces should be bounded.
        for surface in &self._surfaces {
            if !surface.borrow().is_bounded() {
                return false;
            }
        }

        // Empty set is not bounded.
        return !self._surfaces.is_empty();
    }

    /// ```
    /// use vox_geometry_rust::surface3::*;
    /// use vox_geometry_rust::surface_set3::SurfaceSet3;
    /// use vox_geometry_rust::sphere3::*;
    /// use vox_geometry_rust::plane3::*;
    /// use vox_geometry_rust::bounding_box3::*;
    /// use vox_geometry_rust::vector3::{Vector3D, Vector3};
    /// use vox_geometry_rust::unit_tests_utils::*;
    /// let surfaceSet = SurfaceSet3::builder().make_shared();
    ///
    /// assert_eq!(surfaceSet.borrow().is_valid_geometry(), false);
    ///
    /// let domain = BoundingBox3D::new(Vector3D::new_default(), Vector3D::new(1.0, 2.0, 1.0));
    ///
    /// let plane = Plane3::builder()
    ///                      .with_normal(Vector3D::new(0.0, 1.0, 0.0))
    ///                      .with_point(Vector3D::new(0.0, 0.25 * domain.height(), 0.0))
    ///                      .make_shared();
    ///
    /// let sphere = Sphere3::builder()
    ///                       .with_center(domain.mid_point())
    ///                       .with_radius(0.15 * domain.width())
    ///                       .make_shared();
    ///
    /// let surfaceSet3 =
    ///         SurfaceSet3::builder().with_surfaces(vec![plane, sphere]).make_shared();
    ///
    /// assert_eq!(surfaceSet3.borrow().is_valid_geometry(), true);
    ///
    /// surfaceSet3.borrow_mut().add_surface(surfaceSet);
    ///
    /// assert_eq!(surfaceSet3.borrow().is_valid_geometry(), false);
    /// ```
    fn is_valid_geometry(&self) -> bool {
        // All surfaces should be valid.
        for surface in &self._surfaces {
            if !surface.borrow().is_valid_geometry() {
                return false;
            }
        }

        // Empty set is not valid.
        return !self._surfaces.is_empty();
    }

    fn transform(&self) -> &Transform3 {
        return &self.transform;
    }

    fn set_transform(&mut self, t: Transform3) {
        self.transform = t;
    }

    fn is_normal_flipped(&self) -> bool {
        return self.is_normal_flipped;
    }

    fn set_is_normal_flipped(&mut self, f: bool) {
        self.is_normal_flipped = f;
    }
}

/// Shared pointer for the SurfaceSet3 type.
pub type SurfaceSet3Ptr = Rc<RefCell<SurfaceSet3>>;

///
/// # Front-end to create SurfaceSet3 objects step by step.
///
pub struct Builder {
    _surfaces: Vec<Surface3Ptr>,

    _is_normal_flipped: bool,
    _transform: Transform3,
}

impl Builder {
    /// Returns builder with other surfaces.
    pub fn with_surfaces(&mut self, others: Vec<Surface3Ptr>) -> &mut Self {
        self._surfaces = others;
        return self;
    }

    /// Builds SurfaceSet3.
    pub fn build(&mut self) -> SurfaceSet3 {
        return SurfaceSet3::new(self._surfaces.clone(),
                                Some(self._transform.clone()),
                                Some(self._is_normal_flipped));
    }

    /// Builds shared pointer of SurfaceSet3 instance.
    pub fn make_shared(&mut self) -> SurfaceSet3Ptr {
        return SurfaceSet3Ptr::new(RefCell::new(self.build()));
    }

    /// constructor
    pub fn new() -> Builder {
        return Builder {
            _surfaces: vec![],
            _is_normal_flipped: false,
            _transform: Transform3::new_default(),
        };
    }
}

impl SurfaceBuilderBase3 for Builder {
    fn with_is_normal_flipped(&mut self, is_normal_flipped: bool) -> &mut Self {
        self._is_normal_flipped = is_normal_flipped;
        return self;
    }

    fn with_transform(&mut self, transform: Transform3) -> &mut Self {
        self._transform = transform;
        return self;
    }

    fn transform(&mut self) -> &mut Transform3 {
        return &mut self._transform;
    }
}