use std::fmt;
use delaunator::EMPTY;

use super::{
    Voronoi,
    Point,
    iterator::NeighborSiteIterator
};

/// Represents a Voronoi cell. This is an ergonomic way to access cell details.
///
/// Use [Voronoi::cell()] or [Voronoi::iter_cells()] to obtain an instance of this type.
#[derive(Clone)]
pub struct VoronoiCell<'v> {
    site: usize,
    voronoi: &'v Voronoi
}

impl<'v> VoronoiCell<'v> {
    #[inline]
    pub (super) fn new(site: usize, voronoi: &'v Voronoi) -> Self {
        Self {
            site,
            voronoi
        }
    }

    /// Gets a reference to the position of the site associated with this cell.
    ///
    /// # Examples
    ///
    ///```
    /// use voronoice::*;
    /// let sites = vec![Point { x: 0.0, y: 0.0 }, Point { x: 0.5, y: 0.0 }, Point { x: 0.0, y: 0.5 }];
    /// let v = VoronoiBuilder::default()
    ///     .set_sites(sites.clone())
    ///     .build()
    ///     .unwrap();
    /// // the first site generated by generate_circle_sites is at the origin
    /// assert_eq!(&sites[0], v.cell(0).site_position());
    /// assert_eq!(&sites[1], v.cell(1).site_position());
    /// assert_eq!(&sites[2], v.cell(2).site_position());
    ///```
    pub fn site_position(&self) -> &Point {
        &self.voronoi.sites[self.site]
    }

    /// Gets the index associated with the site for this cell.
    #[inline]
    pub fn site(&self) -> usize {
        self.site
    }

    /// Gets an iterator for the indices of the triangles in the dual Delaunay triangulation that are associated with this cell.
    /// The Voronoi cell vertices are the circumcenters of the associated Delaunay triangles.
    /// This is a way to index into the underlying Delaunay triangles and this cell's vertices.
    ///
    /// If this cell is on the hull of the diagram (```cell.is_on_hull() == true```), or has had one of its edges clipped, some indices will not match to
    /// Delaunay triangles, but to virtual points added during the process of hull closing and clipping. These values will still correctly index into the [Voronoi::vertices()] vector.
    #[inline]
    pub fn iter_triangles(&self) -> impl Iterator<Item = usize> + 'v + Clone {
        self.voronoi.cells[self.site].iter().copied()
    }

    /// Gets an iterator for the vertices of this cell.
    ///
    /// Vertices are returned in sequential counter-clockwise order.
    /// Please see [Self::iter_triangles] and [Voronoi::vertices] for additional details regarding hull closing and clipping effects on vertices.
    #[inline]
    pub fn iter_vertices(&self) -> impl Iterator<Item = &Point> {
        self.iter_triangles().map(move |t| &self.voronoi.circumcenters[t])
    }

    /// Gets an iterator that returns the index of each site that shared an edge with this cell/site, in a counter-clockwise manner.
    ///
    /// # Example
    ///
    ///```
    /// use voronoice::*;
    /// let sites = vec![Point { x: -0.5, y: 0.0 }, Point { x: 0.5, y: 0.0 }, Point { x: 0.0, y: 0.0 }, Point { x: 0.0, y: 0.5 }, Point { x: 0.0, y: -0.5 }];
    /// let v = VoronoiBuilder::default()
    ///     .set_sites(sites.clone())
    ///     .build()
    ///     .unwrap();
    /// let neighbors: Vec<usize> = v.cell(0).iter_neighbors().collect();
    /// assert_eq!(neighbors[0], 4);
    /// assert_eq!(neighbors[1], 2);
    /// assert_eq!(neighbors[2], 3);
    ///```
    #[inline]
    pub fn iter_neighbors(&self) -> NeighborSiteIterator {
        NeighborSiteIterator::new(self.voronoi, self.site)
    }

    /// Gets an iterator that returns the shortest path on the Voronoi diagram to the destination point, starting from the current cell.
    #[inline]
    pub fn iter_path<'p>(&self, dest: Point) ->  impl Iterator<Item = usize> + 'v {
        crate::iterator::shortest_path_iter(self.voronoi, self.site, dest)
    }

    /// Returns a boolean indicating whether this cell is on the hull (edge) of the diagram.
    ///
    /// A Voronoi cell is on the hull if its associated site is on the Delaunay hull or if clipping is enabled and the cell intersects with the bounding box.
    pub fn is_on_hull(&self) -> bool {
        // if there is no half-edge associated with the left-most edge, the edge is on the hull
        let incoming_leftmost_edge = self.voronoi.site_to_incoming_leftmost_halfedge[self.site];
        self.voronoi.triangulation.halfedges[incoming_leftmost_edge] == EMPTY
            // if the cell vertex index is higher than the # of triangles/circumcenters, it means the vertex was added either because
            // it was extending a hull cell or because of clipping (agaisnt bounding box), thus the cell is on the hull
            || self.iter_triangles().any(|t| t > self.voronoi.number_of_triangles())
    }
}

impl<'v> fmt::Debug for  VoronoiCell<'v> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        #[derive(Debug)]
        struct Edge {
            edge: usize,
            incoming_site: usize,
            outgoing_site: usize
        }

        #[derive(Debug)]
        struct Site {
            site: usize,
            position: Point,
            is_on_hull: bool,
            leftmost_incoming_edge: Edge
        }

        #[derive(Debug)]
        struct Cellvertices {
            /// Each vertex is the circumcenter of a associated Delaunay triangle
            triangles: Vec<usize>,
            positions: Vec<Point>
        }
        let leftmost_edge = self.voronoi.site_to_incoming_leftmost_halfedge[self.site];

        f.debug_struct("VoronoiCell")
            .field("site", &Site {
                site: self.site,
                position: self.site_position().clone(),
                is_on_hull: self.is_on_hull(),
                leftmost_incoming_edge: Edge {
                    edge: leftmost_edge,
                    incoming_site: self.site,
                    outgoing_site: self.voronoi.triangulation.triangles[leftmost_edge]
                }
            })
            .field("vertices", &Cellvertices {
                triangles: self.iter_triangles().collect(),
                positions: self.iter_triangles().map(|t| self.voronoi.circumcenters[t].clone()).collect()
            })
            .finish()
    }
}