use std::env;
use std::fs;
use std::fs::File;
use std::path::Path;
use std::thread;
use std::io::{stdin, stdout, Read, Write};
use std::net::{TcpListener, TcpStream, Shutdown};
use std::str;

fn get_content_from_db(db_filename: String, args: Vec<String>) -> String {
    let value = fs::read_to_string(db_filename)
        .expect("Error to read the file");
    let mut output = "".to_string();
    let lines_split = value.split("\n");
    let lines: Vec<&str> = lines_split.collect();
    assert!(lines.len() > 0);
    for line in lines {
        let line_split = line.split(" ");
        let line_args: Vec<&str> = line_split.collect();
        for arg in &args {
            if arg.trim() == line_args[0].trim() {
                output = line_args[1].to_string();
                break;
            }
        }
    }
    return output;
}

fn set_content_in_db(db_filename: String, args: Vec<String>) -> std::io::Result<()> {
    let path = Path::new(&db_filename);
    if !path.exists() {
        fs::write(db_filename, format!("{}\n", args.join(" ")))?;
    } else {
        let mut file = match File::open(&path) {
            Err(_) => panic!("Error to open the database file"),
            Ok(file) => file,
        };

        let mut s = String::new();
        match file.read_to_string(&mut s) {
            Err(_) => panic!("Error to read the database file"),
            Ok(_) => {
                let mut line_acc: Vec<String> = vec![];
                let mut has_changed_line = false;
                let lines_split = s.split("\n");
                let lines: Vec<&str> = lines_split.collect();
                assert!(lines.len() > 0);
                for line in s.split("\n") {
                    let mut data_acc: Vec<String> = vec![];
                    for data in line.split(" ") {
                        if data == args[0] {
                            data_acc.push(data.to_string());
                        }
                    }
                    for arg in &args[1..] {
                        if data_acc.len() > 0 {
                            data_acc.push(arg.to_string().trim().to_string());
                        }
                    }
                    if data_acc.len() > 0 {
                        line_acc.push(data_acc.join(" "));
                        has_changed_line = true;
                    } else {
                        line_acc.push(line.to_string());
                    }
                }
                if has_changed_line {
                    fs::write(db_filename, line_acc.join("\n"))?;
                } else {
                    fs::write(db_filename, format!("{}{}\n", line_acc.join("\n"), args.join(" ")))?;
                }
            }
        }
    }
    Ok(())
}

fn handle_client(mut stream: TcpStream) {
    #![allow(unreachable_code)]
    let mut data = [0 as u8; 50];
    while match stream.read(&mut data) {
        Ok(size) => {
            let buf = &data[0..size];
            let raw_commands = String::from_utf8((&buf).to_vec()).unwrap();
            let raw_commands_split = raw_commands.split(" ");
            let commands: Vec<&str> = raw_commands_split.collect();
            let set_method = "set";
            let get_method = "get";
            let has_set = commands.iter().position(|r| r == &set_method) != None;
            let has_get = commands.iter().position(|r| r == &get_method) != None;
            let db_path = env::current_dir().unwrap();
            let db_filename = db_path.join("volt.db");
            let mut args: Vec<String> = vec![];
            if has_set {
                for arg in commands {
                    args.push(arg.to_string());
                }
                set_content_in_db(db_filename.display().to_string(), args[1..].to_vec()).unwrap();
                let mut s = String::new();
                s.push_str("OK\n");
                stream.write(s.as_bytes()).unwrap();
            } else if has_get {
                for arg in commands {
                    args.push(arg.to_string());
                }
                let value = get_content_from_db(db_filename.display().to_string(), args);
                let mut s = String::new();
                s.push_str(&value);
                s.push_str("\n");
                stream.write(s.as_bytes()).unwrap();
            }
            true
        },
        Err(_) => {
            println!("An error occurred, terminating connection with {}", stream.peer_addr().unwrap());
            stream.shutdown(Shutdown::Both).unwrap();
            false
        },
    } {};
}

fn main() -> std::io::Result<()> {
    let args: Vec<String> = env::args().collect();
    let has_repl = args.iter().position(|r| r == "repl") != None;
    let db_path = env::current_dir()?;
    let db_filename = db_path.join("volt.db");
    if has_repl {
        loop {
            let mut s = String::new();
            print!("> ");
            let _ = stdout().flush();
            stdin()
            .read_line(&mut s)
            .expect("Did not enter a correct string");
            if let Some('\n') = s.chars().next_back() {
                s.pop();
            } if let Some('\r') = s.chars().next_back() {
                s.pop();
            }
            let split = s.split(" ");
            let commands: Vec<&str> = split.collect();
            let mut args: Vec<String> = vec![];
            let get_method = "get";
            let set_method = "set";
            let has_get_in_commands = commands.iter().position(|r| r == &get_method) != None;
            let has_set_in_commands = commands.iter().position(|r| r == &set_method) != None;
            for arg in commands {
                args.push(arg.to_string());
            }
            if has_get_in_commands {
                let value = get_content_from_db(db_filename.display().to_string(), args);
                println!("{}", value);
            } else if has_set_in_commands {
                set_content_in_db(db_filename.display().to_string(), args[1..].to_vec())?;
                println!("OK");
            }
        }
    } else {
        let listener = TcpListener::bind("0.0.0.0:6380").unwrap();
        println!("Server listening on port 6380");
        for stream in listener.incoming() {
            match stream {
                Ok(stream) => {
                    println!("New connection: {}", stream.peer_addr().unwrap());
                    thread::spawn(move || {
                        handle_client(stream);
                    });
                },
                Err(e) => {
                    println!("Error: {}", e);
                }
            }
        }
        drop(listener);
    }
    Ok(())
}
