use async_std::net::{SocketAddr, TcpListener, TcpStream, ToSocketAddrs};

use crate::Result;

/// A server that handles http requests asynchronously.
#[derive(Debug)]
pub struct Server {
    listener: TcpListener,
}

impl Server {
    /// Creates a new [Server] with the specified address.
    ///
    /// ## Examples
    ///
    /// Create a [Server] bound to `127.0.0.1:8000`.
    ///
    /// ```no_run
    /// use voidy::Server;
    ///
    /// # fn main() -> voidy::Result { async_std::task::block_on(async {
    /// let server = Server::bind("127.0.0.1:8000").await?;
    /// # Ok(()) }) }
    /// ```
    pub async fn bind(addrs: impl ToSocketAddrs) -> Result<Self> {
        let listener = TcpListener::bind(addrs).await?;

        Ok(Self { listener })
    }

    /// Accept a new request.
    ///
    /// ## Examples
    ///
    /// ```no_run
    /// use voidy::Server;
    ///
    /// # fn main() -> voidy::Result { async_std::task::block_on(async {
    /// let server = Server::bind("127.0.0.1:8000").await?;
    /// let request = server.accept().await?;
    /// # Ok(()) }) }
    /// ```
    pub async fn accept(&self) -> Result<(TcpStream, SocketAddr)> {
        let conn = self.listener.accept().await?;

        Ok(conn)
    }

    /// Returns the local address that [Server] is bound to.
    ///
    /// ## Examples
    ///
    /// Assert that the local address bound to [Server] is `127.0.0.1:8000`.
    ///
    /// ```no_run
    /// use {
    ///     std::net::{SocketAddr, IpAddr, Ipv4Addr},
    ///     voidy::Server,
    /// };
    ///
    /// # fn main() -> voidy::Result { async_std::task::block_on(async {
    /// let server = Server::bind("127.0.0.1:8000").await?;
    /// let socket = SocketAddr::new(IpAddr::V4(Ipv4Addr::new(127, 0, 0, 1)), 8000);
    ///
    /// assert_eq!(server.local_addr()?, socket);
    /// # Ok(()) }) }
    /// ```
    pub fn local_addr(&self) -> Result<SocketAddr> {
        Ok(self.listener.local_addr()?)
    }
}

impl From<TcpListener> for Server {
    fn from(listener: TcpListener) -> Self {
        Self { listener }
    }
}
