use crate::error::Error;

/// [Http response status code](https://developer.mozilla.org/en-US/docs/Web/HTTP/Status).
#[derive(Debug, Clone, Copy, PartialEq)]
pub enum StatusCode {
    /// 100 Continue.
    Continue,
    /// 101 Switching Protocol.
    SwitchingProtocol,
    /// 102 Processing.
    Processing,
    /// 103 Early Hints.
    EarlyHints,
    /// 200 OK.
    Ok,
    /// 201 Created.
    Created,
    /// 202 Accepted.
    Accepted,
    /// 203 Non-Authoritative Information.
    NonAuthoritativeInformation,
    /// 204 No Content.
    NoContent,
    /// 205 Reset Content.
    ResetContent,
    /// 206 Partial Content.
    PartialContent,
    /// 300 Multiple Choice.
    MultipleChoice,
    /// 301 Moved Permanently.
    MovedPermanently,
    /// 302 Found.
    Found,
    /// 303 See Other.
    SeeOther,
    /// 304 Not Modified.
    NotModified,
    /// 307 Temporary Redirect.
    TemporaryRedirect,
    /// 308 Permanent Redirect.
    PermanentRedirect,
    /// 400 Bad Request.
    BadRequest,
    /// 401 Unauthorized.
    Unauthorized,
    /// 402 Payment Required.
    PaymentRequired,
    /// 403 Forbidden.
    Forbidden,
    /// 404 Not Found.
    NotFound,
    /// 405 Method Not Allowed.
    MethodNotAllowed,
    /// 406 Not Acceptable.
    NotAcceptable,
    /// 407 Proxy Authentication Required.
    ProxyAuthenticationRequired,
    /// 408 Request Timeout.
    RequestTimeout,
    /// 409 Conflict.
    Conflict,
    /// 410 Gone.
    Gone,
    /// 411 Length Required.
    LengthRequired,
    /// 412 Precondition Failed.
    PreconditionFailed,
    /// 413 Payload Too Large.
    PayloadTooLarge,
    /// 414 URI Too Long.
    UriTooLong,
    /// 415 Unsupported Media Type.
    UnsupportedMediaType,
    /// 416 Requested Range Not Satisfiable.
    RequestedRangeNotSatisfiable,
    /// 417 Expectation Failed.
    ExpectationFailed,
    /// 418 I'm a teapot.
    ImATeapot,
    /// 421 Misdirected Request.
    MisdirectedRequest,
    /// 422 Unprocessable Entity.
    UnprocessableEntity,
    /// 423 Locked.
    Locked,
    /// 424 Failed Dependency.
    FailedDependency,
    /// 425 Too Early.
    TooEarly,
    /// 426 Upgrade Required.
    UpgradeRequired,
    /// 428 Precondition Required.
    PreconditionRequired,
    /// 429 Too Many Requests.
    TooManyRequests,
    /// 431 Request Header Fields Too Large.
    RequestHeaderFieldsTooLarge,
    /// 451 Unavailable For Legal Reasons.
    UnavailableForLegalReasons,
    /// 500 Internal Server Error.
    InternalServerError,
    /// 501 Not Implemented.
    NotImplemented,
    /// 502 Bad Gateway.
    BadGateway,
    /// 503 Service Unavailable.
    ServiceUnavailable,
    /// 504 Gateway Timeout.
    GatewayTimeout,
    /// 505 HTTP Version Not Supported.
    HttpVersionNotSupported,
    /// 506 Variant Also Negotiates.
    VariantAlsoNegotiates,
    /// 507 Insufficient Storage.
    InsufficientStorage,
    /// 508 Loop Detected.
    LoopDetected,
    /// 510 Not Extended.
    NotExtended,
    /// 511 Network Authentication Required.
    NetworkAuthenticationRequired,
}

const CONTINUE: &[u8] = b"100 Continue";
const SWITCHING_PROTOCOL: &[u8] = b"101 Switching Protocol";
const PROCESSING: &[u8] = b"102 Processing";
const EARLY_HINTS: &[u8] = b"103 Early Hints";

const OK: &[u8] = b"200 OK";
const CREATED: &[u8] = b"201 Created";
const ACCEPTED: &[u8] = b"202 Accepted";
const NON_AUTHORITATIVE_INFORMATION: &[u8] = b"203 Non-Authoritative Information";
const NO_CONTENT: &[u8] = b"204 No Content";
const RESET_CONTENT: &[u8] = b"205 Reset Content";
const PARTIAL_CONTENT: &[u8] = b"206 Partial Content";

const MULTIPLE_CHOICE: &[u8] = b"300 Multiple Choice";
const MOVED_PERMANENTLY: &[u8] = b"301 Moved Permanently";
const FOUND: &[u8] = b"302 Found";
const SEE_OTHER: &[u8] = b"303 See Other";
const NOT_MODIFIED: &[u8] = b"304 Not Modified";
const TEMPORARY_REDIRECT: &[u8] = b"307 Temporary Redirect";
const PERMANENT_REDIRECT: &[u8] = b"308 Permanent Redirect";

const BAD_REQUEST: &[u8] = b"400 Bad Request";
const UNAUTHORIZED: &[u8] = b"401 Unauthorized";
const PAYMENT_REQUIRED: &[u8] = b"402 Payment Required";
const FORBIDDEN: &[u8] = b"403 Forbidden";
const NOT_FOUND: &[u8] = b"404 Not Found";
const METHOD_NOT_ALLOWED: &[u8] = b"405 Method Not Allowed";
const NOT_ACCEPTABLE: &[u8] = b"406 Not Acceptable";
const PROXY_AUTHENTICATION_REQUIRED: &[u8] = b"407 Proxy Authentication Required";
const REQUEST_TIMEOUT: &[u8] = b"408 Request Timeout";
const CONFLICT: &[u8] = b"409 Conflict";
const GONE: &[u8] = b"410 Gone";
const LENGTH_REQUIRED: &[u8] = b"411 Length Required";
const PRECONDITION_FAILED: &[u8] = b"412 Precondition Failed";
const PAYLOAD_TOO_LARGE: &[u8] = b"413 Payload Too Large";
const URI_TOO_LONG: &[u8] = b"414 URI Too Long";
const UNSUPPORTED_MEDIA_TYPE: &[u8] = b"415 Unsupported Media Type";
const REQUESTED_RANGE_NOT_SATISFIABLE: &[u8] = b"416 Requested Range Not Satisfiable";
const EXPECTATION_FAILED: &[u8] = b"417 Expectation Failed";
const IM_A_TEAPOT: &[u8] = b"418 I'm a teapot";
const MISDIRECTED_REQUEST: &[u8] = b"421 Misdirected Request";
const UNPROCESSABLE_ENTITY: &[u8] = b"422 Unprocessable Entity";
const LOCKED: &[u8] = b"423 Locked";
const FAILED_DEPENDENCY: &[u8] = b"424 Failed Dependency";
const TOO_EARLY: &[u8] = b"425 Too Early";
const UPGRADE_REQUIRED: &[u8] = b"426 Upgrade Required";
const PRECONDITION_REQUIRED: &[u8] = b"428 Precondition Required";
const TOO_MANY_REQUESTS: &[u8] = b"429 Too Many Requests";
const REQUEST_HEADER_FIELDS_TOO_LARGE: &[u8] = b"431 Request Header Fields Too Large";
const UNAVAILABLE_FOR_LEGAL_REASONS: &[u8] = b"451 Unavailable For Legal Reasons";

const INTERNAL_SERVER_ERROR: &[u8] = b"500 Internal Server Error";
const NOT_IMPLEMENTED: &[u8] = b"501 Not Implemented";
const BAD_GATEWAY: &[u8] = b"502 Bad Gateway";
const SERVICE_UNAVAIBLE: &[u8] = b"503 Service Unavailable";
const GATEWAY_TIMEOUT: &[u8] = b"504 Gateway Timeout";
const HTTP_VERSION_NOT_SUPPORTED: &[u8] = b"505 HTTP Version Not Supported";
const VARIANT_ALSO_NEGOTIATES: &[u8] = b"506 Variant Also Negotiates";
const INSUFFICIENT_STORAGE: &[u8] = b"507 Insufficient Storage";
const LOOP_DETECTED: &[u8] = b"508 Loop Detected";
const NOT_EXTENDED: &[u8] = b"510 Not Extended";
const NETWORK_AUTHENTICATION_REQUIRED: &[u8] = b"511 Network Authentication Required";

impl Default for StatusCode {
    fn default() -> Self {
        Self::Ok
    }
}

impl Into<&'static [u8]> for StatusCode {
    fn into(self) -> &'static [u8] {
        match self {
            Self::Continue => CONTINUE,
            Self::SwitchingProtocol => SWITCHING_PROTOCOL,
            Self::Processing => PROCESSING,
            Self::EarlyHints => EARLY_HINTS,
            Self::Ok => OK,
            Self::Created => CREATED,
            Self::Accepted => ACCEPTED,
            Self::NonAuthoritativeInformation => NON_AUTHORITATIVE_INFORMATION,
            Self::NoContent => NO_CONTENT,
            Self::ResetContent => RESET_CONTENT,
            Self::PartialContent => PARTIAL_CONTENT,
            Self::MultipleChoice => MULTIPLE_CHOICE,
            Self::MovedPermanently => MOVED_PERMANENTLY,
            Self::Found => FOUND,
            Self::SeeOther => SEE_OTHER,
            Self::NotModified => NOT_MODIFIED,
            Self::TemporaryRedirect => TEMPORARY_REDIRECT,
            Self::PermanentRedirect => PERMANENT_REDIRECT,
            Self::BadRequest => BAD_REQUEST,
            Self::Unauthorized => UNAUTHORIZED,
            Self::PaymentRequired => PAYMENT_REQUIRED,
            Self::Forbidden => FORBIDDEN,
            Self::NotFound => NOT_FOUND,
            Self::MethodNotAllowed => METHOD_NOT_ALLOWED,
            Self::NotAcceptable => NOT_ACCEPTABLE,
            Self::ProxyAuthenticationRequired => PROXY_AUTHENTICATION_REQUIRED,
            Self::RequestTimeout => REQUEST_TIMEOUT,
            Self::Conflict => CONFLICT,
            Self::Gone => GONE,
            Self::LengthRequired => LENGTH_REQUIRED,
            Self::PreconditionFailed => PRECONDITION_FAILED,
            Self::PayloadTooLarge => PAYLOAD_TOO_LARGE,
            Self::UriTooLong => URI_TOO_LONG,
            Self::UnsupportedMediaType => UNSUPPORTED_MEDIA_TYPE,
            Self::RequestedRangeNotSatisfiable => REQUESTED_RANGE_NOT_SATISFIABLE,
            Self::ExpectationFailed => EXPECTATION_FAILED,
            Self::ImATeapot => IM_A_TEAPOT,
            Self::MisdirectedRequest => MISDIRECTED_REQUEST,
            Self::UnprocessableEntity => UNPROCESSABLE_ENTITY,
            Self::Locked => LOCKED,
            Self::FailedDependency => FAILED_DEPENDENCY,
            Self::TooEarly => TOO_EARLY,
            Self::UpgradeRequired => UPGRADE_REQUIRED,
            Self::PreconditionRequired => PRECONDITION_REQUIRED,
            Self::TooManyRequests => TOO_MANY_REQUESTS,
            Self::RequestHeaderFieldsTooLarge => REQUEST_HEADER_FIELDS_TOO_LARGE,
            Self::UnavailableForLegalReasons => UNAVAILABLE_FOR_LEGAL_REASONS,
            Self::InternalServerError => INTERNAL_SERVER_ERROR,
            Self::NotImplemented => NOT_IMPLEMENTED,
            Self::BadGateway => BAD_GATEWAY,
            Self::ServiceUnavailable => SERVICE_UNAVAIBLE,
            Self::GatewayTimeout => GATEWAY_TIMEOUT,
            Self::HttpVersionNotSupported => HTTP_VERSION_NOT_SUPPORTED,
            Self::VariantAlsoNegotiates => VARIANT_ALSO_NEGOTIATES,
            Self::InsufficientStorage => INSUFFICIENT_STORAGE,
            Self::LoopDetected => LOOP_DETECTED,
            Self::NotExtended => NOT_EXTENDED,
            Self::NetworkAuthenticationRequired => NETWORK_AUTHENTICATION_REQUIRED,
        }
    }
}

impl TryFrom<&[u8]> for StatusCode {
    type Error = Error;

    fn try_from(bytes: &[u8]) -> Result<Self, Self::Error> {
        match bytes {
            CONTINUE => Ok(Self::Continue),
            SWITCHING_PROTOCOL => Ok(Self::SwitchingProtocol),
            PROCESSING => Ok(Self::Processing),
            EARLY_HINTS => Ok(Self::EarlyHints),
            OK => Ok(Self::Ok),
            CREATED => Ok(Self::Created),
            ACCEPTED => Ok(Self::Accepted),
            NON_AUTHORITATIVE_INFORMATION => Ok(Self::NonAuthoritativeInformation),
            NO_CONTENT => Ok(Self::NoContent),
            RESET_CONTENT => Ok(Self::ResetContent),
            PARTIAL_CONTENT => Ok(Self::PartialContent),
            MULTIPLE_CHOICE => Ok(Self::MultipleChoice),
            MOVED_PERMANENTLY => Ok(Self::MovedPermanently),
            FOUND => Ok(Self::Found),
            SEE_OTHER => Ok(Self::SeeOther),
            NOT_MODIFIED => Ok(Self::NotModified),
            TEMPORARY_REDIRECT => Ok(Self::TemporaryRedirect),
            PERMANENT_REDIRECT => Ok(Self::PermanentRedirect),
            BAD_REQUEST => Ok(Self::BadRequest),
            UNAUTHORIZED => Ok(Self::Unauthorized),
            PAYMENT_REQUIRED => Ok(Self::PaymentRequired),
            FORBIDDEN => Ok(Self::Forbidden),
            NOT_FOUND => Ok(Self::NotFound),
            METHOD_NOT_ALLOWED => Ok(Self::MethodNotAllowed),
            NOT_ACCEPTABLE => Ok(Self::NotAcceptable),
            PROXY_AUTHENTICATION_REQUIRED => Ok(Self::ProxyAuthenticationRequired),
            REQUEST_TIMEOUT => Ok(Self::RequestTimeout),
            CONFLICT => Ok(Self::Conflict),
            GONE => Ok(Self::Gone),
            LENGTH_REQUIRED => Ok(Self::LengthRequired),
            PRECONDITION_FAILED => Ok(Self::PreconditionFailed),
            PAYLOAD_TOO_LARGE => Ok(Self::PayloadTooLarge),
            URI_TOO_LONG => Ok(Self::UriTooLong),
            UNSUPPORTED_MEDIA_TYPE => Ok(Self::UnsupportedMediaType),
            REQUESTED_RANGE_NOT_SATISFIABLE => Ok(Self::RequestedRangeNotSatisfiable),
            EXPECTATION_FAILED => Ok(Self::ExpectationFailed),
            IM_A_TEAPOT => Ok(Self::ImATeapot),
            MISDIRECTED_REQUEST => Ok(Self::MisdirectedRequest),
            UNPROCESSABLE_ENTITY => Ok(Self::UnprocessableEntity),
            LOCKED => Ok(Self::Locked),
            FAILED_DEPENDENCY => Ok(Self::FailedDependency),
            TOO_EARLY => Ok(Self::TooEarly),
            UPGRADE_REQUIRED => Ok(Self::UpgradeRequired),
            PRECONDITION_REQUIRED => Ok(Self::PreconditionRequired),
            TOO_MANY_REQUESTS => Ok(Self::TooManyRequests),
            REQUEST_HEADER_FIELDS_TOO_LARGE => Ok(Self::RequestHeaderFieldsTooLarge),
            UNAVAILABLE_FOR_LEGAL_REASONS => Ok(Self::UnavailableForLegalReasons),
            INTERNAL_SERVER_ERROR => Ok(Self::InternalServerError),
            NOT_IMPLEMENTED => Ok(Self::NotImplemented),
            BAD_GATEWAY => Ok(Self::BadGateway),
            SERVICE_UNAVAIBLE => Ok(Self::ServiceUnavailable),
            GATEWAY_TIMEOUT => Ok(Self::GatewayTimeout),
            HTTP_VERSION_NOT_SUPPORTED => Ok(Self::HttpVersionNotSupported),
            VARIANT_ALSO_NEGOTIATES => Ok(Self::VariantAlsoNegotiates),
            INSUFFICIENT_STORAGE => Ok(Self::InsufficientStorage),
            LOOP_DETECTED => Ok(Self::LoopDetected),
            NOT_EXTENDED => Ok(Self::NotExtended),
            NETWORK_AUTHENTICATION_REQUIRED => Ok(Self::NetworkAuthenticationRequired),
            _ => Err(Error::InvalidRequest),
        }
    }
}
