use std::fmt::{self, Debug, Formatter};

use crate::http::{Body, Headers, StatusCode};

/// Http reponse type.
#[derive(Clone, PartialEq)]
pub struct Response {
    status_code: StatusCode,
    headers: Headers,
    body: Body,
}

impl Response {
    /// Returns the [response](Response) [status code](StatusCode).
    pub fn status_code(&self) -> StatusCode {
        self.status_code
    }

    /// Returns the [response](Response) headers.
    pub fn headers(&self) -> &Headers {
        &self.headers
    }

    /// Returns the [response](Response) body.
    pub fn body(&self) -> Option<&[u8]> {
        match self.body {
            Some(ref body) => Some(body),
            None => None,
        }
    }

    /// Creates a new [Response].
    pub fn new(status_code: StatusCode, headers: Headers, body: Body) -> Self {
        Self {
            status_code,
            headers,
            body,
        }
    }
}

impl Debug for Response {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        if let Some(ref body) = self.body {
            f.debug_struct("Response")
                .field("status_code", &self.status_code)
                .field("headers", &self.headers)
                .field("body", &String::from_utf8_lossy(body))
                .finish()
        } else {
            f.debug_struct("Response")
                .field("status_code", &self.status_code)
                .field("headers", &self.headers)
                .finish()
        }
    }
}

impl From<StatusCode> for Response {
    fn from(status_code: StatusCode) -> Self {
        Self::new(status_code, Headers::new(), None)
    }
}
