use {
    async_std::{io::Write, net::TcpStream},
    std::fmt::{self, Debug, Formatter},
};

use crate::http::{Body, Headers, Method};

/// Http request type.
#[derive(Clone, PartialEq)]
pub struct Request<T: Write + Unpin = TcpStream> {
    method: Method,
    path: String,
    host: String,
    headers: Headers,
    body: Body,
    stream: Option<T>,
}

impl<T: Write + Unpin> Request<T> {
    /// Returns the [request](Request) [method](Method).
    pub fn method(&self) -> Method {
        self.method
    }

    /// Returns the [request](Request) path.
    pub fn path(&self) -> &str {
        &self.path
    }

    /// Returns the [request](Request) host.
    pub fn host(&self) -> &str {
        &self.host
    }

    /// Returns the [request](Request) headers.
    pub fn headers(&self) -> &Headers {
        &self.headers
    }

    /// Returns the [request](Request) body.
    pub fn body(&self) -> Option<&[u8]> {
        match self.body {
            Some(ref body) => Some(body),
            None => None,
        }
    }

    pub(crate) fn new(
        method: Method,
        path: String,
        host: String,
        headers: Headers,
        body: Body,
        stream: Option<T>,
    ) -> Self {
        Self {
            method,
            path,
            host,
            headers,
            body,
            stream,
        }
    }
}

impl<T: Write + Unpin> Debug for Request<T> {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        if let Some(ref body) = self.body {
            f.debug_struct("Request")
                .field("method", &self.method)
                .field("path", &self.path)
                .field("host", &self.host)
                .field("headers", &self.headers)
                .field("body", &String::from_utf8_lossy(body))
                .finish()
        } else {
            f.debug_struct("Request")
                .field("method", &self.method)
                .field("path", &self.path)
                .field("host", &self.host)
                .field("headers", &self.headers)
                .finish()
        }
    }
}

impl<T: Write + Unpin> Drop for Request<T> {
    fn drop(&mut self) {
        if let Some(_) = self.stream {
            panic!("Request not responded: {:?}", self);
        }
    }
}
