use crate::error::Error;

/// Http protocol request methods.
#[derive(Debug, Clone, Copy, PartialEq)]
pub enum Method {
    /// `GET` method.
    Get,
    /// `HEAD` method.
    Head,
    /// `POST` method.
    Post,
    /// `PUT` method.
    Put,
    /// `DELETE` method.
    Delete,
    /// `CONNECT` method.
    Connect,
    /// `OPTIONS` method.
    Options,
    /// `TRACE` method.
    Trace,
    /// `PATCH` method.
    Patch,
}

const GET: &[u8] = b"GET";
const HEAD: &[u8] = b"HEAD";
const POST: &[u8] = b"POST";
const PUT: &[u8] = b"PUT";
const DELETE: &[u8] = b"DELETE";
const CONNECT: &[u8] = b"CONNECT";
const OPTIONS: &[u8] = b"OPTIONS";
const TRACE: &[u8] = b"TRACE";
const PATCH: &[u8] = b"PATCH";

impl Into<&'static [u8]> for Method {
    fn into(self) -> &'static [u8] {
        match self {
            Self::Get => GET,
            Self::Head => HEAD,
            Self::Post => POST,
            Self::Put => PUT,
            Self::Delete => DELETE,
            Self::Connect => CONNECT,
            Self::Options => OPTIONS,
            Self::Trace => TRACE,
            Self::Patch => PATCH,
        }
    }
}

impl TryFrom<&[u8]> for Method {
    type Error = Error;

    fn try_from(bytes: &[u8]) -> Result<Self, Self::Error> {
        match bytes {
            GET => Ok(Self::Get),
            HEAD => Ok(Self::Head),
            POST => Ok(Self::Post),
            PUT => Ok(Self::Put),
            DELETE => Ok(Self::Delete),
            CONNECT => Ok(Self::Connect),
            OPTIONS => Ok(Self::Options),
            TRACE => Ok(Self::Trace),
            PATCH => Ok(Self::Patch),
            _ => Err(Error::InvalidRequest),
        }
    }
}
