//! [Error] and [Result] types module.

use std::{
    error::Error as StdError,
    fmt::{self, Display, Formatter},
    io,
    result::Result as StdResult,
};

/// Result type for [Error].
pub type Result<T = ()> = StdResult<T, Error>;

/// Errors that can happen using [voidy](crate).
#[derive(Debug)]
pub enum Error {
    /// I/O operations error.
    Io(io::Error),
    /// Cannot parse request.
    InvalidRequest,
}

impl Display for Error {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        match self {
            Self::Io(err) => err.fmt(f),
            Self::InvalidRequest => writeln!(f, "Error: {:?}", Self::InvalidRequest),
        }
    }
}

impl StdError for Error {
    fn source(&self) -> Option<&(dyn StdError + 'static)> {
        match self {
            Self::Io(err) => Some(err),
            _ => None,
        }
    }
}

impl From<io::Error> for Error {
    fn from(err: io::Error) -> Self {
        Self::Io(err)
    }
}
