/*
	VMKS exam generator - A simple program for pseudo-randomly generating different variants of an embedded programming exam
	Copyright (C) 2021 Vladimir Garistov <vl.garistov@gmail.com>

	This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as
	published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along with this program; if not, write to the
	Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

use std::fs::File;
use std::io::Read;
use std::error::Error;
use std::fmt::{Display, Formatter};
use rand_pcg::Lcg128Xsl64;
use rand::seq::SliceRandom;
use rand::Rng;
use xml::reader::{XmlEvent, EventReader, ParserConfig};

const PICK_ALL: i32 = -1;
const PICK_ATTRIBUTE_NAME: &str = "pick";
const QUESTION_BANK_TAG_NAME: &str = "question_bank";
const QUESTION_GROUP_TAG_NAME: &str = "question_group";
const QUESTION_MULTIPLE_CHOICE_TAG_NAME: &str = "question_mc";
const QUESTION_VARIABLE_TAG_NAME: &str = "question_var";
const QUESTION_TEXT_TAG_NAME: &str = "question_text";
const ANSWER_MULTIPLE_CHOICE_TAG_NAME: &str = "answer_mc";
const VARIABLE_TEXT_TAG_NAME: &str = "var_text";
const TEXT_OPTION_TAG_NAME: &str = "option";

#[derive(Debug)]
pub enum QuestionBankError
{
	UnexpectedTag
	{
		expected: String,
		received: String
	},
	UnexpectedAttribute
	{
		expected: String,
		received: String
	},
	UnexpectedText
	{
		expected: String,
		received: String
	},
	PickTooMany,
	InvalidPickValue
	{
		received: String
	},
	GenericParseError
}

impl Display for QuestionBankError
{
	// TODO: make the quotes more consistent
	fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result
	{
		match self
		{
			QuestionBankError::UnexpectedTag{expected, received} =>
				if expected == ""
				{
					write!(f, "unexpected tag. Expected no more tags, got {}.", received)
				}
				else
				{
					write!(f, "unexpected tag. Expected {}, got {}.", expected, received)
				},
			QuestionBankError::UnexpectedAttribute{expected, received} =>
				if expected == ""
				{
					write!(f, "unexpected attribute. Expected no more attributes, got \"{}\".", received)
				}
				else
				{
					write!(f, "unexpected attribute. Expected \"{}\", got \"{}\".", expected, received)
				},
			QuestionBankError::UnexpectedText{expected, received} =>
				write!(f, "unexpected text. Expected {}, got \"{}\".", expected, received),
			QuestionBankError::PickTooMany =>
				write!(f, "specified number of questions to pick from a question group is larger than the number of questions in that group."),
			QuestionBankError::InvalidPickValue{received: expected} =>
				write!(f, "invalid value provided for \"pick\" argument. Expected a non-negative integer, got \"{}\"", expected),
			QuestionBankError::GenericParseError =>
				write!(f, "unable to parse question bank file."),
		}
	}
}

impl Error for QuestionBankError
{}

enum VariableQuestionSegment
{
	QuestionText(String),
	VariableText(Vec<String>)
}

// Multiple choice question parsing finite state machine
#[allow(non_camel_case_types)]
#[derive(PartialEq)]
enum MCQ_Parsing_FSM
{
	EndOrQuestionTextOpen,
	QuestionTextBody,
	QuestionTextClose,
	EndOrAnswerOpen,
	AnswerBody,
	AnswerClose
}

impl MCQ_Parsing_FSM
{
	fn get_expected(&self) -> String
	{
		match self
		{
			MCQ_Parsing_FSM::EndOrQuestionTextOpen =>
				format!("{} or /{}", QUESTION_TEXT_TAG_NAME, QUESTION_MULTIPLE_CHOICE_TAG_NAME),
			MCQ_Parsing_FSM::QuestionTextBody =>
				"question text".to_string(),
			MCQ_Parsing_FSM::QuestionTextClose =>
				QUESTION_TEXT_TAG_NAME.to_string(),
			MCQ_Parsing_FSM::EndOrAnswerOpen =>
				format!("{} or /{}", ANSWER_MULTIPLE_CHOICE_TAG_NAME, QUESTION_MULTIPLE_CHOICE_TAG_NAME),
			MCQ_Parsing_FSM::AnswerBody =>
				"answer text".to_string(),
			MCQ_Parsing_FSM::AnswerClose =>
				ANSWER_MULTIPLE_CHOICE_TAG_NAME.to_string(),
		}
	}
}

// Variable question parsing finite state machine
#[allow(non_camel_case_types)]
#[derive(PartialEq)]
enum VQ_Parsing_FSM
{
	EndOrQuestionTextOpenOrVarTextOpen,
	QuestionTextBody,
	QuestionTextClose,
	VarTextCloseOrOptionOpen,
	OptionBody,
	OptionClose
}

impl VQ_Parsing_FSM
{
	fn get_expected(&self) -> String
	{
		match self
		{
			VQ_Parsing_FSM::EndOrQuestionTextOpenOrVarTextOpen =>
				format!("{}, {} or /{}", QUESTION_TEXT_TAG_NAME, VARIABLE_TEXT_TAG_NAME, QUESTION_VARIABLE_TAG_NAME),
			VQ_Parsing_FSM::QuestionTextBody =>
				"question text".to_string(),
			VQ_Parsing_FSM::QuestionTextClose =>
				format!("/{}", QUESTION_TEXT_TAG_NAME),
			VQ_Parsing_FSM::VarTextCloseOrOptionOpen =>
				format!("{} or /{}", TEXT_OPTION_TAG_NAME, TEXT_OPTION_TAG_NAME),
			VQ_Parsing_FSM::OptionBody =>
				"optional text".to_string(),
			VQ_Parsing_FSM::OptionClose =>
				format!("/{}", TEXT_OPTION_TAG_NAME)
		}
	}
}

enum Question
{
	MultipleChoice
	{
		question_text: String,
		possible_answers: Vec<String>
	},
	Variable
	{
		segments: Vec<VariableQuestionSegment>
	}
}

impl Question
{
	pub fn multiple_choice_from_xml_reader<R: Read>(xml_reader: &mut EventReader<R>) -> Result<Question, Box<dyn Error>>
	{
		let mut state = MCQ_Parsing_FSM::EndOrQuestionTextOpen;
		let mut question_text = String::new();
		let mut possible_answers: Vec<String> = Vec::new();

		loop
		{
			let xml_event = xml_reader.next()?;
			match xml_event
			{
				XmlEvent::Whitespace(_) => continue,
				XmlEvent::StartElement{name, attributes, ..} =>
					{
						if let Some(attr) = attributes.iter().next()
						{
							return Err(Box::new(QuestionBankError::UnexpectedAttribute
							{
								expected: "".to_string(),
								received: attr.name.local_name.clone()
							}));
						};
						match &name.local_name[..]
						{
							QUESTION_TEXT_TAG_NAME =>
								{
									if state == MCQ_Parsing_FSM::EndOrQuestionTextOpen
									{
										state = MCQ_Parsing_FSM::QuestionTextBody;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							ANSWER_MULTIPLE_CHOICE_TAG_NAME =>
								{
									if state == MCQ_Parsing_FSM::EndOrAnswerOpen
									{
										state = MCQ_Parsing_FSM::AnswerBody;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							_ =>
								return Err(Box::new(QuestionBankError::UnexpectedTag
								{
									expected: state.get_expected(),
									received: name.local_name.clone()
								}))
						}
					},
				XmlEvent::Characters(text) =>
					{
						match state
						{
							MCQ_Parsing_FSM::QuestionTextBody =>
								{
									question_text = text;
									state = MCQ_Parsing_FSM::QuestionTextClose;
								},
							MCQ_Parsing_FSM::AnswerBody =>
								{
									possible_answers.push(text);
									state = MCQ_Parsing_FSM::AnswerClose;
								},
							_ =>
								return Err(Box::new(QuestionBankError::UnexpectedText
								{
									expected: state.get_expected(),
									received: text
								}))
						}
					},
				XmlEvent::EndElement{name} =>
					{
						match &name.local_name[..]
						{
							QUESTION_TEXT_TAG_NAME =>
								{
									if state == MCQ_Parsing_FSM::QuestionTextClose
									{
										state = MCQ_Parsing_FSM::EndOrAnswerOpen;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							ANSWER_MULTIPLE_CHOICE_TAG_NAME =>
								{
									if state == MCQ_Parsing_FSM::AnswerClose
									{
										state = MCQ_Parsing_FSM::EndOrAnswerOpen;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							QUESTION_MULTIPLE_CHOICE_TAG_NAME =>
								{
									if state == MCQ_Parsing_FSM::EndOrQuestionTextOpen || state == MCQ_Parsing_FSM::EndOrAnswerOpen
									{
										break;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							_ =>
								return Err(Box::new(QuestionBankError::UnexpectedTag
								{
									expected: state.get_expected(),
									received: name.local_name.clone()
								}))
						}
					},
				_ => return Err(Box::new(QuestionBankError::GenericParseError))
			}
		}

		Ok(Question::MultipleChoice
		{
			question_text,
			possible_answers
		})
	}

	pub fn variable_from_xml_reader<R: Read>(xml_reader: &mut EventReader<R>) -> Result<Question, Box<dyn Error>>
	{
		let mut state = VQ_Parsing_FSM::EndOrQuestionTextOpenOrVarTextOpen;
		let mut segments: Vec<VariableQuestionSegment> = Vec::new();

		loop
		{
			let xml_event = xml_reader.next()?;
			match xml_event
			{
				XmlEvent::Whitespace(_) => continue,
				XmlEvent::StartElement{name, attributes, ..} =>
					{
						if let Some(attr) = attributes.iter().next()
						{
							return Err(Box::new(QuestionBankError::UnexpectedAttribute
							{
								expected: "".to_string(),
								received: attr.name.local_name.clone()
							}));
						};
						match &name.local_name[..]
						{
							QUESTION_TEXT_TAG_NAME =>
								{
									if state == VQ_Parsing_FSM::EndOrQuestionTextOpenOrVarTextOpen
									{
										state = VQ_Parsing_FSM::QuestionTextBody;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							VARIABLE_TEXT_TAG_NAME =>
								{
									if state == VQ_Parsing_FSM::EndOrQuestionTextOpenOrVarTextOpen
									{
										segments.push(VariableQuestionSegment::VariableText(Vec::new()));
										state = VQ_Parsing_FSM::VarTextCloseOrOptionOpen;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							TEXT_OPTION_TAG_NAME =>
								{
									if state == VQ_Parsing_FSM::VarTextCloseOrOptionOpen
									{
										state = VQ_Parsing_FSM::OptionBody;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							_ =>
								return Err(Box::new(QuestionBankError::UnexpectedTag
								{
									expected: state.get_expected(),
									received: name.local_name.clone()
								}))
						}
					},
				XmlEvent::Characters(text) =>
					{
						match state
						{
							VQ_Parsing_FSM::QuestionTextBody =>
								{
									segments.push(VariableQuestionSegment::QuestionText(text));
									state = VQ_Parsing_FSM::QuestionTextClose;
								},
							VQ_Parsing_FSM::OptionBody =>
								{
									let last_segment_index = segments.len() - 1;
									match &mut segments[last_segment_index]
									{
										VariableQuestionSegment::VariableText(options) =>
											{
												options.push(text);
											},
										_ => panic!("attempted to add options to question_text")
									}
									state = VQ_Parsing_FSM::OptionClose;
								},
							_ =>
								return Err(Box::new(QuestionBankError::UnexpectedText
								{
									expected: state.get_expected(),
									received: text
								}))
						}
					},
				XmlEvent::EndElement{name} =>
					{
						match &name.local_name[..]
						{
							QUESTION_TEXT_TAG_NAME =>
								{
									if state == VQ_Parsing_FSM::QuestionTextClose
									{
										state = VQ_Parsing_FSM::EndOrQuestionTextOpenOrVarTextOpen;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							VARIABLE_TEXT_TAG_NAME =>
								{
									if state == VQ_Parsing_FSM::VarTextCloseOrOptionOpen
									{
										state = VQ_Parsing_FSM::EndOrQuestionTextOpenOrVarTextOpen;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							TEXT_OPTION_TAG_NAME =>
								{
									if state == VQ_Parsing_FSM::OptionClose
									{
										state = VQ_Parsing_FSM::VarTextCloseOrOptionOpen;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							QUESTION_VARIABLE_TAG_NAME =>
								{
									if state == VQ_Parsing_FSM::EndOrQuestionTextOpenOrVarTextOpen
									{
										break;
									}
									else
									{
										return Err(Box::new(QuestionBankError::UnexpectedTag
										{
											expected: state.get_expected(),
											received: name.local_name.clone()
										}));
									}
								},
							_ =>
								return Err(Box::new(QuestionBankError::UnexpectedTag
								{
									expected: state.get_expected(),
									received: name.local_name.clone()
								}))
						}
					},
				_ => return Err(Box::new(QuestionBankError::GenericParseError))
			}
		}

		Ok(Question::Variable
		{
			segments
		})
	}

	pub fn randomise(&self, rng: &mut Lcg128Xsl64) -> String
	{
		let mut randomised_question = String::with_capacity(1024);

		match self
		{
			Question::MultipleChoice{question_text, possible_answers} =>
				{
					randomised_question.push_str(question_text);
					randomised_question.push('\n');
					let num_answers = possible_answers.len();
					let answers_range = 0..num_answers;
					let mut answers_range = answers_range.collect::<Vec<usize>>();
					answers_range.shuffle(rng);
					for i in 0..num_answers
					{
						randomised_question.push_str("\t- ");
						randomised_question.push_str(&possible_answers[answers_range[i]]);
						randomised_question.push('\n');
					}
					randomised_question.push('\n');
				},
			Question::Variable{segments} =>
				{
					for segment in segments
					{
						match segment
						{
							VariableQuestionSegment::QuestionText(text) => randomised_question.push_str(text),
							VariableQuestionSegment::VariableText(options) =>
								randomised_question.push_str(&options[rng.gen_range(0, options.len())])
						}
					}
					randomised_question.push_str("\n\n");
				}
		}

		randomised_question
	}
}

struct QuestionGroup
{
	questions: Vec<Question>,
	pick: u32
}

impl QuestionGroup
{
	pub fn from_xml_reader<R: Read>(xml_reader: &mut EventReader<R>, questions_to_pick: i32) -> Result<QuestionGroup, Box<dyn Error>>
	{
		let mut questions: Vec<Question> = Vec::new();
		loop
		{
			let xml_event = xml_reader.next()?;
			match xml_event
			{
				XmlEvent::Whitespace(_) => continue,
				XmlEvent::StartElement{name, attributes, ..} =>
				{
					if let Some(attr) = attributes.iter().next()
					{
						return Err(Box::new(QuestionBankError::UnexpectedAttribute
						{
							expected: "".to_string(),
							received: attr.name.local_name.clone()
						}));
					};
					match &name.local_name[..]
					{
						QUESTION_MULTIPLE_CHOICE_TAG_NAME =>
							questions.push(Question::multiple_choice_from_xml_reader(xml_reader)?),
						QUESTION_VARIABLE_TAG_NAME =>
							questions.push(Question::variable_from_xml_reader(xml_reader)?),
						_ =>
						{
							let mut expected = QUESTION_MULTIPLE_CHOICE_TAG_NAME.to_string();
							expected.push_str(" or ");
							expected.push_str(QUESTION_VARIABLE_TAG_NAME);
							return Err(Box::new(QuestionBankError::UnexpectedTag
							{
								expected,
								received: name.local_name.clone()
							}));
						}
					}
				},
				XmlEvent::Characters(text) =>
				{
					let expected = format!("{}, {} or /{}", QUESTION_MULTIPLE_CHOICE_TAG_NAME,
										   QUESTION_VARIABLE_TAG_NAME, QUESTION_GROUP_TAG_NAME);
					return Err(Box::new(QuestionBankError::UnexpectedText
					{
						expected,
						received: text
					}))
				},
				XmlEvent::EndElement{name} =>
				{
					if name.local_name == QUESTION_GROUP_TAG_NAME
					{
						break;
					}
					else
					{
						let mut expected = String::from("/");
						expected.push_str(QUESTION_GROUP_TAG_NAME);
						return Err(Box::new(QuestionBankError::UnexpectedTag
						{
							expected,
							received: name.local_name.clone()
						}));
					}
				},
				_ => return Err(Box::new(QuestionBankError::GenericParseError))
			}
		}

		let pick = match questions_to_pick
		{
			PICK_ALL => questions.len() as u32,
			n if n as usize > questions.len() => return Err(Box::new(QuestionBankError::PickTooMany)),
			n => n as u32
		};

		Ok(QuestionGroup
		{
			questions,
			pick
		})
	}

	pub fn randomise(&self, rng: &mut Lcg128Xsl64, question_counter: &mut u32) -> String
	{
		let mut random_selection = String::with_capacity(2048);
		let questions_range = 0..self.questions.len();
		let mut questions_range = questions_range.collect::<Vec<usize>>();
		questions_range.shuffle(rng);
		for i in 0..self.pick as usize
		{
			*question_counter += 1;
			random_selection.push_str(&format!("Задача {}: {}",
				*question_counter,
				self.questions[questions_range[i]].randomise(rng)));
		}

		random_selection
	}
}

pub struct QuestionBank
{
	question_groups: Vec<QuestionGroup>
}

impl QuestionBank
{
	pub fn from_file(filename: &str) -> Result<QuestionBank, Box<dyn Error>>
	{
		let parser_config = ParserConfig::new().cdata_to_characters(true);
		let mut question_groups: Vec<QuestionGroup> = Vec::new();
		let question_bank_file = File::open(filename)?;
		let mut question_bank_reader = EventReader::new_with_config(question_bank_file, parser_config);

		// Verify that the first tag is a start-of-document tag
		if let XmlEvent::StartDocument{..} = question_bank_reader.next()? {}
		else
		{
			return Err(Box::new(QuestionBankError::GenericParseError));
		}
		// Find opening question_bank tag and make sure it has no attributes
		loop
		{
			let xml_event = question_bank_reader.next()?;
			match xml_event
			{
				XmlEvent::Whitespace(_) => continue,
				XmlEvent::StartElement{name, attributes, ..} =>
				{
					if name.local_name == QUESTION_BANK_TAG_NAME
					{
						if let Some(attr) = attributes.iter().next()
						{
							return Err(Box::new(QuestionBankError::UnexpectedAttribute
							{
								expected: "".to_string(),
								received: attr.name.local_name.clone()
							}));
						};
						break;
					}
					else
					{
						return Err(Box::new(QuestionBankError::UnexpectedTag
						{
							expected: QUESTION_BANK_TAG_NAME.to_string(),
							received: name.local_name.clone()
						}));
					}
				},
				XmlEvent::Characters(text) =>
					return Err(Box::new(QuestionBankError::UnexpectedText
					{
						expected: QUESTION_BANK_TAG_NAME.to_string(),
						received: text
					})),
				_ => return Err(Box::new(QuestionBankError::GenericParseError))
			}
		}

		// Find closing question_bank tag
		// Parse all encountered question groups
		loop
		{
			let xml_event = question_bank_reader.next()?;
			match xml_event
			{
				XmlEvent::Whitespace(_) => continue,
				XmlEvent::StartElement{name, attributes, ..} =>
				{
					if name.local_name == QUESTION_GROUP_TAG_NAME
					{
						let mut questions_to_pick = PICK_ALL;
						let mut attrib_iter = attributes.iter();
						if let Some(attr) = attrib_iter.next()
						{
							if attr.name.local_name == PICK_ATTRIBUTE_NAME
							{
								match attr.value.parse::<i32>()
								{
									Ok(pick_val) if pick_val >= 0 => questions_to_pick = pick_val,
									_ => return Err(Box::new(QuestionBankError::InvalidPickValue{received: attr.value.clone()}))
								}
								if let Some(attr) = attrib_iter.next()
								{
									return Err(Box::new(QuestionBankError::UnexpectedAttribute
									{
										expected: "".to_string(),
										received: attr.name.local_name.clone()
									}));
								}
							}
							else
							{
								return Err(Box::new(QuestionBankError::UnexpectedAttribute
								{
									expected: PICK_ATTRIBUTE_NAME.to_string(),
									received: attr.name.local_name.clone()
								}));
							}
						}
						question_groups.push(QuestionGroup::from_xml_reader(&mut question_bank_reader, questions_to_pick)?);
					}
					else
					{
						return Err(Box::new(QuestionBankError::UnexpectedTag
						{
							expected: QUESTION_GROUP_TAG_NAME.to_string(),
							received: name.local_name.clone()
						}));
					}
				},
				XmlEvent::EndElement{name} =>
				{
					if name.local_name == QUESTION_BANK_TAG_NAME
					{
						break;
					}
					else
					{
						let mut expected = String::from("/");
						expected.push_str(QUESTION_BANK_TAG_NAME);
						return Err(Box::new(QuestionBankError::UnexpectedTag
						{
							expected,
							received: name.local_name.clone()
						}));
					}
				},
				XmlEvent::Characters(text) =>
				{
					let expected = format!("{} or /{}", QUESTION_GROUP_TAG_NAME, QUESTION_BANK_TAG_NAME);
					return Err(Box::new(QuestionBankError::UnexpectedText
					{
						expected,
						received: text
					}))
				},
				_ => return Err(Box::new(QuestionBankError::GenericParseError))
			}
		}
		// Find end of document and make sure there is no junk before it
		loop
		{
			let xml_event = question_bank_reader.next()?;
			match xml_event
			{
				XmlEvent::Whitespace(_) => continue,
				XmlEvent::EndDocument => break,
				XmlEvent::Characters(text) =>
					return Err(Box::new(QuestionBankError::UnexpectedText
					{
						expected: "end of document".to_string(),
						received: text
					})),
				_ => return Err(Box::new(QuestionBankError::GenericParseError))
			}
		}

		Ok(QuestionBank
		{
			question_groups
		})
	}

	pub fn random_variant(&self, rng: &mut Lcg128Xsl64, question_counter: &mut u32) -> String
	{
		let mut variant = String::with_capacity(4096);

		for q_group in &self.question_groups
		{
			variant.push_str(&q_group.randomise(rng, question_counter));
		}

		variant
	}
}
