/*
	VMKS exam generator - A simple program for pseudo-randomly generating different variants of an embedded programming exam
	Copyright (C) 2021 Vladimir Garistov <vl.garistov@gmail.com>

	This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as
	published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along with this program; if not, write to the
	Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

use std::process;
use clap::{Arg, App};
use vmks_exam_generator::Config;

fn main()
{
	// Declare and parse command-line arguments
	let arg_matches = App::new("VMKS Exam Generator")
		.version("0.2.0")
		.author("Vladimir Garistov <vl.garistov@gmail.com>")
		.about("A simple program for pseudo-randomly generating different variants of an embedded programming exam.")
		.long_about("A simple CLI program for pseudo-randomly generating different variants of an embedded programming exam.\n\
						   Designed for Technological school \"Electronic Systems\" in Sofia, Bulgaria.\n\n\
						   When a new variant is generated an adjustable subset of questions is picked at random from a question bank.\n\
						   Both questions and possible answers for multiple choice questions are shuffled.\n\
						   Supports randomly picking parts of the question text from a predefined list of possible values.\n\
						   This makes it possible to generate slightly different tasks such as\n\
						   \"Write a program that prints asterisks in the shape of a square\"\n\
						   and\n\
						   \"Write a program that prints hashtags in the shape of a triangle\".")
		.arg(Arg::with_name("variants")
			.short("n")
			.long("variants")
			.value_name("VARIANTS")
			.help("Number of variants to generate")
			.takes_value(true)
			.multiple(false)
			.required(false))
		.arg(Arg::with_name("question-bank")
			.short("q")
			.long("question-bank")
			.value_name("FILENAME")
			.help("XML file specifying the questions")
			.takes_value(true)
			.multiple(false)
			.required(false))
		.arg(Arg::with_name("num-seed")
			.short("s")
			.long("num-seed")
			.value_name("SEED")
			.help("Use the number SEED to seed the random number generator")
			.takes_value(true)
			.multiple(false)
			.required(false))
		.arg(Arg::with_name("text-seed")
			.short("S")
			.long("text-seed")
			.value_name("SEED")
			.help("Use the hashsum of the string SEED to seed the random number generator")
			.takes_value(true)
			.multiple(false)
			.required(false)
			.conflicts_with("num-seed"))
		.after_help("If no seed is provided for the random number generator then the number of seconds since the \
						  start of the UNIX epoch is used. The default number of variants to generate is 30 and the \
						  default name for the question bank file is \"questions.xml\".\n\
						  Visit https://gitlab.com/vl_garistov/vmks_exam_generator for an explanation of the question \
						  bank file format.")
		.get_matches();

	// Construct configuration values from command line arguments
	process::exit(match Config::new(arg_matches)
	{
		// Run the main part of the application
		Ok(config) => match vmks_exam_generator::run(config)
		{
			Ok(()) => 0,
			Err(err) =>
			{
				eprintln!("Application error: {}", err);
				2
			}
		},
		Err(err) =>
		{
			eprintln!("Problem parsing arguments: {}", err);
			1
		}
	});
}
