/*
	VMKS exam generator - A simple program for pseudo-randomly generating different variants of an embedded programming exam
	Copyright (C) 2021 Vladimir Garistov <vl.garistov@gmail.com>

	This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as
	published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along with this program; if not, write to the
	Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

mod questions;

use std::error::Error;
use std::time::SystemTime;
use std::fs::File;
use std::io::Write;
use std::collections::hash_map::DefaultHasher;
use std::hash::{Hash, Hasher};
use clap::ArgMatches;
use rand::prelude::*;
use rand_pcg::Lcg128Xsl64;
use questions::QuestionBank;

pub struct Config
{
	pub seed: u64,
	pub num_variants: u32,
	pub question_bank_filename: String
}

impl Config
{
	pub fn new(arguments: ArgMatches) -> Result<Config, &'static str>
	{
		let mut num_variants: u32 = 30;
		let question_bank_filename = String::from(arguments.value_of("question-bank").unwrap_or("question.xml"));
		let mut seed: u64 = match SystemTime::now().duration_since(SystemTime::UNIX_EPOCH)
		{
			Ok(current_unix_time) => current_unix_time.as_secs(),
			// In case the system clock is set to a point in time before the beginning of the UNIX epoch
			Err(system_time_error) => system_time_error.duration().as_secs()
		};

		if let Some(text_seed) = arguments.value_of("text-seed")
		{
			let mut hasher = DefaultHasher::new();
			text_seed.hash(&mut hasher);
			seed = hasher.finish();
		}

		if let Some(num_seed_str) = arguments.value_of("num-seed")
		{
			match num_seed_str.parse::<u64>()
			{
				Ok(num_seed) => seed = num_seed,
				Err(_) => return Err("Invalid numeric seed! Use -S for non-numeric seeds.")
			}
		}

		if let Some(variants) = arguments.value_of("variants")
		{
			num_variants = match variants.parse::<u32>()
			{
				Ok(num) => num,
				Err(_) => return Err("Invalid number of variants!")
			};
			if num_variants == 0
			{
				return Err("Please provide a number of variants greater than 0.");
			}
		}

		Ok(Config
		{
			seed,
			num_variants,
			question_bank_filename
		})
	}
}

pub fn run(config: Config) -> Result<(), Box<dyn Error>>
{
	let mut rng = Lcg128Xsl64::seed_from_u64(config.seed);
	let question_bank = QuestionBank::from_file(&config.question_bank_filename)?;

	for i in 0..config.num_variants
	{
		let mut question_counter: u32 = 0;
		let mut file = File::create(format!("variant_{}.txt", i + 1))?;
		file.write_all(question_bank.random_variant(&mut rng, &mut question_counter).as_bytes())?;
	}

	Ok(())
}
