use phf::{self, phf_map};
use std::slice::from_ref;

/// How a list is delimited.
#[derive(Copy, Clone, Debug)]
pub(crate) struct Config {
    terminate: bool,
    allowed_delims: &'static [u8],
    preferred_delim: u8,
}
impl Config {
    /// List puts delimiter between items.
    const fn between(allowed_delims: &'static [u8], preferred_delim: u8) -> Config {
        Config {
            terminate: false,
            allowed_delims,
            preferred_delim,
        }
    }

    /// List puts delimiter after items.
    const fn after(allowed_delims: &'static [u8], preferred_delim: u8) -> Config {
        Config {
            terminate: true,
            allowed_delims,
            preferred_delim,
        }
    }

    /// Checks if the delimiter can occur at the end of the string.
    pub(crate) fn allow_terminate(&self) -> bool {
        self.terminate
    }

    /// Checks if a byte is an allowed delimiter.
    pub(crate) fn is_delim(&self, byte: u8) -> bool {
        self.allowed_delims.contains(&byte)
    }

    /// Strips the delimiter from the end of a given string, if necessary.
    pub(crate) fn strip_delim<'a>(&self, bytes: &'a [u8]) -> &'a [u8] {
        if self.terminate {
            bytes.split_last().map_or(
                bytes,
                |(last, init)| if self.is_delim(*last) { init } else { bytes },
            )
        } else {
            bytes
        }
    }

    /// Returns the delimiter as a string.
    pub(crate) fn delim(&self) -> &[u8] {
        from_ref(&self.preferred_delim)
    }

    /// Gets the configuration for a given environment variable.
    pub(crate) fn for_var<B: AsRef<[u8]>>(var: B) -> Option<Config> {
        VARS.get(var.as_ref()).copied()
    }
}

const JUST_COLON: &[u8] = &[b':'];
const JUST_SEMICOLON: &[u8] = &[b';'];
const COLON_AND_SEMICOLON: &[u8] = &[b':', b';'];
const COLON_AND_SPACE: &[u8] = &[b':', b' '];
const COLON: u8 = b':';
const SEMICOLON: u8 = b';';

const PATH_CONFIG: Config = if cfg!(windows) {
    Config::between(JUST_SEMICOLON, SEMICOLON)
} else {
    Config::between(JUST_COLON, COLON)
};

/// All known environment variables that are lists to be merged.
static VARS: phf::Map<&'static [u8], Config> = phf_map! {
    b"GOPATH" => Config::between(JUST_COLON, COLON),
    b"LD_LIBRARY_PATH" => Config::between(COLON_AND_SEMICOLON, COLON),
    b"LD_PRELOAD" => Config::between(COLON_AND_SPACE, COLON),
    b"LS_COLORS" => Config::after(JUST_COLON, COLON),
    b"MANPATH" => Config::between(JUST_COLON, COLON),
    b"PATH" => PATH_CONFIG,
    b"PERL5LIB" => Config::between(JUST_COLON, COLON),
    b"PERLLIB" => Config::between(JUST_COLON, COLON),
    b"PYTHONPATH" => Config::between(JUST_COLON, COLON),
};
