//! Loads your environment so you don't have to.
//!
//! This is intended as a portable way to share environments between processes, particularly for
//! loading scripts in `/etc/profile.d`.
//!
//! This crate offers a simple way of serializing an environment and a binary that can both read
//! environment variables from stdin and combine them into stdout.
//!
//! See the [`Viro`] type for more details.
#![feature(doc_cfg, try_trait_v2, type_alias_impl_trait)]

#[cfg(any(doc, test, feature = "sync"))]
use std::io::{self, BufRead as SyncBufRead, Write as SyncWrite};

#[cfg(any(doc, test, feature = "sync-stdio"))]
use std::io::{BufReader as SyncBufReader, BufWriter as SyncBufWriter};

#[cfg(any(doc, test, feature = "async"))]
use {
    futures::stream::{self, TryStreamExt},
    futures_io::{AsyncBufRead, AsyncWrite},
    futures_util::io::{AsyncBufReadExt, AsyncWriteExt},
    std::pin::Pin,
};

#[cfg(any(doc, test, feature = "async-stdio"))]
use futures_util::io::{BufReader as AsyncBufReader, BufWriter as AsyncBufWriter};

use bstr::{BStr, BString};
use either::Either;
use indexmap::{map, set, IndexMap, IndexSet};
use phf::{self, phf_set};
use std::iter::Peekable;
use std::iter::{once, Once};
use std::iter::{Chain, FlatMap};
use std::ops::Try;
use std::{array, option};

/// Variables containing a list of values.
mod vector;

/// Known environment variables that are outliers and must not be counted.
static BAD_VARS: phf::Set<&'static [u8]> = phf_set! {
    b"_",
    b"GPG_TTY",
    b"HOME",
    b"NOTIFY_SOCKET",
    b"PWD",
    b"SHELL",
    b"SHLVL",
    b"USER",
};

/// An environment.
///
/// This uses a hard-coded set of environment variables to determine whether a variable should be a
/// "scalar", having a singular value, or a "vector", having multiple values. It also checks what
/// separators are used for particular variables, and excludes certain variables which are not
/// intended to be passed around.
///
/// For example, `PATH` is a vector which separates its values using semicolons on Windows and
/// colons on other platforms; and the `SHELL` variable is one that should not be included.
///
/// The currently known variables are:
///
/// Key               | Behavior
/// ------------------|----------------------------------------------------------------
/// `_`               | Excluded
/// `GOPATH`          | Separated by colons
/// `GOPATH`          | Separated by colons
/// `GPG_TTY`         | Excluded
/// `HOME`            | Excluded
/// `LD_LIBRARY_PATH` | Separated by colons; may also be separated by semicolons
/// `LD_PRELOAD`      | Separated by colons; may also be separated by spaces
/// `LS_COLORS`       | Terminated by colons
/// `MANPATH`         | Separated by colons
/// `NOTIFY_SOCKET`   | Excluded
/// `PATH`            | Separated by semicolons on windows, colons on other platforms
/// `PERL5LIB`        | Separated by colons
/// `PERLLIB`         | Separated by colons
/// `PWD`             | Excluded
/// `PYTHONPATH`      | Separated by colons
/// `SHELL`           | Excluded
/// `SHLVL`           | Excluded
/// `USER`            | Excluded
#[derive(Default, Clone, Debug)]
pub struct Viro {
    scalars: IndexMap<BString, BString>,
    vectors: IndexMap<BString, (vector::Config, IndexSet<BString>)>,
}
impl Viro {
    /// Makes an empty environment.
    ///
    /// # Examples
    ///
    /// ```
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    ///
    /// let mut ser = env.write_bytes();
    /// assert_eq!(ser, &[]);
    /// ```
    pub fn new() -> Viro {
        Viro::default()
    }

    fn insert_bstring(&mut self, key: BString, mut val: BString) {
        let (items, config) = match vector::Config::for_var(&key) {
            None => {
                return match self.scalars.entry(key) {
                    map::Entry::Occupied(mut entry) => {
                        let entry = entry.get_mut();
                        entry.clear();
                        entry.append(&mut val);
                    }
                    map::Entry::Vacant(entry) => {
                        entry.insert(val);
                    }
                }
            }
            Some(config) => (config.strip_delim(&val), config),
        };

        self.vectors
            .entry(key)
            .or_insert_with(|| (config, IndexSet::new()))
            .1
            .extend(
                items
                    .split(|b| config.is_delim(*b))
                    .filter(|s| !s.iter().all(u8::is_ascii_whitespace))
                    .map(BString::from),
            );
    }

    /// Dumps a key-value pair into the environment.
    ///
    /// This does *not* perform any validation on the keys or values, accepting all possible byte
    /// sequences. If the key contains a `\x3D` byte (equals sign), or the key or value contain a
    /// `\0` (NUL) byte, the resulting serialization will be incorrect.
    ///
    /// # Examples
    ///
    /// ```
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin");
    /// env.insert("PATH", "/sbin:/bin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    ///
    /// let ser = env.write_bytes();
    /// assert_eq!(ser, &b"PATH=/bin:/sbin:/usr/bin\0"[..]);
    /// ```
    pub fn insert<S: AsRef<[u8]> + Into<Vec<u8>>, T: AsRef<[u8]> + Into<Vec<u8>>>(
        &mut self,
        key: S,
        val: T,
    ) {
        if !BAD_VARS.contains(key.as_ref()) {
            self.insert_bstring(BString::from(key.into()), BString::from(val.into()))
        }
    }

    fn remove_bstr(&mut self, key: &BStr) {
        match vector::Config::for_var(&**key) {
            Some(_) => {
                self.vectors.remove(key);
            }
            None => {
                self.scalars.remove(key);
            }
        }
    }

    /// Removes a key-value pair from the environment.
    ///
    /// This completely removes the key from the serialized format.
    ///
    /// # Examples
    ///
    /// ```
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin");
    /// env.remove("PATH");
    /// env.insert("PATH", "/sbin:/bin:/usr/bin");
    ///
    /// let ser = env.write_bytes();
    /// assert_eq!(ser, &b"PATH=/sbin:/bin:/usr/bin\0"[..]);
    /// ```
    pub fn remove<S: AsRef<[u8]>>(&mut self, key: S) {
        self.remove_bstr(key.as_ref().into())
    }

    /// Processes a serialized item.
    ///
    /// This is equivalent to [`insert`][Self::insert] if the item contains an equals sign, and
    /// [`remove`][Self::remove] if the item does not contain an equal sign.
    ///
    /// Like [`insert`][Self::insert], this does *not* check for NUL bytes in the item, and
    /// including them will result in the serialization being incorrect.
    ///
    /// # Examples
    ///
    /// ```
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.process("PATH=/bin:/sbin");
    /// env.process("PATH");
    /// env.process("PATH=/sbin:/bin:/usr/bin");
    ///
    /// let ser = env.write_bytes();
    /// assert_eq!(ser, &b"PATH=/sbin:/bin:/usr/bin\0"[..]);
    /// ```
    pub fn process<S: AsRef<[u8]>>(&mut self, item: S) {
        let item = item.as_ref();
        let last_nonzero = item
            .iter()
            .rposition(|b| *b != b'\0')
            .map(|idx| idx + 1)
            .unwrap_or(0);
        let item = &item[..last_nonzero];
        if item.is_empty() {
            return;
        }
        if let Some(pos) = item.iter().position(|b| *b == b'=') {
            let key = &item[..pos];
            let val = &item[pos + 1..];
            self.insert::<&[u8], &[u8]>(key, val);
        } else {
            self.remove(item);
        }
    }

    /// Reads an environment from a reader.
    ///
    /// The interprets the input as a NUL-delimited list. If an item contains an equal sign,
    /// it is treated as a key-value pair to be included in the environment. If it does not, it is
    /// treated as a variable to be removed.
    ///
    /// # Examples
    ///
    /// ```
    /// # fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// let mut data = &b"PATH=/bin:/sbin\0PATH\0PATH=/sbin:/bin:/usr/bin"[..];
    /// env.read(&mut data)?;
    ///
    /// let ser = env.write_bytes();
    /// assert_eq!(ser, &b"PATH=/sbin:/bin:/usr/bin\0"[..]);
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "sync"))]
    #[doc(cfg(feature = "sync"))]
    pub fn read<R: ?Sized + SyncBufRead>(&mut self, reader: &mut R) -> io::Result<()> {
        let mut buf = Vec::new();
        while reader.read_until(b'\0', &mut buf)? > 0 {
            self.process(&buf);
            buf.clear();
        }
        Ok(())
    }

    /// Async version of [`read`][Self::read].
    ///
    /// # Examples
    ///
    /// ```
    /// # #[async_std::main]
    /// # async fn main() -> std::io::Result<()> {
    /// use std::pin::Pin;
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// let mut data = &b"PATH=/bin:/sbin\0PATH\0PATH=/sbin:/bin:/usr/bin"[..];
    /// env.read_async(Pin::new(&mut data)).await?;
    ///
    /// let ser = env.write_bytes();
    /// assert_eq!(ser, &b"PATH=/sbin:/bin:/usr/bin\0"[..]);
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "async"))]
    #[doc(cfg(feature = "async"))]
    pub async fn read_async<R: ?Sized + AsyncBufRead>(
        &mut self,
        mut reader: Pin<&mut R>,
    ) -> io::Result<()> {
        let mut buf = Vec::new();
        while reader.read_until(b'\0', &mut buf).await? > 0 {
            self.process(&buf);
            buf.clear();
        }
        Ok(())
    }

    /// Reads the environment from a byte slice.
    ///
    /// # Examples
    ///
    /// ```
    /// # fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// let mut data = &b"PATH=/bin:/sbin\0PATH\0PATH=/sbin:/bin:/usr/bin"[..];
    /// env.read_bytes(data);
    ///
    /// let ser = env.write_bytes();
    /// assert_eq!(ser, &b"PATH=/sbin:/bin:/usr/bin\0"[..]);
    /// # Ok(())
    /// # }
    /// ```
    pub fn read_bytes(&mut self, bytes: &[u8]) {
        for pair in bytes.split(|b| *b == b'\0') {
            self.process(pair);
        }
    }

    /// Reads the environment from stdin.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// # fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.read_stdin()?;
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "sync-stdio"))]
    #[doc(cfg(feature = "sync-stdio"))]
    pub fn read_stdin(&mut self) -> io::Result<()> {
        self.read(&mut SyncBufReader::new(std::io::stdin()))
    }

    /// Async version of [`read_stdin`][Self::read_stdin].
    ///
    /// # Examples
    ///
    /// ```no_run
    /// # #[async_std::main]
    /// # async fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.read_stdin_async().await?;
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "async-stdio"))]
    #[doc(cfg(feature = "async-stdio"))]
    pub async fn read_stdin_async(&mut self) -> io::Result<()> {
        self.read_async(Pin::new(&mut AsyncBufReader::new(async_std::io::stdin())))
            .await
    }

    /// Creates an iterator over the serialized byte-slices of an environment.
    ///
    /// This returns an iterator to avoid allocating any new memory; simply concatenate the values
    /// returned by this iterator to get the full serialized version. Sorts the variables
    /// in byte order before serializing.
    ///
    /// The format is exactly the same as the one for [`read`][Self::read], although every item
    /// in the resulting NUL-delimited list will contain an equals sign; variables that would be
    /// removed are simply omitted.
    ///
    /// The actual separation of the output may change in the future, although this is unlikely.
    /// You should only rely on the concatenation of the iterands, and not their values.
    ///
    /// # Examples
    ///
    /// ```
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    /// env.insert("LD_LIBRARY_PATH", "/lib");
    ///
    /// let mut ser = Vec::new();
    /// for slice in env.serialized() {
    ///     ser.extend_from_slice(slice);
    /// }
    /// assert_eq!(ser, &b"LD_LIBRARY_PATH=/lib\0PATH=/bin:/sbin:/usr/bin\0"[..]);
    /// ```
    pub fn serialized(&mut self) -> ViroIter<'_> {
        self.scalars.sort_keys();
        self.vectors.sort_keys();
        ViroIter {
            scalars: self.scalars.iter().peekable(),
            vectors: self.vectors.iter().peekable(),
            curr: None,
        }
    }

    /// Used by [`write`][Self::write].
    #[cfg(any(doc, test, feature = "sync"))]
    #[doc(cfg(feature = "sync"))]
    fn fold_writer<'w, W: ?Sized + SyncWrite>(
        writer: &'w mut W,
        item: &[u8],
    ) -> io::Result<&'w mut W> {
        writer.write_all(&item)?;
        Ok(writer)
    }

    /// Writes the environment to a writer and flushes it.
    ///
    /// # Examples
    ///
    /// ```
    /// # fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    ///
    /// let mut ser = Vec::new();
    /// env.write(&mut ser)?;
    /// assert_eq!(ser, &b"PATH=/bin:/sbin:/usr/bin\0"[..]);
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "sync"))]
    #[doc(cfg(feature = "sync"))]
    pub fn write<W: ?Sized + SyncWrite>(&mut self, writer: &mut W) -> io::Result<()> {
        self.serialized()
            .try_fold(writer, Self::fold_writer)?
            .flush()
    }

    /// Used by [`write_async`][Self::write_async].
    #[cfg(any(doc, test, feature = "async"))]
    #[doc(cfg(feature = "async"))]
    async fn fold_writer_async<'w, W: ?Sized + AsyncWrite>(
        mut writer: Pin<&'w mut W>,
        item: &[u8],
    ) -> io::Result<Pin<&'w mut W>> {
        writer.write_all(&item).await?;
        Ok(writer)
    }

    /// Async version of [`write`][Self::write].
    ///
    /// # Examples
    ///
    /// ```
    /// # #[async_std::main]
    /// # async fn main() -> std::io::Result<()> {
    /// use std::pin::Pin;
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    ///
    /// let mut ser = Vec::new();
    /// env.write_async(Pin::new(&mut ser)).await?;
    /// assert_eq!(ser, &b"PATH=/bin:/sbin:/usr/bin\0"[..]);
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "async"))]
    #[doc(cfg(feature = "async"))]
    pub async fn write_async<'w, W: ?Sized + AsyncWrite>(
        &'w mut self,
        writer: Pin<&'w mut W>,
    ) -> io::Result<()> {
        stream::iter(self.serialized().map(io::Result::Ok))
            .try_fold(writer, Self::fold_writer_async)
            .await?
            .flush()
            .await
    }

    /// Writes the environment to a byte vector.
    ///
    /// # Examples
    ///
    /// ```
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    ///
    /// let mut ser = env.write_bytes();
    /// assert_eq!(ser, &b"PATH=/bin:/sbin:/usr/bin\0"[..]);
    /// ```
    pub fn write_bytes(&mut self) -> Vec<u8> {
        let mut bytes = Vec::new();
        for slice in self.serialized() {
            bytes.extend_from_slice(slice);
        }
        bytes
    }

    /// Writes the environment out to stdout.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// # fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    ///
    /// env.write_stdout()?;
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "sync-stdio"))]
    #[doc(cfg(feature = "sync-stdio"))]
    pub fn write_stdout(&mut self) -> io::Result<()> {
        self.write(&mut SyncBufWriter::new(std::io::stdout()))
    }

    /// Async version of [`write_stdout`][Self::write_stdout].
    ///
    /// # Examples
    ///
    /// ```
    /// # #[async_std::main]
    /// # async fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    ///
    /// env.write_stdout_async().await?;
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "async-stdio"))]
    #[doc(cfg(feature = "async-stdio"))]
    pub async fn write_stdout_async(&mut self) -> io::Result<()> {
        self.write_async(Pin::new(&mut AsyncBufWriter::new(async_std::io::stdout())))
            .await
    }

    /// Writes the environment out to stderr.
    ///
    /// # Examples
    ///
    /// ```no_run
    /// # fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    ///
    /// env.write_stderr()?;
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "sync-stdio"))]
    #[doc(cfg(feature = "sync-stdio"))]
    pub fn write_stderr(&mut self) -> io::Result<()> {
        self.write(&mut SyncBufWriter::new(std::io::stderr()))
    }

    /// Async version of [`write_stderr`][Self::write_stderr].
    ///
    /// # Examples
    ///
    /// ```
    /// # #[async_std::main]
    /// # async fn main() -> std::io::Result<()> {
    /// use viro::Viro;
    ///
    /// let mut env = Viro::new();
    /// env.insert("PATH", "/bin:/sbin:/usr/bin");
    /// env.insert("SHELL", "/bin/bash");
    ///
    /// env.write_stderr_async().await?;
    /// # Ok(())
    /// # }
    /// ```
    #[cfg(any(doc, test, feature = "async-stdio"))]
    #[doc(cfg(feature = "async-stdio"))]
    pub async fn write_stderr_async(&mut self) -> io::Result<()> {
        self.write_async(Pin::new(&mut AsyncBufWriter::new(async_std::io::stderr())))
            .await
    }
}

/// Iterator over the serialized byte-slices of an environment.
pub struct ViroIter<'a> {
    scalars: Peekable<map::Iter<'a, BString, BString>>,
    vectors: Peekable<map::Iter<'a, BString, (vector::Config, IndexSet<BString>)>>,
    curr: Option<ItemIter<'a>>,
}
impl<'a> ViroIter<'a> {
    fn less_than(&mut self) -> Option<bool> {
        match (self.scalars.peek(), self.vectors.peek()) {
            (Some(left), Some(right)) => Some(left.0 < right.0),
            (Some(_), None) => Some(true),
            (None, Some(_)) => Some(false),
            (None, None) => None,
        }
    }
}
impl<'a> Iterator for ViroIter<'a> {
    type Item = &'a [u8];
    fn next(&mut self) -> Option<&'a [u8]> {
        loop {
            if let Some(iter) = self.curr.as_mut() {
                if let Some(item) = iter.next() {
                    return Some(item);
                }
                self.curr = None;
            }
            if self.less_than()? {
                if let Some((key, val)) = self.scalars.next() {
                    self.curr = Some(Either::Left(scalar_iter(key, val)));
                }
            } else if let Some((key, val)) = self.vectors.next() {
                self.curr = Some(Either::Right(vector_iter(key, val)));
            }
        }
    }
    fn for_each<F>(mut self, mut f: F)
    where
        F: FnMut(&'a [u8]),
    {
        if let Some(iter) = self.curr.take() {
            iter.for_each(&mut f);
        }
        while let Some(less_than) = self.less_than() {
            if less_than {
                if let Some((key, val)) = self.scalars.next() {
                    scalar_iter(key, val).for_each(&mut f);
                }
            } else if let Some((key, val)) = self.vectors.next() {
                vector_iter(key, val).for_each(&mut f);
            }
        }
    }
    fn try_for_each<F, R>(&mut self, mut f: F) -> R
    where
        F: FnMut(&'a [u8]) -> R,
        R: Try<Output = ()>,
    {
        if let Some(mut iter) = self.curr.take() {
            iter.try_for_each(&mut f)?;
        }
        while let Some(less_than) = self.less_than() {
            if less_than {
                if let Some((key, val)) = self.scalars.next() {
                    scalar_iter(key, val).try_for_each(&mut f)?;
                }
            } else if let Some((key, val)) = self.vectors.next() {
                vector_iter(key, val).try_for_each(&mut f)?;
            }
        }
        Try::from_output(())
    }
}

type PairIter<'a, I> = Chain<Chain<array::IntoIter<&'a [u8], 2>, I>, Once<&'a [u8]>>;
fn pair_iter<'a, I: Iterator<Item = &'a [u8]>>(key: &'a [u8], val: I) -> PairIter<'a, I> {
    array::IntoIter::new([key, b"="])
        .chain(val)
        .chain(once(b"\0" as &[u8]))
}

type ScalarIter<'a> = PairIter<'a, Once<&'a [u8]>>;
fn scalar_iter<'a>(key: &'a BString, val: &'a BString) -> ScalarIter<'a> {
    pair_iter(&**key, once(&**val))
}

type DelimAfterFn<'a> = impl FnMut(&'a BString) -> array::IntoIter<&'a [u8], 2>;
type DelimAfter<'a> =
    FlatMap<set::Iter<'a, BString>, array::IntoIter<&'a [u8], 2>, DelimAfterFn<'a>>;
fn delim_after<'a>(set: &'a IndexSet<BString>, delim: &'a [u8]) -> DelimAfter<'a> {
    set.iter()
        .flat_map(move |item| array::IntoIter::new([&***item, delim]))
}

type DelimBeforeFn<'a> = impl FnMut(&'a BString) -> array::IntoIter<&'a [u8], 2>;
type DelimBetween<'a> = Chain<
    option::IntoIter<&'a [u8]>,
    FlatMap<set::Iter<'a, BString>, array::IntoIter<&'a [u8], 2>, DelimBeforeFn<'a>>,
>;
fn delim_between<'a>(set: &'a IndexSet<BString>, delim: &'a [u8]) -> DelimBetween<'a> {
    let mut set = set.iter();
    set.next()
        .map(|item| &***item)
        .into_iter()
        .chain(set.flat_map(move |item| array::IntoIter::new([delim, &***item])))
}

type VectorIter<'a> = PairIter<'a, Either<DelimBetween<'a>, DelimAfter<'a>>>;
fn vector_iter<'a>(
    key: &'a BString,
    val: &'a (vector::Config, IndexSet<BString>),
) -> VectorIter<'a> {
    pair_iter(
        key,
        if val.0.allow_terminate() {
            Either::Right(delim_after(&val.1, val.0.delim()))
        } else {
            Either::Left(delim_between(&val.1, val.0.delim()))
        },
    )
}

type ItemIter<'a> = Either<ScalarIter<'a>, VectorIter<'a>>;
