![Maintenance](https://img.shields.io/badge/maintenance-activly--developed-brightgreen.svg)

# vino-codec

The Vino Codec crate contains the serialization and deserialization functions
and structures for communicating in and out of Vino Components.

#### JSON

Serializes to a [serde_json::Value] which can be printed as a JSON string.

##### `json::serialize`

```rust
use vino_codec::{json, Error};
use serde::{Serialize, Deserialize};

pub fn main() -> Result<(), Error> {

  #[derive(Serialize, Deserialize)]
  struct Point {
    x: i32,
    y: i32,
  }

  let point = Point { x: 200, y: 193 };

  let value = json::serialize(&point)?;
  println!("{:?}", value);

  assert_eq!(value, r#"{"x":200,"y":193}"#);
  Ok(())
}
```

##### `json::deserialize`

```rust
use vino_codec::{json, Error};
use serde::{Serialize, Deserialize};

pub fn main() -> Result<(), Error> {
  #[derive(Serialize, Deserialize, Debug, PartialEq)]
  struct Point {
    x: i32,
    y: i32,
  }

  let json = r#"{"x":200,"y":193}"#;

  let instance: Point = json::deserialize(&json)?;

  assert_eq!(instance, Point { x: 200, y: 193 });
  Ok(())
}

```

#### MessagePack

Serializes to a MessagePack [Vec<u8>].

##### `messagepack::serialize`

```rust
use vino_codec::{messagepack, Error};
use serde::{Serialize, Deserialize};

pub fn main() -> Result<(), Error> {
  #[derive(Serialize, Deserialize)]
  struct Point {
    x: i32,
    y: i32,
  }

  let point = Point { x: 200, y: 193 };

  let value = messagepack::serialize(&point)?;
  println!("{:?}", value);

  let expected: Vec<u8> = vec![130, 161, 120, 204, 200, 161, 121, 204, 193];
  assert_eq!(value, expected);
  Ok(())
}
```

##### `messagepack::deserialize`

```rust
use vino_codec::{messagepack, Error};
use serde::{Serialize, Deserialize};

pub fn main() -> Result<(), Error> {
  #[derive(Serialize, Deserialize, Debug, PartialEq)]
  struct Point {
    x: i32,
    y: i32,
  }

  let slice = vec![146, 204, 200, 204, 193];

  let instance: Point = messagepack::deserialize(&slice)?;

  assert_eq!(instance, Point { x: 200, y: 193 });
  Ok(())
}
```

#### Raw

The [raw] module uses [serde_value] as an intermediary format to pass around.

```rust
use vino_codec::{raw, Error};
use serde::{Serialize, Deserialize};

pub fn main() -> Result<(), Error> {
  #[derive(Serialize, Deserialize, Debug, PartialEq)]
  struct Point {
    x: i32,
    y: i32,
  }

  let point = Point { x: 200, y: 193 };

  let value = raw::serialize(&point)?;
  let instance: Point = raw::deserialize(value)?;

  assert_eq!(instance, Point { x: 200, y: 193 });
  Ok(())
}
```


License: BSD-3-Clause
