//! [`cranelift_entity::EntityRef`][ref] implementations for ABI types.
//!
//! [ref]: https://docs.rs/cranelift-entity/latest/cranelift_entity/trait.EntityRef.html

use super::types::{
    BodyHandle, DictionaryHandle, EndpointHandle, PendingRequestHandle, RequestHandle,
    ResponseHandle,
};

/// Macro which implements a 32-bit entity reference for handles generated by Wiggle.
///
/// This is for all intents and purposes, a use-case specific version of the [`entity_impl`][impl]
/// macro provided by [`cranelift-entity`][entity]. For handles generated by a call to
/// [`wiggle::from_witx`][from-witx], we have to implement entity reference trait slightly
/// differently than normal, due to these types having a private constructor. Instead, we use their
/// `From<u32>` trait implementations to convert back and forth from `usize` values.
///
/// [entity]: https://docs.rs/cranelift-entity/latest/cranelift_entity/
/// [from-witx]: https://docs.rs/wiggle/latest/wiggle/macro.from_witx.html
/// [impl]: https://docs.rs/cranelift-entity/latest/cranelift_entity/macro.entity_impl.html
// TODO KTM 2020-06-29: If this ever becomes a maintenance burden, this could be submitted upstream
// as an alternative mode for the `cranelift_entity::entity_impl` macro.
macro_rules! wiggle_entity {
    ($entity:ident) => {
        /// `EntityRef` allows a small integer type to be used as the key to an entity map, such as
        /// `PrimaryMap`, `SecondaryMap`, or `SparseMap`.
        impl cranelift_entity::EntityRef for $entity {
            /// Create a new entity reference from a small integer.
            fn new(index: usize) -> Self {
                debug_assert!(index < (std::u32::MAX as usize));
                (index as u32).into()
            }
            /// Get the index that was used to create this entity reference.
            fn index(self) -> usize {
                let i: u32 = self.into();
                i as usize
            }
        }
    };
}

wiggle_entity!(BodyHandle);
wiggle_entity!(RequestHandle);
wiggle_entity!(ResponseHandle);
wiggle_entity!(EndpointHandle);
wiggle_entity!(PendingRequestHandle);
wiggle_entity!(DictionaryHandle);
