//! A library implementing the *Sender Policy Framework* (SPF) specification,
//! version 1, described in [RFC 7208].
//!
//! # Usage
//!
//! The function [`evaluate_sender`] is the main public API item. It allows
//! checking the MAIL FROM and HELO identities according to specification.
//!
//! The function [`expand_explain_string`] additionally allows performing macro
//! expansion on *explain strings* (macro strings).
//!
//! The above functions both require to be passed a `Lookup` implementation,
//! which functions as a DNS resolver. The [`Lookup`][crate::lookup::Lookup]
//! trait allows you to plug in any DNS resolver yourself, by providing an
//! implementation. See the module [`lookup`][crate::lookup] for more.
//!
//! A `Lookup` implementation for the Tokio-based
//! [`trust_dns_resolver::TokioAsyncResolver`] is available when the Cargo
//! feature `trust-dns-resolver` is enabled. When using viaspf as a dependency,
//! enable Cargo feature `trust-dns-resolver` to use `TokioAsyncResolver` as a
//! `Lookup`.
//!
//! # Cargo features
//!
//! The feature `trust-dns-resolver` can be enabled to make a `Lookup`
//! implementation available for the Trust-DNS resolver.
//!
//! The feature `tokio-timeout` enables timeout logic using the Tokio `time`
//! module. This feature is enabled by default. If you want to use a different
//! async runtime, you can disable this feature and apply a timeout to the
//! evaluation functions yourself. This lets you use this library without using
//! Tokio at all.
//!
//! [RFC 7208]: https://www.rfc-editor.org/rfc/rfc7208
//! [*check_host()*]: https://www.rfc-editor.org/rfc/rfc7208#section-4
//! [`trust_dns_resolver::TokioAsyncResolver`]: https://docs.rs/trust-dns-resolver/0.21.0-alpha.4/trust_dns_resolver/type.TokioAsyncResolver.html

macro_rules! trace {
    ($query:ident, $tracepoint:expr) => {
        if $query.config.capture_trace() {
            $query.trace.add_event($tracepoint);
        }
    };
}

mod config;
mod eval;
pub mod lookup;
mod params;
mod result;
pub mod trace;

pub use crate::{
    config::{Config, ConfigBuilder},
    eval::{evaluate_sender, expand_explain_string, EvalError},
    params::{DomainName, ParseParamError, Sender},
    result::{
        ErrorCause, ExpansionResult, ExplanationString, QueryResult, SpfResult, SpfResultCause,
    },
};

// The implementation includes quotations (marked with ‘§’) from RFC 7208, to
// explain inline certain implementation choices. The RFC document can be found
// at https://www.rfc-editor.org/rfc/rfc7208.
