//! A library implementing the *Sender Policy Framework* (SPF) protocol, version
//! 1, as specified in [RFC 7208].
//!
//! # Usage
//!
//! The function [`viaspf::evaluate_spf`][`evaluate_spf`] is the main public API
//! item. This function corresponds to the [*check_host()*] abstraction of RFC
//! 7208. The function signature differs from the one given in the
//! specification, but it produces results that are semantically equivalent and
//! thus fulfil the requirements for a compliant SPF implementation.
//!
//! `evaluate_spf` takes five arguments: the first two arguments control
//! properties of protocol handling, the remaining arguments cover the inputs of
//! the *check_host()* function. The result type conveys the SPF result. Refer
//! to the API documentation for the complete reference.
//!
//! The function’s first argument, `lookup`, is a reference of a type that
//! implements [`viaspf::Lookup`][`Lookup`]. This implementation must be
//! provided by API consumers; it functions as the core DNS resolution facility
//! through which all DNS queries are performed. See the `Lookup` documentation
//! for implementation requirements.
//!
//! A `Lookup` implementation for the Tokio-based
//! [`trust_dns_resolver::AsyncResolver`] is available when the feature
//! `trust-dns-resolver` is enabled.
//!
//! Given some implementation of `Lookup`, authorising a host is
//! straightforward:
//!
//! ```
//! # use std::net::IpAddr;
//! # use viaspf::*;
//! # async fn f(resolver: impl Lookup + Send + Sync) {
//! let config = Default::default();
//!
//! let result = evaluate_spf(
//!     &resolver,
//!     &config,
//!     IpAddr::from([9, 9, 9, 9]),
//!     "amy@example.org",
//!     "mail.example.org",
//! )
//! .await;
//!
//! println!("spf={}", result.result);
//! # }
//! ```
//!
//! [RFC 7208]: https://www.rfc-editor.org/rfc/rfc7208
//! [`Lookup`]: trait.Lookup.html
//! [`evaluate_spf`]: fn.evaluate_spf.html
//! [*check_host()*]: https://www.rfc-editor.org/rfc/rfc7208#section-4
//! [`trust_dns_resolver::AsyncResolver`]: https://crates.io/crates/trust-dns-resolver

macro_rules! trace {
    ($query:ident, $tracepoint:expr) => {
        if $query.config.capture_trace() {
            $query.trace.add_event($tracepoint);
        }
    };
}

mod config;
mod eval;
mod name;
pub mod record;
mod resolve;
pub mod trace;

pub use crate::{
    config::*,
    eval::*,
    name::{Name, ParseNameError},
    resolve::*,
};

// The implementation includes quotations (marked with ‘§’) from RFC 7208, to
// explain inline certain implementation choices. The RFC document can be found
// at https://www.rfc-editor.org/rfc/rfc7208.
