use std::{env, process};
use trust_dns_resolver::AsyncResolver;
use viaspf::{Config, SpfResultCause};

#[tokio::main]
async fn main() {
    let mut args = env::args();

    let (ip, domain) = match (
        args.next().as_deref(),
        args.next().and_then(|ip| ip.parse().ok()),
        args.next(),
    ) {
        (_, Some(ip), Some(domain)) => (ip, domain),
        (program, ..) => {
            eprintln!("usage: {} <ip> <domain>", program.unwrap_or("spfquery"));
            process::exit(1);
        }
    };

    let resolver = AsyncResolver::tokio(Default::default(), Default::default()).unwrap();
    let config = Config::builder().capture_trace(true).build();

    let result = viaspf::evaluate_spf(&resolver, &config, ip, &domain, &domain).await;

    println!("IP: {}", ip);
    println!("Domain: {}", domain);
    println!("SPF result: {}", result.result);

    if let Some(cause) = result.cause {
        match cause {
            SpfResultCause::Match(mechanism) => println!("Mechanism: {}", mechanism),
            SpfResultCause::Error(error) => println!("Problem: {}", error),
        }
    }

    if let Some(trace) = result.trace {
        println!("Trace:");
        for event in trace {
            println!("  {}", event.tracepoint);
        }
    }
}
