use super::{Lookup, LookupError, LookupResult, Name};
use std::{
    error::Error,
    net::{IpAddr, Ipv4Addr, Ipv6Addr},
};
use trust_dns_resolver::{
    error::{ResolveError, ResolveErrorKind},
    Name as TrustDnsName, Resolver,
};

impl Lookup for Resolver {
    fn lookup_a(&self, name: &Name) -> LookupResult<Vec<Ipv4Addr>> {
        Ok(self
            .ipv4_lookup(to_trust_dns_name(name)?)
            .map_err(to_lookup_error)?
            .into_iter()
            .collect())
    }

    fn lookup_aaaa(&self, name: &Name) -> LookupResult<Vec<Ipv6Addr>> {
        Ok(self
            .ipv6_lookup(to_trust_dns_name(name)?)
            .map_err(to_lookup_error)?
            .into_iter()
            .collect())
    }

    fn lookup_mx(&self, name: &Name) -> LookupResult<Vec<Name>> {
        let mut mxs = self
            .mx_lookup(to_trust_dns_name(name)?)
            .map_err(to_lookup_error)?
            .into_iter()
            .collect::<Vec<_>>();
        mxs.sort_by_key(|mx| mx.preference());
        mxs.into_iter()
            .map(|mx| Name::new(&mx.exchange().to_ascii()).map_err(wrap_error))
            .collect()
    }

    fn lookup_txt(&self, name: &Name) -> LookupResult<Vec<String>> {
        Ok(self
            .txt_lookup(to_trust_dns_name(name)?)
            .map_err(to_lookup_error)?
            .into_iter()
            .map(|txt| {
                txt.iter()
                    .map(|data| String::from_utf8_lossy(data))
                    .collect()
            })
            .collect())
    }

    fn lookup_ptr(&self, ip: IpAddr) -> LookupResult<Vec<Name>> {
        self.reverse_lookup(ip)
            .map_err(to_lookup_error)?
            .into_iter()
            .map(|name| Name::new(&name.to_ascii()).map_err(wrap_error))
            .collect()
    }
}

fn to_trust_dns_name(name: &Name) -> LookupResult<TrustDnsName> {
    TrustDnsName::from_ascii(name).map_err(wrap_error)
}

fn to_lookup_error(error: ResolveError) -> LookupError {
    match error.kind() {
        ResolveErrorKind::NoRecordsFound { .. } => LookupError::NoRecords,
        ResolveErrorKind::Timeout => LookupError::Timeout,
        _ => wrap_error(error),
    }
}

fn wrap_error(error: impl Error + Send + Sync + 'static) -> LookupError {
    LookupError::Dns(Some(error.into()))
}
