//
// SPDX-License-Identifier: BSD-3-Clause
// Copyright Contributors to the OpenEXR Project.
//

//
// 3-channel and 4-channel color representations
//

#ifndef INCLUDED_IMATHCOLOR_H
#define INCLUDED_IMATHCOLOR_H

#include "ImathExport.h"
#include "ImathNamespace.h"

#include "ImathVec.h"
#include "half.h"

IMATH_INTERNAL_NAMESPACE_HEADER_ENTER

///
/// 3-channel color class that inherits from Vec3.
///
/// This class does not impose interpretation on the channels, which
/// can represent either rgb or hsv color values.
///
/// Note: because Color3 inherits from Vec3, its member fields are
/// called `x`, `y`, and `z`.

template <class T> class IMATH_EXPORT_TEMPLATE_TYPE Color3 : public Vec3<T>
{
  public:

    /// @{
    /// @name Constructors and Assignemt

    /// No initialization by default
    IMATH_HOSTDEVICE Color3() noexcept;                         

    /// Initialize to (a a a)
    IMATH_HOSTDEVICE constexpr explicit Color3 (T a) noexcept;  

    /// Initialize to (a b c)
    IMATH_HOSTDEVICE constexpr Color3 (T a, T b, T c) noexcept; 

    /// Construct from Color3
    IMATH_HOSTDEVICE constexpr Color3 (const Color3& c) noexcept; 

    /// Construct from Vec3
    template <class S> IMATH_HOSTDEVICE constexpr Color3 (const Vec3<S>& v) noexcept; 

    /// Destructor
    ~Color3() = default;

    /// Component-wise assignment
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color3& operator= (const Color3& c) noexcept; 

    /// @}
    
    /// @{
    /// @name Arithmetic
    
    /// Component-wise addition
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color3& operator+= (const Color3& c) noexcept; 

    /// Component-wise addition
    IMATH_HOSTDEVICE constexpr Color3 operator+ (const Color3& c) const noexcept;  

    /// Component-wise subtraction
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color3& operator-= (const Color3& c) noexcept; 

    /// Component-wise subtraction
    IMATH_HOSTDEVICE constexpr Color3 operator- (const Color3& c) const noexcept; 

    /// Component-wise multiplication by -1
    IMATH_HOSTDEVICE constexpr Color3 operator-() const noexcept; 

    /// Component-wise multiplication by -1
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color3& negate() noexcept; 

    /// Component-wise multiplication
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color3& operator*= (const Color3& c) noexcept; 

    /// Component-wise multiplication
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color3& operator*= (T a) noexcept;  

    /// Component-wise multiplication
    IMATH_HOSTDEVICE constexpr Color3 operator* (const Color3& c) const noexcept;  

    /// Component-wise multiplication
    IMATH_HOSTDEVICE constexpr Color3 operator* (T a) const noexcept;  

    /// Component-wise division
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color3& operator/= (const Color3& c) noexcept; 

    /// Component-wise division
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color3& operator/= (T a) noexcept; 

    /// Component-wise division
    IMATH_HOSTDEVICE constexpr Color3 operator/ (const Color3& c) const noexcept;  

    /// Component-wise division
    IMATH_HOSTDEVICE constexpr Color3 operator/ (T a) const noexcept;  

    /// @}
};

///
/// A 4-channel color class: 3 channels plus alpha. 
///
/// For convenience, the fields are named `r`, `g`, and `b`, although
/// this class does not impose interpretation on the channels, which
/// can represent either rgb or hsv color values.
///

template <class T> class IMATH_EXPORT_TEMPLATE_TYPE Color4
{
  public:

    /// @{
    /// @name Direct access to elements

    T r, g, b, a;

    /// @}

    /// @{
    /// @name Constructors and Assignment

    /// No initialization by default
    IMATH_HOSTDEVICE Color4() noexcept;                                      

    /// Initialize to `(a a a a)`
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 explicit Color4 (T a) noexcept;       

    /// Initialize to `(a b c d)`
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 Color4 (T a, T b, T c, T d) noexcept; 

    /// Construct from Color4
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 Color4 (const Color4& v) noexcept; 

    /// Construct from Color4
    template <class S> IMATH_HOSTDEVICE IMATH_CONSTEXPR14 Color4 (const Color4<S>& v) noexcept; 

    /// Destructor
    ~Color4() = default;

    /// Assignment
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color4& operator= (const Color4& v) noexcept; 

    /// Component-wise value
    IMATH_HOSTDEVICE T& operator[] (int i) noexcept; 

    /// Component-wise value
    IMATH_HOSTDEVICE const T& operator[] (int i) const noexcept; 

    /// @}
    
    /// @{
    /// @name Arithmetic and Comparison
    
    /// Equality
    template <class S> IMATH_HOSTDEVICE constexpr bool operator== (const Color4<S>& v) const noexcept; 

    /// Inequality
    template <class S> IMATH_HOSTDEVICE constexpr bool operator!= (const Color4<S>& v) const noexcept; 

    /// Component-wise addition
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color4& operator+= (const Color4& v) noexcept; 

    /// Component-wise addition
    IMATH_HOSTDEVICE constexpr Color4 operator+ (const Color4& v) const noexcept; 

    /// Component-wise subtraction
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color4& operator-= (const Color4& v) noexcept; 

    /// Component-wise subtraction
    IMATH_HOSTDEVICE constexpr Color4 operator- (const Color4& v) const noexcept; 

    /// Component-wise multiplication by -1
    IMATH_HOSTDEVICE constexpr Color4 operator-() const noexcept; 

    /// Component-wise multiplication by -1
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color4& negate() noexcept; 

    /// Component-wise multiplication
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color4& operator*= (const Color4& v) noexcept; 

    /// Component-wise multiplication
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color4& operator*= (T a) noexcept; 

    /// Component-wise multiplication
    IMATH_HOSTDEVICE constexpr Color4 operator* (const Color4& v) const noexcept; 

    /// Component-wise multiplication
    IMATH_HOSTDEVICE constexpr Color4 operator* (T a) const noexcept; 

    /// Component-wise division
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color4& operator/= (const Color4& v) noexcept; 

    /// Component-wise division
    IMATH_HOSTDEVICE IMATH_CONSTEXPR14 const Color4& operator/= (T a) noexcept; 

    /// Component-wise division
    IMATH_HOSTDEVICE constexpr Color4 operator/ (const Color4& v) const noexcept; 

    /// Component-wise division
    IMATH_HOSTDEVICE constexpr Color4 operator/ (T a) const noexcept; 

    /// @}

    /// @{
    /// @name Numeric Limits
    
    /// Number of dimensions (channels), i.e. 4 for a Color4
    IMATH_HOSTDEVICE constexpr static unsigned int dimensions() noexcept { return 4; }

    /// Largest possible negative value
    IMATH_HOSTDEVICE constexpr static T baseTypeLowest() noexcept { return std::numeric_limits<T>::lowest(); }

    /// Largest possible positive value
    IMATH_HOSTDEVICE constexpr static T baseTypeMax() noexcept { return std::numeric_limits<T>::max(); }

    /// Smallest possible positive value
    IMATH_HOSTDEVICE constexpr static T baseTypeSmallest() noexcept { return std::numeric_limits<T>::min(); }

    /// Smallest possible e for which 1+e != 1
    IMATH_HOSTDEVICE constexpr static T baseTypeEpsilon() noexcept { return std::numeric_limits<T>::epsilon(); }

    /// @}
    
    /// The base type: In templates that accept a parameter `V` (could
    /// be a Color4), you can refer to `T` as `V::BaseType`
    typedef T BaseType;

    /// @{
    /// @name Compatibilty with Sb

    /// Set the value
    template <class S> IMATH_HOSTDEVICE void setValue (S a, S b, S c, S d) noexcept; 

    /// Set the value
    template <class S> IMATH_HOSTDEVICE void setValue (const Color4<S>& v) noexcept; 

    /// Return the value
    template <class S> IMATH_HOSTDEVICE void getValue (S& a, S& b, S& c, S& d) const noexcept; 

    /// Return the value
    template <class S> IMATH_HOSTDEVICE void getValue (Color4<S>& v) const noexcept; 

    /// Return raw pointer to the value
    IMATH_HOSTDEVICE T* getValue() noexcept; 

    /// Return raw pointer to the value
    IMATH_HOSTDEVICE const T* getValue() const noexcept; 

    /// @}
};

/// std::ostream output
template <class T> std::ostream& operator<< (std::ostream& s, const Color4<T>& v);

/// Reverse multiplication: S * Color4
template <class S, class T> constexpr Color4<T> operator* (S a, const Color4<T>& v) noexcept;

/// 3 float channels
typedef Color3<float> Color3f;

/// 3 half channels
typedef Color3<half> Color3h;

/// 3 8-bit integer channels
typedef Color3<unsigned char> Color3c;

/// 3 half channels
typedef Color3<half> C3h;

/// 3 float channels
typedef Color3<float> C3f;

/// 3 8-bit integer channels
typedef Color3<unsigned char> C3c;

/// 4 float channels
typedef Color4<float> Color4f;

/// 4 half channels
typedef Color4<half> Color4h;

/// 4 8-bit integer channels
typedef Color4<unsigned char> Color4c;

/// 4 float channels
typedef Color4<float> C4f;

/// 4 half channels
typedef Color4<half> C4h;

/// 4 8-bit integer channels
typedef Color4<unsigned char> C4c;

/// Packed 32-bit integer
typedef unsigned int PackedColor;

//
// Implementation of Color3
//

template <class T> inline Color3<T>::Color3() noexcept : Vec3<T>()
{
    // empty
}

template <class T> constexpr inline Color3<T>::Color3 (T a) noexcept : Vec3<T> (a)
{
    // empty
}

template <class T> constexpr inline Color3<T>::Color3 (T a, T b, T c) noexcept : Vec3<T> (a, b, c)
{
    // empty
}

template <class T> constexpr inline Color3<T>::Color3 (const Color3& c) noexcept : Vec3<T> (c)
{
    // empty
}

template <class T>
template <class S>
constexpr inline Color3<T>::Color3 (const Vec3<S>& v) noexcept : Vec3<T> (v)
{
    //empty
}

template <class T>
IMATH_CONSTEXPR14 inline const Color3<T>&
Color3<T>::operator= (const Color3& c) noexcept
{
    *((Vec3<T>*) this) = c;
    return *this;
}

template <class T>
IMATH_CONSTEXPR14 inline const Color3<T>&
Color3<T>::operator+= (const Color3& c) noexcept
{
    *((Vec3<T>*) this) += c;
    return *this;
}

template <class T>
constexpr inline Color3<T>
Color3<T>::operator+ (const Color3& c) const noexcept
{
    return Color3 (*(Vec3<T>*) this + (const Vec3<T>&) c);
}

template <class T>
IMATH_CONSTEXPR14 inline const Color3<T>&
Color3<T>::operator-= (const Color3& c) noexcept
{
    *((Vec3<T>*) this) -= c;
    return *this;
}

template <class T>
constexpr inline Color3<T>
Color3<T>::operator- (const Color3& c) const noexcept
{
    return Color3 (*(Vec3<T>*) this - (const Vec3<T>&) c);
}

template <class T>
constexpr inline Color3<T>
Color3<T>::operator-() const noexcept
{
    return Color3 (-(*(Vec3<T>*) this));
}

template <class T>
IMATH_CONSTEXPR14 inline const Color3<T>&
Color3<T>::negate() noexcept
{
    ((Vec3<T>*) this)->negate();
    return *this;
}

template <class T>
IMATH_CONSTEXPR14 inline const Color3<T>&
Color3<T>::operator*= (const Color3& c) noexcept
{
    *((Vec3<T>*) this) *= c;
    return *this;
}

template <class T>
IMATH_CONSTEXPR14 inline const Color3<T>&
Color3<T>::operator*= (T a) noexcept
{
    *((Vec3<T>*) this) *= a;
    return *this;
}

template <class T>
constexpr inline Color3<T>
Color3<T>::operator* (const Color3& c) const noexcept
{
    return Color3 (*(Vec3<T>*) this * (const Vec3<T>&) c);
}

template <class T>
constexpr inline Color3<T>
Color3<T>::operator* (T a) const noexcept
{
    return Color3 (*(Vec3<T>*) this * a);
}

template <class T>
IMATH_CONSTEXPR14 inline const Color3<T>&
Color3<T>::operator/= (const Color3& c) noexcept
{
    *((Vec3<T>*) this) /= c;
    return *this;
}

template <class T>
IMATH_CONSTEXPR14 inline const Color3<T>&
Color3<T>::operator/= (T a) noexcept
{
    *((Vec3<T>*) this) /= a;
    return *this;
}

template <class T>
constexpr inline Color3<T>
Color3<T>::operator/ (const Color3& c) const noexcept
{
    return Color3 (*(Vec3<T>*) this / (const Vec3<T>&) c);
}

template <class T>
constexpr inline Color3<T>
Color3<T>::operator/ (T a) const noexcept
{
    return Color3 (*(Vec3<T>*) this / a);
}

//
// Implementation of Color4
//

template <class T>
inline T&
Color4<T>::operator[] (int i) noexcept
{
    return (&r)[i];
}

template <class T>
inline const T&
Color4<T>::operator[] (int i) const noexcept
{
    return (&r)[i];
}

template <class T> inline Color4<T>::Color4() noexcept
{
    // empty
}

template <class T> IMATH_CONSTEXPR14 inline Color4<T>::Color4 (T x) noexcept
{
    r = g = b = a = x;
}

template <class T> IMATH_CONSTEXPR14 inline Color4<T>::Color4 (T x, T y, T z, T w) noexcept
{
    r = x;
    g = y;
    b = z;
    a = w;
}

template <class T> IMATH_CONSTEXPR14 inline Color4<T>::Color4 (const Color4& v) noexcept
{
    r = v.r;
    g = v.g;
    b = v.b;
    a = v.a;
}

template <class T>
template <class S>
IMATH_CONSTEXPR14 inline Color4<T>::Color4 (const Color4<S>& v) noexcept
{
    r = T (v.r);
    g = T (v.g);
    b = T (v.b);
    a = T (v.a);
}

template <class T>
IMATH_CONSTEXPR14 inline const Color4<T>&
Color4<T>::operator= (const Color4& v) noexcept
{
    r = v.r;
    g = v.g;
    b = v.b;
    a = v.a;
    return *this;
}

template <class T>
template <class S>
inline void
Color4<T>::setValue (S x, S y, S z, S w) noexcept
{
    r = T (x);
    g = T (y);
    b = T (z);
    a = T (w);
}

template <class T>
template <class S>
inline void
Color4<T>::setValue (const Color4<S>& v) noexcept
{
    r = T (v.r);
    g = T (v.g);
    b = T (v.b);
    a = T (v.a);
}

template <class T>
template <class S>
inline void
Color4<T>::getValue (S& x, S& y, S& z, S& w) const noexcept
{
    x = S (r);
    y = S (g);
    z = S (b);
    w = S (a);
}

template <class T>
template <class S>
inline void
Color4<T>::getValue (Color4<S>& v) const noexcept
{
    v.r = S (r);
    v.g = S (g);
    v.b = S (b);
    v.a = S (a);
}

template <class T>
inline T*
Color4<T>::getValue() noexcept
{
    return (T*) &r;
}

template <class T>
inline const T*
Color4<T>::getValue() const noexcept
{
    return (const T*) &r;
}

template <class T>
template <class S>
constexpr inline bool
Color4<T>::operator== (const Color4<S>& v) const noexcept
{
    return r == v.r && g == v.g && b == v.b && a == v.a;
}

template <class T>
template <class S>
constexpr inline bool
Color4<T>::operator!= (const Color4<S>& v) const noexcept
{
    return r != v.r || g != v.g || b != v.b || a != v.a;
}

template <class T>
IMATH_CONSTEXPR14 inline const Color4<T>&
Color4<T>::operator+= (const Color4& v) noexcept
{
    r += v.r;
    g += v.g;
    b += v.b;
    a += v.a;
    return *this;
}

template <class T>
constexpr inline Color4<T>
Color4<T>::operator+ (const Color4& v) const noexcept
{
    return Color4 (r + v.r, g + v.g, b + v.b, a + v.a);
}

template <class T>
IMATH_CONSTEXPR14 inline const Color4<T>&
Color4<T>::operator-= (const Color4& v) noexcept
{
    r -= v.r;
    g -= v.g;
    b -= v.b;
    a -= v.a;
    return *this;
}

template <class T>
constexpr inline Color4<T>
Color4<T>::operator- (const Color4& v) const noexcept
{
    return Color4 (r - v.r, g - v.g, b - v.b, a - v.a);
}

template <class T>
constexpr inline Color4<T>
Color4<T>::operator-() const noexcept
{
    return Color4 (-r, -g, -b, -a);
}

template <class T>
IMATH_CONSTEXPR14 inline const Color4<T>&
Color4<T>::negate() noexcept
{
    r = -r;
    g = -g;
    b = -b;
    a = -a;
    return *this;
}

template <class T>
IMATH_CONSTEXPR14 inline const Color4<T>&
Color4<T>::operator*= (const Color4& v) noexcept
{
    r *= v.r;
    g *= v.g;
    b *= v.b;
    a *= v.a;
    return *this;
}

template <class T>
IMATH_CONSTEXPR14 inline const Color4<T>&
Color4<T>::operator*= (T x) noexcept
{
    r *= x;
    g *= x;
    b *= x;
    a *= x;
    return *this;
}

template <class T>
constexpr inline Color4<T>
Color4<T>::operator* (const Color4& v) const noexcept
{
    return Color4 (r * v.r, g * v.g, b * v.b, a * v.a);
}

template <class T>
constexpr inline Color4<T>
Color4<T>::operator* (T x) const noexcept
{
    return Color4 (r * x, g * x, b * x, a * x);
}

template <class T>
IMATH_CONSTEXPR14 inline const Color4<T>&
Color4<T>::operator/= (const Color4& v) noexcept
{
    r /= v.r;
    g /= v.g;
    b /= v.b;
    a /= v.a;
    return *this;
}

template <class T>
IMATH_CONSTEXPR14 inline const Color4<T>&
Color4<T>::operator/= (T x) noexcept
{
    r /= x;
    g /= x;
    b /= x;
    a /= x;
    return *this;
}

template <class T>
constexpr inline Color4<T>
Color4<T>::operator/ (const Color4& v) const noexcept
{
    return Color4 (r / v.r, g / v.g, b / v.b, a / v.a);
}

template <class T>
constexpr inline Color4<T>
Color4<T>::operator/ (T x) const noexcept
{
    return Color4 (r / x, g / x, b / x, a / x);
}

template <class T>
std::ostream&
operator<< (std::ostream& s, const Color4<T>& v)
{
    return s << '(' << v.r << ' ' << v.g << ' ' << v.b << ' ' << v.a << ')';
}

//
// Implementation of reverse multiplication
//

template <class S, class T>
constexpr inline Color4<T>
operator* (S x, const Color4<T>& v) noexcept
{
    return Color4<T> (x * v.r, x * v.g, x * v.b, x * v.a);
}

IMATH_INTERNAL_NAMESPACE_HEADER_EXIT

#endif // INCLUDED_IMATHCOLOR_H
