use anyhow::{anyhow, Result};
use jacklog::{debug, info, instrument, trace};
use serde::{Deserialize, Serialize};
use std::{
    convert::TryFrom,
    fs,
    io::ErrorKind,
    path::{Path, PathBuf},
    process,
};
use structopt::StructOpt;

#[derive(Debug, Deserialize, Serialize, StructOpt, Default)]
#[structopt(setting = structopt::clap::AppSettings::ColoredHelp)]
#[structopt(rename_all = "kebab-case")]
struct Cli {
    /// Path to the repo to version. Defaults to current directory
    #[serde(skip)]
    #[structopt(short, long)]
    path: Option<PathBuf>,

    #[structopt(long)]
    prefix: Option<String>,

    /// Don't take any destructive actions. This won't call any methods on
    /// plugins that are expected to mutate state. Can be useful when combined with
    /// `--verbose`
    #[serde(skip)]
    #[structopt(long, short = "n")]
    dry_run: bool,

    /// List available plugins compiled into this binary
    #[structopt(long)]
    #[serde(skip)]
    plugin_list: bool,

    /// Plugins to explicitly disable, even if the plugin detects that it should
    /// be activated for this project
    #[structopt(long)]
    disable: Vec<String>,

    /// Version a branch other than master. Attempting to version a branch other
    /// than master will result in an error unless this option is used
    #[structopt(long, short)]
    branch: Option<String>,

    /// Enable verbose mode. This just sets the log level to DEBUG. You can fine-tune your
    /// verbosity by setting RUST_LOG= to your desired log level
    #[structopt(long, short, parse(from_occurrences))]
    #[serde(skip)]
    verbose: usize,

    /// Get the current version, but don't increment the version
    #[structopt(long, short)]
    #[serde(skip)]
    read: bool,

    /// Use the version from the specified plugin, ignoring versions from other plugins
    #[structopt(long)]
    read_from: Option<String>,

    /// Use only this plugin to calculate the next version
    #[structopt(long)]
    increment_with: Option<String>,

    /// Save the passed configuration options to verto.toml to be used
    /// automatically on future invocations.
    ///
    /// If the invocation of verto fails, the options won't be written to disk.
    #[structopt(long)]
    #[serde(skip)]
    save: bool,
}

impl TryFrom<&Path> for Cli {
    type Error = anyhow::Error;

    fn try_from(path: &Path) -> Result<Self> {
        let s = match fs::read_to_string(path) {
            Ok(s) => s,
            Err(e) if e.kind() == ErrorKind::NotFound => {
                debug!("no config file; returning defaults");
                return Ok(Self::default());
            }
            Err(e) => return Err(anyhow!(e)),
        };

        let cfg = toml::from_str(&s)?;
        debug!(?cfg, "found config file");

        Ok(cfg)
    }
}

#[instrument]
fn main() -> Result<()> {
    // Parse the input args.
    let opts = Cli::from_args();

    jacklog::from_level(2 + opts.verbose, Some(&["verto"]))?;

    // Determine the path to the target directory, defaulting to current directory
    let path = opts.path.as_ref().map(|p| p.to_owned()).unwrap_or_default();
    let config_path = path.join("verto.toml");

    // Read a config file from the path, if present.
    let mut cfg = Cli::try_from(config_path.as_path())?;
    trace!(
        ?cfg,
        "loaded config (maybe from disk, maybe a default struct)"
    );

    // Merge CLI options into the config file options.
    if opts.prefix.is_some() {
        cfg.prefix = opts.prefix;
    }
    if !opts.disable.is_empty() {
        cfg.disable = opts.disable;
    }
    if opts.branch.is_some() {
        cfg.branch = opts.branch;
    }
    if opts.read_from.is_some() {
        cfg.read_from = opts.read_from;
    }
    if opts.increment_with.is_some() {
        cfg.increment_with = opts.increment_with;
    }

    // Figure out which branch we're acting on.
    let branch = cfg
        .branch
        .as_ref()
        .cloned()
        .or_else(|| Some("master".to_string()));
    debug!(?branch);

    // Parse the prefix to optionally add to the version
    let prefix = cfg
        .prefix
        .as_ref()
        .map(|s| s.to_owned())
        .unwrap_or_default();
    debug!(?prefix);

    // Define closures for adding and removing any desired prefix. TODO: What is this comment
    // about?

    // Set up verto instance
    let mut verto = verto::Verto::new(&path, &prefix, opts.dry_run);

    // If we just want to know what plugins are available, print them and exit
    if opts.plugin_list {
        for p in &verto.plugin_names() {
            println!("{}", p);
        }

        process::exit(0);
    }

    // Initialize all the plugins
    if let Err(e) = verto.initialize(&branch, &cfg.disable) {
        println!("{}", e);

        process::exit(1);
    }
    trace!("done initializing plugins");

    let current_version = match verto.current_version(&cfg.read_from) {
        Ok(v) => v,
        Err(e) => {
            println!("{}", e);
            process::exit(1);
        }
    };
    info!(?current_version);

    // If the read arg is true, just print the version and exit
    if opts.read {
        debug!("read-only mode enabled; not making changes");
        println!("{}", current_version);
        process::exit(0);
    }

    let next_version = verto.next_version(&current_version, &cfg.increment_with)?;
    info!(?next_version);
    let files = verto.write(&next_version);
    verto.commit(&next_version, &files);

    if opts.save {
        info!("saving configuration to verto.toml");
        fs::write(config_path.as_path(), &toml::to_string_pretty(&cfg)?)?;
    }

    // Print the new version to stdout, so if this is running in a script it's easy to capture the
    // output
    println!("{}", &next_version);

    info!("==> done!");

    Ok(())
}
