//! The top level `lib.rs` for the Vermilion virtual machine.

pub mod instruction;
pub mod object;

use instruction::Opcode;
pub use object::VermilionObject;
use std::collections::HashMap;

/// A function used for external symbols.
pub type ExternalFunction = fn(&VirtualMachine);

/// A vermilion 2021 R2 compliant symbol.
#[derive(Clone)]
pub enum VermilionSymbol {

    /// An internal symbol; with a name and a pointer to an address
    /// in memory.
    Internal(String, i64),

    /// An external symbol that has not yet been specified a function.
    UndefinedExternal(String),

    /// An external symbol defined by the pointer.
    External(String, ExternalFunction),

}

// A Vermilion 2021 R2 compilant memory section.
#[derive(Clone)]
pub struct VermilionSection(pub usize, pub usize);

/// A Vermilion 2021 R2 compliant virtual machine.
pub struct VirtualMachine {

    /// The memory accessible to the virtual machine.  Due to restrictions
    /// in the Vermilion 2021 R2 specification, there is a maximum of 4 GB
    /// of memory available.
    pub heap: Vec<u8>,

    /// The stack, as specified in the Vermilion 2021 R2 specification.
    pub stack: Vec<VermilionObject>,

    /// A call stack, used for resolving where a return instruction should
    /// jump to.
    pub call_stack: Vec<i64>,

    /// The list of symbols accessible to the virtual machine.
    pub symbols: HashMap<i64, VermilionSymbol>,

    /// List of allocated memory sections
    pub sections: HashMap<i64, VermilionSection>,

    /// The program counter, the current position in memory that the virtual
    /// machine is reading bytecode from.
    pub pc: usize,

    /// The exit code of the virtual machine.
    pub exit_code: Option<i64>,

}

/// An implementation of the Vermilion 2021 R2 specification.
impl VirtualMachine {
    
    /// Creates a new empty virtual machine.
    pub fn new() -> Self {
        Self {
            heap: vec![],
            stack: vec![],
            call_stack: vec![],
            symbols: HashMap::new(),
            sections: HashMap::new(),
            pc: 0,
            exit_code: None,
        }
    }

    /// Calls a function based on its symbol ID.
    pub fn call_function_id(&mut self, id: i64) {
        // Search through the symbol table to find a function
        // with a machine name.

        let symbols = self.symbols.clone();

        for symbol in symbols {
            // Resolve the function and call it if it was found.

            match symbol.1 {
                VermilionSymbol::Internal(_, addr) => {
                    if id == symbol.0 {
                        self.call_stack.push(self.pc as i64);
                        self.pc = addr.clone() as usize;
                        self.run();
                    }
                },
                VermilionSymbol::External(_, f) => {
                    if id == symbol.0 {
                        f(self);
                    }
                },
                VermilionSymbol::UndefinedExternal(_) => {
                    if id == symbol.0 {
                        panic!("Call of undefined external function.");
                    }
                }
            }
        }
    }

    /// Calls an internal function.
    pub fn call_function(&mut self, name: String) {
        // Search through the symbol table to find a function
        // with a machine name.

        let mut symbols = self.symbols.clone();
        let values = symbols.values_mut();

        for symbol in values {
            // Resolve the function and call it if it was found.

            match symbol {
                VermilionSymbol::Internal(n, addr) => {
                    if n == &name {
                        self.call_stack.push(self.pc as i64);
                        self.pc = addr.clone() as usize;
                        self.run();
                        //self.pc = self.call_stack.pop().unwrap() as usize;
                    }
                },
                VermilionSymbol::External(n, f) => {
                    if n == &name {
                        f(self);
                    }
                },
                VermilionSymbol::UndefinedExternal(n) => {
                    if n == &name {
                        panic!("Call of undefined external function.");
                    }
                }
            }
        }
    }

    /// Returns the opcode of the current byte at the program counter.
    fn get_opcode(&mut self) -> Opcode {
        // Resolve the current byte in the heap
        let opcode = match self.heap[self.pc] {
            0x01 => Opcode::Boolean,
            0x02 => Opcode::Byte,
            0x03 => Opcode::Integer,
            0x04 => Opcode::Float,
            0x05 => Opcode::Pop,
            0x06 => Opcode::Clone,
            0x07 => Opcode::Clear,
            0x0a => Opcode::Trap,
            0x0e => Opcode::BooleanAdd,
            0x0f => Opcode::BooleanSubtract,
            0x10 => Opcode::ByteAdd,
            0x11 => Opcode::ByteSubtract,
            0x12 => Opcode::ByteMultiply,
            0x13 => Opcode::ByteDivide,
            0x14 => Opcode::ByteRemainder,
            0x15 => Opcode::IntegerAdd,
            0x16 => Opcode::IntegerSubtract,
            0x17 => Opcode::IntegerMultiply,
            0x18 => Opcode::IntegerDivide,
            0x19 => Opcode::IntegerRemainder,
            0x1a => Opcode::FloatAdd,
            0x1b => Opcode::FloatSubtract,
            0x1c => Opcode::FloatMultiply,
            0x1d => Opcode::FloatDivide,
            0x1e => Opcode::FloatRemainder,
            0x1f => Opcode::CastBoolean,
            0x20 => Opcode::CastByte,
            0x21 => Opcode::CastInteger,
            0x22 => Opcode::CastFloat,
            0x23 => Opcode::NegateBoolean,
            0x24 => Opcode::NegateInteger,
            0x25 => Opcode::NegateFloat,
            0x26 => Opcode::Load,
            0x27 => Opcode::Store,
            0x28 => Opcode::Realloc,
            0x29 => Opcode::HeapSize,
            0x2a => Opcode::Alloc,
            0x2b => Opcode::ReallocSection,
            0x2c => Opcode::SectionAddr,
            0x2d => Opcode::SectionShiftLeft,
            0x2e => Opcode::SectionShiftRight,
            0x2f => Opcode::Free,
            0x30 => Opcode::FreeAndShift,
            0x31 => Opcode::Branch,
            0x32 => Opcode::BranchIfZero,
            0x33 => Opcode::BranchIfNotZero,
            0x34 => Opcode::Equal,
            0x35 => Opcode::GreaterThan,
            0x36 => Opcode::LessThan,
            0x37 => Opcode::GreaterThanOrEqual,
            0x38 => Opcode::LessThanOrEqual,
            0x39 => Opcode::Call,
            0x3a => Opcode::Return,
            _ => Opcode::Illegal,
        };

        // Move on to the next byte
        self.pc += 1;

        opcode
    }

    /// Returns the next byte as an unsigned 8-bit integer.
    fn next_byte(&mut self) -> u8 {
        // get the next byte
        let val = self.heap[self.pc];

        // move on to the next byte
        self.pc += 1;

        val
    }

    /// Returns the next 8 bytes as a 64 bit integer.
    fn next_64bit_int(&mut self) -> i64 {
        let val = i64::from_le_bytes([
            self.heap[self.pc],
            self.heap[self.pc + 1],
            self.heap[self.pc + 2],
            self.heap[self.pc + 3],
            self.heap[self.pc + 4],
            self.heap[self.pc + 5],
            self.heap[self.pc + 6],
            self.heap[self.pc + 7],
        ]);

        self.pc += 8;

        val
    }

    /// Returns the next 8 bytes as a 64 bit float.
    fn next_64bit_float(&mut self) -> f64 {
        let val = f64::from_le_bytes([
            self.heap[self.pc],
            self.heap[self.pc + 1],
            self.heap[self.pc + 2],
            self.heap[self.pc + 3],
            self.heap[self.pc + 4],
            self.heap[self.pc + 5],
            self.heap[self.pc + 6],
            self.heap[self.pc + 7],
        ]);

        self.pc += 8;

        val
    }

    /// Returns the highest section
    fn get_highest_section(&mut self) -> usize {
        let mut highest = 0;

        let keys = self.sections.keys();

        for key in keys {
            if key > &highest {
                highest = *key;
            }
        }

        highest as usize
    }

    /// Shifts a section to the left.
    fn shift_left(&mut self, sect: i64, amount: i64) {
        let s = &mut self.sections.get_mut(&sect).unwrap();
        let old_end_position = s.0.clone();
        let size = s.1 - s.0;

        let h = self.heap.clone();

        self.heap.drain(old_end_position..old_end_position + amount as usize);

        s.0 = s.0 - amount as usize;
        s.1 = s.0 + size;

        let mut i = 0;
        while i < amount {
            self.heap.insert((s.1 + 1) + i as usize, *h.get(i as usize).unwrap());
            i += 1;
        }
    }

    /// Shifts a section to the right.
    fn shift_right(&mut self, sect: i64, amount: i64) {
        let s = &mut self.sections.get_mut(&sect).unwrap();
        let old_start_position = s.0.clone();
        let size = s.1 - s.0;

        let h = self.heap.clone();

        self.heap.drain(old_start_position - amount as usize..old_start_position);

        s.0 = s.0 + amount as usize;
        s.1 = s.0 + size;

        let mut i = 0;
        while i < amount {
            self.heap.insert(s.0 - 1, *h.get(i as usize).unwrap());
            i += 1;
        }
    }

    /// Runs a single instruction in memory, from the program counter.  Returns
    /// true if the virtual machine should exit.
    fn run_instruction(&mut self) -> bool {
        // Interpret the current instruction

        if self.pc >= self.heap.len() {
            return true;
        }


        let opcode = self.get_opcode();

        match opcode {
            Opcode::Boolean => {
                // Push a boolean to the stack
                let obj = VermilionObject::Boolean(self.next_byte() != 0);
                self.stack.push(obj);
            },
            Opcode::Byte => {
                // Push a byte to the stack
                let obj = VermilionObject::Byte(self.next_byte());
                self.stack.push(obj);
            },
            Opcode::Integer => {
                // Push an integer to the stack
                let obj = VermilionObject::Integer(self.next_64bit_int());
                self.stack.push(obj);
            },
            Opcode::Float => {
                // Push a float to the stack
                let obj = VermilionObject::Float(self.next_64bit_float());
                self.stack.push(obj);
            },
            Opcode::Pop => {
                // remove the topmost item from the stack
                self.stack.pop();
            },
            Opcode::Clone => {
                // clone the topmost item and push the cloned item back onto
                // the stack
                let v = &self.stack[self.stack.len() - 1];
                let v2 = v.clone();
                self.stack.push(v2);
            },
            Opcode::Clear => {
                // clear the stack
                self.stack.clear();
            },
            Opcode::Trap => {
                // exit the application
                let exit_code = self.stack.pop().unwrap();

                match exit_code {
                    VermilionObject::Integer(i) => {
                        self.exit_code = Some(i);
                        return true;
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::BooleanAdd => {
                // add boolean
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l || r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::BooleanSubtract => {
                // subtract boolean
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                if l {
                                    if !r {
                                        self.stack.push(VermilionObject::Boolean(true));
                                    } else {
                                        self.stack.push(VermilionObject::Boolean(false));
                                    }
                                } else {
                                    self.stack.push(VermilionObject::Boolean(false));
                                }
                                //self.stack.push(VermilionObject::Boolean(l && r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::ByteAdd => {
                // add byte
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Byte(l + r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::ByteSubtract => {
                // subtract byte
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Byte(l - r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::ByteMultiply => {
                // multiply byte
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Byte(l * r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::ByteDivide => {
                // divide byte
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Byte(l / r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::ByteRemainder => {
                // get remainder of byte
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Byte(l % r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::IntegerAdd => {
                // add integer
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Integer(l + r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::IntegerSubtract => {
                // subtract integer
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Integer(l - r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::IntegerMultiply => {
                // multiply integer
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Integer(l * r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::IntegerDivide => {
                // divide integer
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Integer(l / r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::IntegerRemainder => {
                // get remainder of integer
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Integer(l % r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::FloatAdd => {
                // add float
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Float(l + r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::FloatSubtract => {
                // subtract float
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Float(l - r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::FloatMultiply => {
                // multiply float
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Float(l * r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::FloatDivide => {
                // divide float
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Float(l / r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::FloatRemainder => {
                // get remainder of float
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Float(l % r));
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::CastBoolean => {
                // cast anything to a boolean
                let to_cast = self.stack.pop().unwrap();

                match to_cast {
                    VermilionObject::Boolean(r) => {
                        self.stack.push(VermilionObject::Boolean(r));
                    },
                    VermilionObject::Byte(r) => {
                        self.stack.push(VermilionObject::Boolean(r != 0));
                    },
                    VermilionObject::Integer(r) => {
                        self.stack.push(VermilionObject::Boolean(r != 0));
                    },
                    VermilionObject::Float(r) => {
                        self.stack.push(VermilionObject::Boolean(r != 0f64));
                    },
                }
            },
            Opcode::CastByte => {
                // cast anything to a byte
                let to_cast = self.stack.pop().unwrap();

                match to_cast {
                    VermilionObject::Boolean(r) => {
                        self.stack.push(VermilionObject::Byte(r as u8));
                    },
                    VermilionObject::Byte(r) => {
                        self.stack.push(VermilionObject::Byte(r));
                    },
                    VermilionObject::Integer(r) => {
                        self.stack.push(VermilionObject::Byte(r as u8));
                    },
                    VermilionObject::Float(r) => {
                        self.stack.push(VermilionObject::Byte(r as u8));
                    },
                }
            },
            Opcode::CastInteger => {
                // cast anything to an integer
                let to_cast = self.stack.pop().unwrap();

                match to_cast {
                    VermilionObject::Boolean(r) => {
                        self.stack.push(VermilionObject::Integer(r as i64));
                    },
                    VermilionObject::Byte(r) => {
                        self.stack.push(VermilionObject::Integer(r as i64));
                    },
                    VermilionObject::Integer(r) => {
                        self.stack.push(VermilionObject::Integer(r));
                    },
                    VermilionObject::Float(r) => {
                        self.stack.push(VermilionObject::Integer(r as i64));
                    },
                }
            },
            Opcode::CastFloat => {
                // cast anything to a float
                let to_cast = self.stack.pop().unwrap();

                match to_cast {
                    VermilionObject::Boolean(r) => {
                        self.stack.push(VermilionObject::Float((r as i64) as f64));
                    },
                    VermilionObject::Byte(r) => {
                        self.stack.push(VermilionObject::Float(r as f64));
                    },
                    VermilionObject::Integer(r) => {
                        self.stack.push(VermilionObject::Float(r as f64));
                    },
                    VermilionObject::Float(r) => {
                        self.stack.push(VermilionObject::Float(r));
                    },
                }
            },
            Opcode::NegateBoolean => {
                // negate boolean
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        self.stack.push(VermilionObject::Boolean(!r));
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::NegateByte => {
                // negate byte
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Byte(r) => {
                        self.stack.push(VermilionObject::Byte(!r));
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::NegateInteger => {
                // negate integer
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        self.stack.push(VermilionObject::Integer(-r));
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::NegateFloat => {
                // negate float
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Float(r) => {
                        self.stack.push(VermilionObject::Float(-r));
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::Load => {
                // load byte from the heap
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        let val = self.heap[r as usize];
                        self.stack.push(VermilionObject::Byte(val));
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::Store => {
                // store a byte in the heap
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                self.heap[l as usize] = r as u8
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::Realloc => {
                // reallocate the heap
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        self.heap.resize(r as usize, 0);
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::HeapSize => {
                // push heap size to stack
                self.stack.push(VermilionObject::Integer(self.heap.len() as i64));
            },
            Opcode::Alloc => {
                // allocate a section
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        let sectid = self.get_highest_section();
                        let sect = &self.sections[&(sectid as i64)];
                        
                        let sect2 = VermilionSection(sect.1 + 1, sect.1 + 1);
                        self.sections.insert(r, sect2);
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::ReallocSection => {
                // reallocate a section
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                let s = self.sections.get_mut(&l).unwrap();
                                let size = s.1 - s.0;
                                let difference = r - size as i64;

                                if difference >= 0 {
                                    // allocate new memory
                                    let mut i = 0;

                                    while i < difference {
                                        self.heap.insert(s.1 + 1, 0);
                                        i += 1;
                                    }
                                    s.1 = s.0 + r as usize;
                                    

                                    // shift other sections
                                    for section in self.sections.iter_mut() {
                                        if section.0 > &l {
                                            section.1.0 += difference as usize;
                                            section.1.1 += difference as usize;
                                        }
                                    }
                                } else {
                                    // free used memory
                                    // allocate new memory

                                    self.heap.drain(difference as usize..s.1);
                                    s.1 = s.0 + r as usize;
                                    

                                    // shift other sections
                                    for section in self.sections.iter_mut() {
                                        if section.0 > &l {
                                            section.1.0 -= difference as usize;
                                            section.1.1 -= difference as usize;
                                        }
                                    }
                                }
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::SectionAddr => {
                // returns a section's start address
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        let item = &self.sections[&r];
                        self.stack.push(VermilionObject::Integer(item.0 as i64))
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::SectionShiftLeft => {
                // shift a section to the left
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                self.shift_left(l, r)
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::SectionShiftRight => {
                // shift a section to the right
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                self.shift_right(l, r)
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::Free => {
                // frees a section
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        let item = &self.sections[&r];
                        self.heap.drain(item.0..item.1);

                        let mut i = 0;
                        let size = item.1 - item.0;
                        while i < size {
                            self.heap.insert(item.0 + i, 0);
                            i += 1;
                        }
                        //self.stack.push(VermilionObject::Integer(item.0 as i64))
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::FreeAndShift => {
                // frees a section and shifts following sections to fill it in
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        let item = &self.sections[&r];
                        self.heap.drain(item.0..item.1);

                        let mut i = 0;
                        let size = item.1 - item.0;
                        while i < size {
                            self.heap.insert(item.0 + i, 0);
                            i += 1;
                        }

                        let iter = self.sections.clone();
                        for item in iter {
                            if item.0 > r {
                                self.shift_left(item.0, size as i64);
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::Branch => {
                // branch
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        self.pc = r as usize;
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::BranchIfZero => {
                // branch if zero
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                if !r {
                                    self.pc = l as usize;
                                }
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::BranchIfNotZero => {
                // branch if not zero
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Integer(l) => {
                                if r {
                                    self.pc = l as usize;
                                }
                            },
                            _ => {
                                // Invalid arguments
                                self.exit_code = Some(255);
                                return true;
                            }
                        }
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::Equal => {
                // equal
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l == (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l == (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l == (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l == (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l == (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l == (r != 0f64)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l == r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l == (r as i64) as f64));
                            },
                        }
                    },
                }
            },
            Opcode::GreaterThan => {
                // greater than
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l as u8 > r as u8));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l > r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l > r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l > (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l > (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l > r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l > r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l > (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l > (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l > r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l > r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l > (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l > (r != 0f64)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l > r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l > r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l > (r as i64) as f64));
                            },
                        }
                    },
                }
            },
            Opcode::LessThan => {
                // less than
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean((l as u8) < r as u8));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l < r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l < r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l < (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l < (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l < r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l < r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l < (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l < (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l < r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l < r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l < (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l < (r != 0f64)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l < r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l < r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l < (r as i64) as f64));
                            },
                        }
                    },
                }
            },
            Opcode::GreaterThanOrEqual => {
                // greater than or equal to
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l as u8 >= r as u8));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= (r != 0f64)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l >= (r as i64) as f64));
                            },
                        }
                    },
                }
            },
            Opcode::LessThanOrEqual => {
                // less than or equal to
                let right = self.stack.pop().unwrap();
                let left = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Boolean(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean((l as u8) <= r as u8));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Byte(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Integer(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= (r != 0)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= (r as i64) as f64));
                            },
                        }
                    },
                    VermilionObject::Float(r) => {
                        match left {
                            VermilionObject::Boolean(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= (r != 0f64)));
                            },
                            VermilionObject::Byte(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= r as u8));
                            },
                            VermilionObject::Integer(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= r as i64));
                            },
                            VermilionObject::Float(l) => {
                                self.stack.push(VermilionObject::Boolean(l <= (r as i64) as f64));
                            },
                        }
                    },
                }
            },
            Opcode::Call => {
                // call function
                let right = self.stack.pop().unwrap();

                match right {
                    VermilionObject::Integer(r) => {
                        self.call_function_id(r);
                    },
                    _ => {
                        // Invalid arguments
                        self.exit_code = Some(255);
                        return true;
                    }
                }
            },
            Opcode::Return => {
                // return from function call
                self.pc = self.call_stack.pop().expect("unable to get call stack item") as usize;
                return true;
            },
            Opcode::Illegal => {
                self.exit_code = Some(255);
                return true;
            },
        };

        false
    }

    /// Runs the instructions in memory, starting at the program counter.
    pub fn run(&mut self) {
        let mut should_exit = false;

        while !should_exit {
            should_exit = self.run_instruction();
        }
    }

}