//! A module for generating Vermilion IR.

use crate::entities::{Block, GlobalData, Symbol, Value};
use crate::instruction::{ InstructionData, Opcode };
use std::collections::HashMap;

/// A module wrapper for initializing Vermilion IR.
pub struct Module {

    /// A list of symbols declared in the module.
    pub symbols: HashMap<String, Symbol>,

    /// A list of Module data.
    pub data: HashMap<String, GlobalData>

}

/// An enum that allows the different types of data.
#[derive(Clone)]
pub enum FunctionValue {

    /// A standard boolean.
    Boolean(bool),

    /// A standard integer.
    Byte(u8),

    /// A standard 64-bit integer.
    Integer(i64),

    /// A standard 64-bit float.
    Float(f64),

    /// A block pointer.
    Block(Block),

    /// An instruction call.
    Instruction(InstructionData),

    /// A reference to a symbol.
    Symbol(String),

}

/// A function for generating Vermilion IR.
#[derive(Clone)]
pub struct Function {

    /// A list of blocks in the function.
    pub blocks: Vec<FunctionBlock>,

    /// A list of values used by the function.
    pub values: Vec<FunctionValue>,

    /// A pointer to the current block.
    pub current_block: u32,

}

/// A block that contains Vermilion IR.
#[derive(Clone)]
pub struct FunctionBlock {

    /// A list of instructions in the block.
    pub instructions: Vec<InstructionData>,

}

impl Module {

    /// Creates a new empty module.
    pub fn new() -> Self {
        Self {
            symbols: HashMap::new(),
            data: HashMap::new(),
        }
    }

    /// Declare an external function
    pub fn declare_function(&mut self, name: String) {
        self.symbols.insert(name, Symbol::External);
    }

    /// Defines a function.
    pub fn define_function(&mut self, name: String, f: Function) {
        self.symbols.insert(name, Symbol::Local(f));
    }

}

/// The functionality behind the Function struct.
impl Function {

    /// Creates a new empty function.
    pub fn new() -> Self {
        Function {
            blocks: vec![],
            values: vec![],
            current_block: 0,
        }
    }

    /// Creates a new code block.  To use this code block, you
    /// must first switch to it.
    pub fn create_block(&mut self) -> Block {
        let b = FunctionBlock {
            instructions: vec![]
        };
        let block = Block(self.blocks.len() as u32);
        self.blocks.push(b);

        block
    }

    /// Switches the current block to the selected block.
    pub fn switch_to_block(&mut self, block: Block) {
        self.current_block = block.0;
    }

    /// Creates a new SSA boolean.  Uses the `Boolean` instruction
    /// behind the scenes.
    pub fn iconst_boolean(&mut self, val: bool) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Boolean(val));

        v
    }

    /// Creates a new SSA integer.  Uses the `Byte` instruction
    /// behind the scenes.
    pub fn iconst_integer(&mut self, val: i64) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Integer(val));

        v
    }

    /// Creates a new SSA float.  Uses the `Float` instruction
    /// behind the scenes.
    pub fn iconst_float(&mut self, val: f64) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Float(val));

        v
    }

    /// Yields the topmost stack item and removes it from the stack.
    pub fn ipop(&mut self) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::Pop,
            args: vec![],
        }));

        v
    }

    /// Removes the topmost stack item.
    pub fn pop(&mut self) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Pop,
            args: vec![],
        });
    }

    /// Clones the current topmost stack item and yields it.
    pub fn iclone(&mut self) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::Pop,
            args: vec![],
        }));

        v
    }

    /// Clones the topmost stack item and pushes it to the stack.
    pub fn clone(&mut self) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Clone,
            args: vec![],
        });
    }

    /// Clears the stack, so it has no items.
    pub fn clear(&mut self) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Clear,
            args: vec![],
        });
    }

    /// Adds two booleans together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn iboolean_add(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::BooleanAdd,
            args: vec![l, r],
        }));

        v
    }

    /// Adds two booleans together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn boolean_add(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::BooleanAdd,
            args: vec![l, r],
        });
    }

    /// Subtracts two booleans together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn iboolean_sub(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::BooleanSubtract,
            args: vec![l, r],
        }));

        v
    }

    /// Subtracts two booleans together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn boolean_sub(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::BooleanSubtract,
            args: vec![l, r],
        });
    }

    /// Adds two bytes together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ibyte_add(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::ByteAdd,
            args: vec![l, r],
        }));

        v
    }

    /// Adds two bytes together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn byte_add(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::ByteAdd,
            args: vec![l, r],
        });
    }

    /// Subtracts two bytes together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ibyte_sub(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::ByteSubtract,
            args: vec![l, r],
        }));

        v
    }

    /// Subtracts two bytes together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn byte_sub(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::ByteSubtract,
            args: vec![l, r],
        });
    }

    /// Multiplies two bytes together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ibyte_mul(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::ByteMultiply,
            args: vec![l, r],
        }));

        v
    }

    /// Multiplies two bytes together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn byte_mul(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::ByteMultiply,
            args: vec![l, r],
        });
    }

    /// Divides two bytes together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ibyte_div(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::ByteDivide,
            args: vec![l, r],
        }));

        v
    }

    /// Divides two bytes together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn byte_div(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::ByteDivide,
            args: vec![l, r],
        });
    }

    /// Divides two bytes together and yields the remainder.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ibyte_mod(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::ByteRemainder,
            args: vec![l, r],
        }));

        v
    }

    /// Divides two integers together and pushes the remainder to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn integer_mod(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::ByteRemainder,
            args: vec![l, r],
        });
    }

    /// Adds two integers together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn iint_add(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::IntegerAdd,
            args: vec![l, r],
        }));

        v
    }

    /// Adds two integers together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn int_add(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::IntegerAdd,
            args: vec![l, r],
        });
    }

    /// Subtracts two integers together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn iint_sub(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::IntegerSubtract,
            args: vec![l, r],
        }));

        v
    }

    /// Subtracts two integers together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn int_sub(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::IntegerSubtract,
            args: vec![l, r],
        });
    }

    /// Multiplies two integers together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn iint_mul(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::IntegerMultiply,
            args: vec![l, r],
        }));

        v
    }

    /// Multiplies two integers together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn int_mul(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::IntegerMultiply,
            args: vec![l, r],
        });
    }

    /// Divides two integers together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn iint_div(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::IntegerDivide,
            args: vec![l, r],
        }));

        v
    }

    /// Divides two integers together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn int_div(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::IntegerDivide,
            args: vec![l, r],
        });
    }

    /// Divides two integers together and yields the remainder.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn iint_mod(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::IntegerRemainder,
            args: vec![l, r],
        }));

        v
    }

    /// Divides two integers together and pushes the remainder to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn int_mod(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::IntegerRemainder,
            args: vec![l, r],
        });
    }

    /// Adds two floats together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ifloat_add(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::FloatAdd,
            args: vec![l, r],
        }));

        v
    }

    /// Adds two floats together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn float_add(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::FloatAdd,
            args: vec![l, r],
        });
    }

    /// Subtracts two integers together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ifloat_sub(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::FloatSubtract,
            args: vec![l, r],
        }));

        v
    }

    /// Subtracts two floats together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn float_sub(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::FloatSubtract,
            args: vec![l, r],
        });
    }

    /// Multiplies two floats together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ifloat_mul(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::FloatMultiply,
            args: vec![l, r],
        }));

        v
    }

    /// Multiplies two floats together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn float_mul(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::FloatMultiply,
            args: vec![l, r],
        });
    }

    /// Divides two floats together and yields the sum.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ifloat_div(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::FloatDivide,
            args: vec![l, r],
        }));

        v
    }

    /// Divides two floats together and pushes the sum to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn float_div(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::FloatDivide,
            args: vec![l, r],
        });
    }

    /// Divides two floats together and yields the remainder.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn ifloat_mod(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::FloatRemainder,
            args: vec![l, r],
        }));

        v
    }

    /// Divides two floats together and pushes the remainder to the stack.
    /// 
    /// # Arguments
    /// - `l`: The left side of the operation.
    /// - `r`: The right side of the operation.
    pub fn float_mod(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::FloatRemainder,
            args: vec![l, r],
        });
    }

    /// Converts a value into a boolean and yields it.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn icast_boolean(&mut self, l: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::CastBoolean,
            args: vec![l],
        }));

        v
    }

    /// Converts a value into a boolean and pushes it to the stack.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn cast_boolean(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::CastBoolean,
            args: vec![l],
        });
    }

    /// Converts a value into a byte and yields it.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn icast_byte(&mut self, l: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::CastByte,
            args: vec![l],
        }));

        v
    }

    /// Converts a value into a byte and pushes it to the stack.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn cast_byte(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::CastByte,
            args: vec![l],
        });
    }

    /// Converts a value into an integer and yields it.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn icast_int(&mut self, l: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::CastInteger,
            args: vec![l],
        }));

        v
    }

    /// Converts a value into an integer and pushes it to the stack.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn cast_int(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::CastInteger,
            args: vec![l],
        });
    }

    /// Converts a value into an float and yields it.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn icast_float(&mut self, l: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::CastFloat,
            args: vec![l],
        }));

        v
    }

    /// Converts a value into an float and pushes it to the stack.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn cast_float(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::CastFloat,
            args: vec![l],
        });
    }

    /// Negates a boolean and yields it.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn ineg_boolean(&mut self, l: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::NegateBoolean,
            args: vec![l],
        }));

        v
    }

    /// Negates a boolean and pushes it to the stack.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn neg_boolean(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::NegateBoolean,
            args: vec![l],
        });
    }

    /// Negates an integer and yields it.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn ineg_int(&mut self, l: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::NegateInteger,
            args: vec![l],
        }));

        v
    }

    /// Negates an integer and pushes it to the stack.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn neg_byte(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::NegateInteger,
            args: vec![l],
        });
    }

    /// Negates a float and yields it.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn ineg_float(&mut self, l: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::NegateFloat,
            args: vec![l],
        }));

        v
    }

    /// Negates a float and pushes it to the stack.
    /// 
    /// # Arguments
    /// - `l`: The value to convert.
    pub fn neg_float(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::NegateFloat,
            args: vec![l],
        });
    }

    /// Loads a byte from memory and yields it.  For efficiency, if
    /// the offset is 0, the offset calculation is not compiled
    /// 
    /// # Arguments
    /// - `l`: The address to load.
    /// - `offset`: The offset to load it from.
    pub fn iload(&mut self, l: Value, offset: i64) -> Value {
        if offset > 0 {
            let off = self.iconst_integer(offset);
            let addr = self.iint_add(l, off);

            let v = self.values.len();
            let val = Value(v as u32);

            self.values.push(FunctionValue::Instruction(InstructionData {
                opcode: Opcode::Load,
                args: vec![addr],
            }));

            val
        } else if offset < 0 {
            let off = self.iconst_integer(offset);
            let addr = self.iint_sub(l, off);

            let v = self.values.len();
            let val = Value(v as u32);

            self.values.push(FunctionValue::Instruction(InstructionData {
                opcode: Opcode::Load,
                args: vec![addr],
            }));

            val
        } else {
            let v = self.values.len();
            let val = Value(v as u32);

            self.values.push(FunctionValue::Instruction(InstructionData {
                opcode: Opcode::Load,
                args: vec![l],
            }));

            val
        }
    }

    /// Loads a byte from memory and pushes it to the stack. 
    /// For efficiency, if the offset is 0, the offset
    /// calculation is not compiled.
    /// 
    /// # Arguments
    /// - `l`: The address to load.
    /// - `offset`: The offset to load it from.
    pub fn load(&mut self, l: Value, offset: i64) {
        if offset > 0 {
            let off = self.iconst_integer(offset);
            let addr = self.iint_add(l, off);

            self.blocks
                .get_mut(self.current_block as usize).unwrap()
                .instructions.push(InstructionData {
                opcode: Opcode::Load,
                args: vec![addr],
            });
        } else if offset < 0 {
            let off = self.iconst_integer(offset);
            let addr = self.iint_sub(l, off);

            self.blocks
                .get_mut(self.current_block as usize).unwrap()
                .instructions.push(InstructionData {
                opcode: Opcode::Load,
                args: vec![addr],
            });
        } else {
            self.blocks
                .get_mut(self.current_block as usize).unwrap()
                .instructions.push(InstructionData {
                opcode: Opcode::Load,
                args: vec![l],
            });
        }
    }

    /// Stores a byte in memory. For efficiency, if the offset
    /// is 0, the offset calculation is not compiled.
    /// 
    /// # Arguments
    /// - `l`: The address to write to.
    /// - `r`: The byte to write.
    /// - `offset`: The offset to load it from.
    pub fn store(&mut self, l: Value, r: Value, offset: i64) {
        if offset > 0 {
            let off = self.iconst_integer(offset);
            let addr = self.iint_add(l, off);

            self.blocks
                .get_mut(self.current_block as usize).unwrap()
                .instructions.push(InstructionData {
                opcode: Opcode::Store,
                args: vec![addr, r],
            });
        } else if offset < 0 {
            let off = self.iconst_integer(offset);
            let addr = self.iint_sub(l, off);

            self.blocks
                .get_mut(self.current_block as usize).unwrap()
                .instructions.push(InstructionData {
                opcode: Opcode::Store,
                args: vec![addr, r],
            });
        } else {
            self.blocks
                .get_mut(self.current_block as usize).unwrap()
                .instructions.push(InstructionData {
                opcode: Opcode::Store,
                args: vec![l, r],
            });
        }
    }

    /// Reallocates the heap to the specified size.
    /// 
    /// # Arguments
    /// - `l`: The new size.
    pub fn realloc(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Realloc,
            args: vec![l],
        });
    }

    /// Yields the amount of allocated heap memory as an integer.
    pub fn iheap_size(&mut self) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::HeapSize,
            args: vec![],
        }));

        v
    }

    /// Pushes the amount of allocated heap memory to the stack.
    pub fn heap_size(&mut self) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::HeapSize,
            args: vec![],
        });
    }

    /// Allocates a section of memory with the specified ID.
    /// 
    /// # Arguments
    /// - `l`: The ID of the new section. (int)
    pub fn alloc(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Alloc,
            args: vec![l],
        });
    }

    /// Reallocates a section of memory with the specified ID.
    /// 
    /// # Arguments
    /// - `l`: The ID of the new section. (int)
    /// - `r`: The new size of the section. (int)
    pub fn realloc_section(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::ReallocSection,
            args: vec![l, r],
        });
    }

    /// Yields the start address of the specified section.
    /// 
    /// # Arguments
    /// - `l`: The section ID to get (int)
    pub fn isection_addr(&mut self, l: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::SectionAddr,
            args: vec![l],
        }));

        v
    }

    /// Pushes the start address of a section to the stack.
    /// 
    /// # Arguments
    /// - `l`: The section ID to get (int)
    pub fn section_addr(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::SectionAddr,
            args: vec![l],
        });
    }

    /// Shifts a section to the left.
    /// 
    /// # Arguments
    /// - `l`: The section ID to shift. (int)
    /// - `r`: The amount of bytes to shift the section by (int)
    pub fn section_shift_left(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::SectionShiftLeft,
            args: vec![l, r],
        });
    }

    /// Shifts a section to the right.
    /// 
    /// # Arguments
    /// - `l`: The section ID to shift. (int)
    /// - `r`: The amount of bytes to shift the section by (int)
    pub fn section_shift_right(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::SectionShiftRight,
            args: vec![l, r],
        });
    }

    /// Frees a section and fills it with bytes.
    /// 
    /// # Arguments
    /// - `l`: The section ID to free. (int)
    pub fn free(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Free,
            args: vec![l],
        });
    }

    /// Frees a section and fills it with bytes, then shifts
    /// all of the following sections to fill the free space.
    /// 
    /// # Arguments
    /// - `l`: The section ID to free. (int)
    pub fn free_and_shift(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::FreeAndShift,
            args: vec![l],
        });
    }

    /// Branches to the specified block.
    /// 
    /// # Arguments
    /// - `l`: The block to branch to.
    pub fn branch(&mut self, l: Block) {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Block(l));

        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Branch,
            args: vec![v],
        });
    }

    /// Branches if the specified boolean is zero.
    /// 
    /// # Arguments
    /// - `l`: The boolean to test
    pub fn brz(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::BranchIfZero,
            args: vec![l],
        });
    }

    /// Branches if the specified boolean is not zero.
    /// 
    /// # Arguments
    /// - `l`: The boolean to test
    pub fn brnz(&mut self, l: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::BranchIfNotZero,
            args: vec![l],
        });
    }

    /// Tests if the two operands are exactly equal.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn iequal(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::Equal,
            args: vec![l, r],
        }));

        v
    }

    /// Branches if the specified boolean is not zero.  Pushes
    /// the result to the stack.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn equal(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Equal,
            args: vec![l, r],
        });
    }

    /// Tests if the first operand is greater than the second.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn igreater(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::GreaterThan,
            args: vec![l, r],
        }));

        v
    }

    /// Tests if the first operand is greater than the second.
    /// Pushes the result to the stack.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn greater(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::GreaterThan,
            args: vec![l, r],
        });
    }

    /// Tests if the first operand is greater than the second.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn iless(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::LessThan,
            args: vec![l, r],
        }));

        v
    }

    /// Tests if the first operand is greater than the second.
    /// Pushes the result to the stack.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn less(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::LessThan,
            args: vec![l, r],
        });
    }

    /// Tests if the first operand is greater or equal to the second.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn igreater_or_equal(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::GreaterThanOrEqual,
            args: vec![l, r],
        }));

        v
    }

    /// Tests if the first operand is greater or equal to the second.
    /// Pushes the result to the stack.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn greater_or_eqal(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::GreaterThanOrEqual,
            args: vec![l, r],
        });
    }

    /// Tests if the first operand is less or equal to the second.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn iless_or_equal(&mut self, l: Value, r: Value) -> Value {
        let id = self.values.len();
        let v = Value(id as u32);

        self.values.push(FunctionValue::Instruction(InstructionData {
            opcode: Opcode::LessThanOrEqual,
            args: vec![l, r],
        }));

        v
    }

    /// Tests if the first operand is less or equal to the second.
    /// Pushes the result to the stack.
    /// 
    /// # Arguments
    /// - `l`: Left side of the operation.
    /// - `r`: Right side of the operation.
    pub fn less_or_equal(&mut self, l: Value, r: Value) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::LessThanOrEqual,
            args: vec![l, r],
        });
    }
    
    /// Calls a function.
    /// 
    /// # Arguments
    /// - `name`: Name of the function to call.
    /// - `args`: The arguments of the function call.
    pub fn call(&mut self, name: String, args: &[Value]) {
        let id = self.values.len();
        let v = Value(id as u32);

        let sym = FunctionValue::Symbol(name);
        self.values.push(sym);
        
        let mut a = args.to_vec();
        a.push(v);
    
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Call,
            args: a,
        });
    }

    /// Returns from a function call.
    /// 
    /// # Arguments
    /// - `args`: The arguments to return.
    pub fn return_(&mut self, args: &[Value]) {
        self.blocks
            .get_mut(self.current_block as usize).unwrap()
            .instructions.push(InstructionData {
            opcode: Opcode::Return,
            args: args.to_vec(),
        });
    }

}