use crate::physics::Collider;
use bevy::prelude::*;

// Some Bug Fixes
#[derive(Clone, Debug)]
pub struct Actor {

    pub label: String,

    pub preset: Option<ActorPreset>,

    pub filename: String,

    pub translation: Vec2,

    pub layer: f32,

    pub rotation: f32,

    pub scale: f32,

    pub collision: bool,

    pub collider: Collider,
}

impl Default for Actor {
    fn default() -> Self {
        Self {
            label: String::default(),
            preset: None,
            filename: String::default(),
            translation: Vec2::default(),
            layer: f32::default(),
            rotation: f32::default(),
            scale: 1.0,
            collision: false,
            collider: Collider::default(),
        }
    }
}

impl Actor {
    pub(crate) fn bevy_transform(&self) -> Transform {
        let mut transform = Transform::from_translation(self.translation.extend(self.layer));
        transform.rotation = Quat::from_axis_angle(Vec3::Z, self.rotation);
        transform.scale = Vec3::splat(self.scale);
        transform
    }
    pub fn set_collision(&mut self, value: bool) -> &mut Self {
        self.collision = value;
        self
    }
    pub fn set_collider(&mut self, collider: Collider) -> &mut Self {
        self.collider = collider;
        self
    }
}

use std::array::IntoIter;


#[derive(Copy, Clone, Debug, PartialEq)]
pub enum ActorPreset {
    BlueBarrel,
    RedBarrel,
    RedBarrier,
    WhiteBarrier,
    BlackCar,
    BlueCar,
    Car,
    RedCar,
    YellowCar,
    Cone,
}

impl ActorPreset {

    pub fn build_from_name(preset_name: String, label: String) -> Actor {
        use ActorPreset::*;
        match preset_name.as_str() {
            "BlueBarrel" => BlueBarrel,
            "RedBarrel" => RedBarrel,
            "RedBarrier" => RedBarrier,
            "WhiteBarrier" => WhiteBarrier,
            "BlackCar" => BlackCar,
            "BlueCar" => BlueCar,
            "Car" => Car,
            "RedCar" => RedCar,
            "YellowCar" => YellowCar,
            "Cone" => Cone,
            _ => panic!(
                "Cannot Find The Asset {}, Try To Put The Asset Somewhere Else, And Check Your Code Again, Did You Put It Where You Need?",
                preset_name
            ),
        }
        .build(label)
    }
    pub fn build(self, label: String) -> Actor {
        let filename = self.filename();
        let collider = self.collider();
        Actor {
            label,
            preset: Some(self),
            filename,
            collider,
            ..Default::default()
        }
    }

    pub fn collider(&self) -> Collider {
        match self {
            ActorPreset::BlueBarrel => Collider::circle(28.0),
            ActorPreset::RedBarrel => Collider::circle(28.0),
            ActorPreset::RedBarrier => {
                Collider::rect(Vec2::new(-105.0, 31.0), Vec2::new(105.0, -31.0))
            }
            ActorPreset::WhiteBarrier => {
                Collider::rect(Vec2::new(-105.0, 31.0), Vec2::new(105.0, -31.0))
            }
            ActorPreset::BlackCar => Collider::poly(&[
                (-59., 28.),
                (-58., 31.),
                (-54., 34.),
                (51., 34.),
                (56., 31.5),
                (58.5, 28.5),
                (58.5, -26.),
                (57.5, -29.5),
                (52.5, -33.5),
                (-54.5, -33.5),
                (-59., -29.),
            ]),
            ActorPreset::BlueCar => Collider::poly(&[
                (-59., 28.),
                (-58., 31.),
                (-54., 34.),
                (51., 34.),
                (56., 31.5),
                (58.5, 28.5),
                (58.5, -26.),
                (57.5, -29.5),
                (52.5, -33.5),
                (-54.5, -33.5),
                (-59., -29.),
            ]),
            ActorPreset::Car => Collider::poly(&[
                (-59., 28.),
                (-58., 31.),
                (-54., 34.),
                (51., 34.),
                (56., 31.5),
                (58.5, 28.5),
                (58.5, -26.),
                (57.5, -29.5),
                (52.5, -33.5),
                (-54.5, -33.5),
                (-59., -29.),
            ]),
            ActorPreset::RedCar => Collider::poly(&[
                (-59., 28.),
                (-58., 31.),
                (-54., 34.),
                (51., 34.),
                (56., 31.5),
                (58.5, 28.5),
                (58.5, -26.),
                (57.5, -29.5),
                (52.5, -33.5),
                (-54.5, -33.5),
                (-59., -29.),
            ]),
            ActorPreset::YellowCar => Collider::poly(&[
                (-59., 28.),
                (-58., 31.),
                (-54., 34.),
                (51., 34.),
                (56., 31.5),
                (58.5, 28.5),
                (58.5, -26.),
                (57.5, -29.5),
                (52.5, -33.5),
                (-54.5, -33.5),
                (-59., -29.),
            ]),
            ActorPreset::Cone => {
                Collider::rect(Vec2::new(-22.0, 22.0), Vec2::new(22.0, -22.0))
            }
        }
    }

    pub fn filename(&self) -> String {
        match self {
            ActorPreset::BlueBarrel => "sprite/BlueBarrel.png",
            ActorPreset::RedBarrel => "sprite/RedBarrel.png",
            ActorPreset::RedBarrier => "sprite/RedBarrier.png",
            ActorPreset::WhiteBarrier => "sprite/WhiteBarrier.png",
            ActorPreset::BlackCar => "sprite/BlackCar.png",
            ActorPreset::BlueCar => "sprite/BlueCar.png",
            ActorPreset::Car => "sprite/Car.png",
            ActorPreset::RedCar => "sprite/RedCar.png",
            ActorPreset::YellowCar => "sprite/YellowCar.png",
            ActorPreset::Cone => "sprite/Cone.png",
        }
        .into()
    }

    pub fn variant_iter() -> IntoIter<ActorPreset, 10> {
        static ACTOR_PRESETS: [ActorPreset; 10] = [
            ActorPreset::BlueBarrel,
            ActorPreset::RedBarrel,
            ActorPreset::RedBarrier,
            ActorPreset::WhiteBarrier,
            ActorPreset::BlackCar,
            ActorPreset::BlueCar,
            ActorPreset::Car,
            ActorPreset::RedCar,
            ActorPreset::YellowCar,
            ActorPreset::Cone,
        ];
        std::array::IntoIter::new(ACTOR_PRESETS)
    }

    fn shifted_by(&self, amount: isize) -> ActorPreset {
        let len = ActorPreset::variant_iter().len();
        let index = ActorPreset::variant_iter()
            .enumerate()
            .find(|(_, a)| *a == *self)
            .unwrap()
            .0;
        let mut new_index_isize = index as isize + amount;
        while new_index_isize < 0 {
            new_index_isize += len as isize;
        }
        let new_index = (new_index_isize as usize) % len;
        ActorPreset::variant_iter().nth(new_index).unwrap()
    }

    pub fn next(&self) -> ActorPreset {
        self.shifted_by(-1)
    }

    pub fn prev(&self) -> ActorPreset {
        self.shifted_by(1)
    }
}