use num_traits::{real::Real, One, Zero};
use std::{
    cmp::PartialOrd,
    ops::{Add, Div, Mul, Neg, Sub},
};

/// This trait specifies some type to be a vector type.
/// It specifies the scalar type and is required for other vector types.
pub trait VectorSpace: Copy + Zero + PartialEq
where
    Self: Add<Output = Self>,
    Self: Sub<Output = Self>,
    Self: Mul<<Self as VectorSpace>::Scalar, Output = Self>,
    Self: Div<<Self as VectorSpace>::Scalar, Output = Self>,
    Self: Neg<Output = Self>,
{
    /// The scalar type of the vector space.
    type Scalar: Real + PartialOrd;

    /// The linear interpolation of two vectors.
    fn lerp(self, other: Self, amount: Self::Scalar) -> Self {
        self * (Self::Scalar::one() - amount) + other * amount
    }
}

impl VectorSpace for f32 {
    type Scalar = f32;
}

impl VectorSpace for f64 {
    type Scalar = f64;
}
