# VecStorage

Re-use the memory for vectors containing values with different lifetimes

In some situations, e.g. when writing real-time audio software, memory needs to be pre-allocated.
The `Vec` data structure from Rust's standard library can be used for this purpose.
In some situations, however, you want to re-use the same `Vec` for data that has different lifetimes
and Rust's type system doesn't allow this.
This crate intends to help you overcome this problem.

## Example
The following code does not compile:
```rust
let mut v = Vec::with_capacity(2);
{
    let x = 1; let y = 2;
    v.push(&x);
    v.push(&y);
    v.clear(); // We stop borrowing here, but the compiler doesn't know that.
}
{
    let a = 1; let b = 2;
    v.push(&a);
    v.push(&b);
    v.clear(); // We stop borrowing here, but the compiler doesn't know that.
}
```
You can use `VecStorage` to solve this problem:
```rust
use vecstorage::VecStorage;
let mut v = VecStorage::<&u32>::with_capacity(2);
{
    let x = 1; let y = 2;
    let mut guard = v.vec_guard();
    // Now guard behaves like a vector.
    guard.push(&x); // No memory allocation here, we use the memory allocated in `v`.
    guard.push(&y);
    // If we were going to push more items on the guard, we would allocate memory.
    // When guard goes out of scope, it is cleared.
}
{
    let a = 1; let b = 2;
    let mut guard = v.vec_guard();
    // Now guard behaves like a vector.
    // The memory from the previous run has been cleared ...
    assert_eq!(guard.len(), 0);
    guard.push(&a);
    guard.push(&b);
}
```

Documentation
-------------
See the [documentation on docs.rs](https://docs.rs/vecstorage).

Related Crates
--------------

[`rsor`](https://crates.io/crates/rsor) (short for "Reusable Slice Of References")
is developed to re-use the memory for containing references and can be used as an
alternative to `vecstorage` if you want to store references.
`rsor` has a different API that does not rely on run-time checks (in contrast to `vecstorage`
which does some runtime checks).


Contributing
------------

We welcome contributions, both in the form of issues and in the form of pull requests.
Before opening a pull request, please open an issue first so that you know whether a subsequent
pull request would likely be approved.
Alternatively, you can contribute via e-mail (an email address is in the commits).

It's best to run the automated tests before sending your contribution.

Testing
-------

Since this crate uses some `unsafe` blocks, we run the tests both "in a normal way" and under `miri`.

Running the tests normally:
```bash
cargo test
```

In order to run the tests under `miri`, you need to 
[have the miri component installed](https://github.com/rust-lang/miri).
Then you can run the tests under `miri` as follows:

```bash
cargo +nightly miri test -- --skip forgetting
export MIRIFLAGS=-Zmiri-ignore-leaks
cargo +nightly miri test forgetting
```


License
-------

Vecstorage is distributed under the terms of the MIT license or the Apache License (Version 2.0), 
at your choice.
For the application of the MIT license, the examples included in the doc comments are not
considered "substatial portions of this Software".
