//! [![Crates.io][crates-badge]][crates-url]
//! [![Docs.rs][docs-badge]][docs-url]
//! [![MIT licensed][mit-badge]][mit-url]
//! [![Build Status][actions-badge]][actions-url]
//!
//! [crates-badge]: https://img.shields.io/crates/v/vecn.svg
//! [crates-url]: https://crates.io/crates/vecn
//! [docs-badge]: https://docs.rs/vecn/badge.svg
//! [docs-url]: https://docs.rs/vecn
//! [mit-badge]: https://img.shields.io/badge/license-MIT-blue.svg
//! [mit-url]: https://github.com/mapkts/vecn/blob/master/LICENSE
//! [actions-badge]: https://github.com/mapkts/vecn/workflows/CI/badge.svg
//! [actions-url]: https://github.com/mapkts/vecn/actions?query=workflow%3ACI+branch%3Amaster
//!
//! Provides the `#[vector]` macro, which transforms user-defined structs into general vector
//! types by conditionally implementing appropriate traits and methods for them.
//!
//! # Examples
//!
//! Vector types can be either named structs or tuple structs, with primitive elements.
//!
//! ```
//! use vecn::vector;
//!
//! #[vector]
//! pub struct Rgb(u8, u8, u8);
//!
//! #[vector]
//! pub struct Point3 {
//!     pub x: f32,
//!     pub y: f32,
//!     pub z: f32,
//! }
//! ```
//!
//! Generic vector types is supported as well, but you need to manually add the
//! [`num_traits`](https://crates.io/crates/num-traits) crate to your `Cargo.toml`,  as we use
//! some num traits to properly bound methods.
//!
//! It's recommend to implement primitive vector types whenever possible, because unlike generic
//! vector types that will cluster a bunch of bounded methods together, methods implemented for
//! primitive vector types are concise and adequate, which means compiler won't allow you to call a
//! meaningless method accidentally.
//!
//! ```
//! use vecn::vector;
//!
//! #[vector]
//! pub struct Vec4<T> {
//!    pub x: T,
//!    pub y: T,
//!    pub z: T,
//!    pub w: T,
//! }
//!
//! let point1 = Vec4::new(2., 2., 2., 1.);
//! let point2 = Vec4::new(4., 2., 2., 1.);
//! let distance = point1.distance(point2);
//! assert_eq!(distance, 2.);
//! ```
//!
//! It's possible to create a vector type with many lanes, though only a small set of methods will
//! be implemented for it.
//!
//! ```
//! use vecn::vector;
//!
//! #[vector]
//! pub struct Vec8<T>(T, T, T, T, T, T, T, T);
//!
//! let v8: Vec8<u8> = (0u8, 1, 2, 3, 4, 5, 6, 7).into();
//! assert_eq!(v8.sum(), 28);
//! assert_eq!(v8.product(), 0);
//! ```
//!
//! # Documentation
//!
//! You can view the traits and methods implemented for the annotated struct using `cargo doc` in
//! your project's workspace, if the annotated struct is public. For situations where the annotated
//! struct cannot be public, you can look at some example vector types that generated by the
//! `#[vector]` macro from [`vectory`].
//!
//! # Additional Notes
//!
//! `#[vector]` will derive [`Copy`], [`Clone`] and [`PartialEq`] traits for the
//! annotated struct for convenience. Many methods rely on these properties to functionally work.
//!
//! Besides, it's worth noting that vecn will only implement a suitable set of traits and methods
//! for the annotated struct. Some traits like [`Display`] should be better implemented by users,
//! and some methods are missing for varying reasons. Hopefully the annotated struct is totally
//! local to user space, you are free to implement whatever methods and traits you like.
//!
//! ```rust
//! use vecn::vector;
//! use std::fmt;
//!
//! #[vector]
//! pub struct Vec3(f32, f32, f32);
//!
//! #[vector]
//! pub struct Vec4(f32, f32, f32, f32);
//!
//! impl Vec4 {
//!     /// Shrinks a `Vec4` to a `Vec3`, with its last lane discarded.
//!     #[inline]
//!     pub fn shrink(self) -> Vec3 {
//!         let (x, y, z, _) = self.into();
//!         Vec3::new(x, y, z)
//!     }
//! }
//!
//! // As we don't know the relations between vector types, `From` trait needed to implement
//! // manually.
//! impl From<Vec3> for Vec4 {
//!     #[inline]
//!     fn from(v3: Vec3) -> Vec4 {
//!         Vec4::new(v3.0, v3.1, v3.2, 0.)
//!     }
//! }
//!
//! impl fmt::Display for Vec3 {
//!     fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
//!         write!(f, "({}, {}, {})", self.0, self.1, self.2)
//!     }
//! }
//!
//! impl fmt::Display for Vec4 {
//!     fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
//!         write!(f, "({}, {}, {}, {})", self.0, self.1, self.2, self.3)
//!     }
//! }
//!
//! ```
//!
//! [`Copy`]: std::marker::Copy
//! [`Clone`]: std::clone::Clone
//! [`PartialEq`]: std::cmp::PartialEq
//! [`Display`]: std::fmt::Display
//! [`vectory`]: https://mapkts.github.io/vecn/vectory/index.html
use core::iter::{self, Repeat, Take};
use proc_macro::TokenStream;
use proc_macro2::{Ident, Span};
use quote::{format_ident, quote};
use syn::parse_macro_input;
use syn::spanned::Spanned;
use syn::{Error, Fields, Generics, Index, ItemStruct, Path, Type, WhereClause};

// Collects compile error messages into a `TokenStream`.
macro_rules! err {
    ($span:expr, $fmt:literal $(,)? $($arg:expr),*) => {{
        TokenStream::from(
            Error::new($span, format!($fmt, $($arg),*)).into_compile_error()
        )
    }}
}

/// Macro that transforms user-defined structs into general vector types.
#[proc_macro_attribute]
pub fn vector(args: TokenStream, input: TokenStream) -> TokenStream {
    let args = parse_macro_input!(args as syn::AttributeArgs);

    // NOTE:
    // By now we have not implemented attribute parsing logic yet.
    // Emits a compile error if any macro attributes are given.
    if let Some(arg) = args.get(0) {
        return err!(arg.span(), "found unexpected argument in macro attribute");
    }

    let item = parse_macro_input!(input as ItemStruct);
    let output = expand(item);
    match output {
        Ok(tokens) => tokens,
        Err(err) => err,
    }
}

// Returns (is_accepted_primitive, is_signed, is_float)
fn check_primitive(path: &Option<Path>) -> (bool, bool, bool) {
    let path = path.as_ref().expect("type path should always be some path");
    if let Some(seg) = path.segments.last() {
        const PRIMITIVE: [&str; 12] = [
            "f32", "f64", "i8", "i16", "i32", "i64", "i128", "u8", "u16", "u32", "u64", "u128",
        ];

        let mut ret = (false, false, false);
        for (i, x) in PRIMITIVE.iter().enumerate() {
            if seg.ident == x {
                ret.0 = true;
                if i < 2 {
                    ret.2 = true;
                }
                if i < 7 {
                    ret.1 = true;
                }
            }
        }
        return ret;
    }
    (false, false, false)
}

fn repeat<T>(x: T, n: usize) -> Take<Repeat<T>>
where
    T: Clone,
{
    iter::repeat(x).take(n)
}

fn expand(item: ItemStruct) -> std::result::Result<TokenStream, TokenStream> {
    let parse_result = parse_struct(&item);

    // If there is any error, just return it immediately.
    if let Some(err) = parse_result.compile_error {
        return Err(err);
    }

    let ParseResult {
        ident,
        generics,
        generic_ident: _,
        where_clause,
        fields,
        fields_count,
        type_path,
        is_tuple,
        tuple_indexes,
        is_generic,
        is_primitive,
        is_signed_primitive,
        is_float,
        ..
    } = parse_result;

    let where_prelude = if where_clause.is_some() {
        quote!(#where_clause)
    } else {
        quote!(where)
    };

    let item_repr_c = {
        let mut item_repr_c = item;

        // Marks the vector struct as #[repr(C)].
        item_repr_c.attrs.push(syn::parse_quote!(#[repr(C)]));

        // Derives necessary traits.
        item_repr_c
            .attrs
            .push(syn::parse_quote!(#[derive(Clone, Copy, PartialEq)]));

        quote!(#item_repr_c)
    };

    let impl_as_ref = {
        quote!(
        // AsRef<[T]>
        impl #generics core::convert::AsRef<[#type_path]> for #ident #generics #where_clause {
            #[inline]
            fn as_ref(&self) -> &[#type_path] {
                self.as_slice()
            }
        }

        // AsRef<Self>
        impl #generics core::convert::AsRef<Self> for #ident #generics #where_clause {
            #[inline]
            fn as_ref(&self) -> &Self {
                self
            }
        }

        // AsMut<[T]>
        impl #generics core::convert::AsMut<[#type_path]> for #ident #generics #where_clause {
            #[inline]
            fn as_mut(&mut self) -> &mut [#type_path] {
                self.as_mut_slice()
            }
        }

        // AsMut<Self>
        impl #generics core::convert::AsMut<Self> for #ident #generics #where_clause {
            #[inline]
            fn as_mut(&mut self) -> &mut Self {
                self
            }
        }
        )
    };

    let impl_de_ref = {
        quote!(
            impl #generics core::ops::Deref for #ident #generics #where_clause {
                type Target = [#type_path];

                #[inline]
                fn deref(&self) -> &Self::Target {
                    self.as_slice()
                }
            }

            impl #generics core::ops::DerefMut for #ident #generics #where_clause {
                #[inline]
                fn deref_mut(&mut self) -> &mut Self::Target {
                    self.as_mut_slice()
                }
            }
        )
    };

    let impl_borrow = {
        quote!(
            impl #generics core::borrow::Borrow<[#type_path]> for #ident #generics #where_clause {
                #[inline]
                fn borrow(&self) -> &[#type_path] {
                    self.as_slice()
                }
            }

            impl #generics core::borrow::BorrowMut<[#type_path]> for #ident #generics #where_clause {
                #[inline]
                fn borrow_mut(&mut self) -> &mut [#type_path] {
                    self.as_mut_slice()
                }
            }
        )
    };

    let impl_from_inner = {
        let where_clause = if is_generic {
            quote!(#where_prelude #type_path: Copy)
        } else {
            quote!(#where_clause)
        };
        let var = Ident::new("inner", Span::call_site());
        let ts = std::iter::repeat(var.clone()).take(fields_count);

        quote!(
            impl #generics core::convert::From<#type_path> for #ident #generics #where_clause {
                #[inline]
                fn from(#var: #type_path) -> Self {
                    #ident::new(#(#ts),*)
                }
            }
        )
    };

    let impl_from_tuple_array = {
        let ts = repeat(&type_path, fields_count);
        let tuple = quote!((#(#ts),*));
        let array = quote!([#type_path; #fields_count]);

        quote!(
            impl #generics core::convert::From<#tuple> for #ident #generics #where_clause {
                #[inline]
                fn from(tuple: #tuple) -> Self {
                    let (#(#fields),*) = tuple;
                    #ident::new(#(#fields),*)
                }
            }

            impl #generics core::convert::From<#array> for #ident #generics #where_clause {
                #[inline]
                fn from(array: #array) -> Self {
                    let [#(#fields),*] = array;
                    #ident::new(#(#fields),*)
                }
            }
        )
    };

    let impl_into_tuple_array = {
        let ts = repeat(&type_path, fields_count);
        let tuple = quote!((#(#ts),*));
        let array = quote!([#type_path; #fields_count]);
        let inner_bracket = if is_tuple {
            quote!(#(self.#tuple_indexes),*)
        } else {
            quote!(#(self.#fields),*)
        };

        // Although `[T; N]: From<Vec>` is equivalent to `Vec: Into<[T; N]>`, we choose the latter
        // form because it't more explicit than the former (`Into` traits will display in rustdoc).
        quote!(
            #[allow(clippy::from_over_into)]
            impl #generics core::convert::Into<#tuple> for #ident #generics #where_clause {
                #[inline]
                fn into(self) -> #tuple {
                    (#inner_bracket)
                }
            }

            #[allow(clippy::from_over_into)]
            impl #generics core::convert::Into<#array> for #ident #generics #where_clause {
                #[inline]
                fn into(self) -> #array {
                    [#inner_bracket]
                }
            }
        )
    };

    let impl_unop = {
        let impl_unop = |op_trait: proc_macro2::TokenStream,
                         op: proc_macro2::TokenStream|
         -> proc_macro2::TokenStream {
            let inner_new = if is_tuple {
                quote!(#(self.#tuple_indexes.#op()),*)
            } else {
                quote!(#(self.#fields.#op()),*)
            };
            let where_clause = if is_generic {
                quote!(#where_prelude #type_path: core::ops::#op_trait<Output=#type_path>)
            } else {
                quote!(#where_clause)
            };

            if is_generic || is_signed_primitive {
                quote!(
                    impl #generics core::ops::#op_trait for #ident #generics #where_clause {
                        type Output = Self;

                        #[inline]
                        fn #op(self) -> Self::Output {
                            Self::new(#inner_new)
                        }
                    }
                )
            } else {
                quote!()
            }
        };

        let impl_neg = impl_unop(quote!(Neg), quote!(neg));

        quote!(#impl_neg)
    };

    let impl_binop = {
        let impl_binop = |op_trait: proc_macro2::TokenStream,
                          op: proc_macro2::TokenStream,
                          op_assgin: bool|
         -> proc_macro2::TokenStream {
            let output = if !op_assgin {
                quote!(, Output=#type_path)
            } else {
                quote!()
            };
            let where_clause = match is_generic {
                true => {
                    quote!(
                        #where_prelude
                        #type_path: core::ops::#op_trait<#type_path #output>
                    )
                }
                false => {
                    quote!(
                        #where_prelude
                    )
                }
            };
            let inner = match (is_tuple, op_assgin) {
                (true, false) => {
                    quote!(Self::new(#(self.#tuple_indexes.#op(rhs.#tuple_indexes),)*))
                }
                (false, false) => quote!(Self::new(#(self.#fields.#op(rhs.#fields),)*)),
                (true, true) => quote!(#(self.#tuple_indexes.#op(rhs.#tuple_indexes);)*),
                (false, true) => quote!(#(self.#fields.#op(rhs.#fields);)*),
            };

            if !op_assgin {
                quote!(
                    impl #generics core::ops::#op_trait<Self> for #ident #generics #where_clause
                    {
                        type Output = Self;

                        #[inline]
                        fn #op(self, rhs: Self) -> Self::Output {
                            #inner
                        }
                    }
                )
            } else {
                quote!(
                    impl #generics core::ops::#op_trait<Self> for #ident #generics #where_clause
                    {
                        #[inline]
                        fn #op(&mut self, rhs: Self) {
                            #inner
                        }
                    }
                )
            }
        };

        let impl_add = impl_binop(quote!(Add), quote!(add), false);
        let impl_sub = impl_binop(quote!(Sub), quote!(sub), false);
        let impl_mul = impl_binop(quote!(Mul), quote!(mul), false);
        let impl_div = impl_binop(quote!(Div), quote!(div), false);
        let impl_add_assign = impl_binop(quote!(AddAssign), quote!(add_assign), true);
        let impl_sub_assign = impl_binop(quote!(SubAssign), quote!(sub_assign), true);
        let impl_mul_assign = impl_binop(quote!(MulAssign), quote!(mul_assign), true);
        let impl_div_assign = impl_binop(quote!(DivAssign), quote!(div_assign), true);

        quote! {
            #impl_add
            #impl_sub
            #impl_mul
            #impl_div
            #impl_add_assign
            #impl_sub_assign
            #impl_mul_assign
            #impl_div_assign
        }
    };

    // ONLY: This can only implement `mul<T>` and `div<T>`.
    let impl_binop_inner = {
        let impl_mul = {
            let where_clause = if is_generic {
                quote!(#where_prelude #type_path: core::marker::Copy + core::ops::Mul<Output=T>)
            } else {
                quote!()
            };
            let inner = if is_tuple {
                quote!(Self::new(#(self.#tuple_indexes * rhs),*))
            } else {
                quote!(Self::new(#(self.#fields * rhs),*))
            };
            quote!(
                impl #generics core::ops::Mul<#type_path> for #ident #generics #where_clause
                {
                    type Output = Self;

                    #[inline]
                    fn mul(self, rhs: #type_path) -> Self::Output {
                        #inner
                    }
                }
            )
        };

        let impl_mul_assign = {
            let where_clause = if is_generic {
                quote!(#where_prelude #type_path: core::marker::Copy + core::ops::MulAssign)
            } else {
                quote!()
            };
            let inner = if is_tuple {
                quote!(#(self.#tuple_indexes *= rhs;)*)
            } else {
                quote!(#(self.#fields *= rhs;)*)
            };
            quote!(
                impl #generics core::ops::MulAssign<#type_path> for #ident #generics #where_clause
                {
                    #[inline]
                    fn mul_assign(&mut self, rhs: #type_path) {
                        #inner
                    }
                }
            )
        };

        // Divisions are generally much slower than multiplications on modern CPUs, thus at
        // here we use a single division to compute the scalar's reciprocal and then perform
        // three component-wise multiplications.
        //
        // It is a common misconception that these sorts of optimizations are unnecessary
        // because the compiler will perform the necessary analysis. Compilers are
        // generally restricted from performing many transformations of this type.
        // For division, the IEEE floating-point standard requires that x/x = 1 for
        // all x, but if we compute 1/x and store it in a variable and then
        // multiply x by that value, it is not guaranteed that 1 will be the result.
        // In this case, we are willing to lose that guarantee in exchange for higher
        // performance.
        //
        // See https://www.pbr-book.org/3ed-2018/Geometry_and_Transformations/Vectors
        let impl_div = {
            let where_clause = if is_generic {
                quote!(
                    #where_prelude #type_path: core::marker::Copy
                    + core::ops::Mul<Output=T>
                    + core::ops::Div<Output=T>
                    + num_traits::identities::One
                )
            } else {
                quote!()
            };
            let one = if is_generic {
                quote!(#type_path::one())
            } else if is_float {
                quote!(1.0)
            } else {
                quote!(1)
            };
            let inner = if is_tuple {
                quote!(Self::new(#(self.#tuple_indexes * inv),*))
            } else {
                quote!(Self::new(#(self.#fields * inv),*))
            };
            quote!(
                impl #generics core::ops::Div<#type_path> for #ident #generics #where_clause
                {
                    type Output = Self;

                    #[inline]
                    fn div(self, rhs: #type_path) -> Self::Output {
                        let inv = #one / rhs;
                        #inner
                    }
                }
            )
        };

        let impl_div_assign = {
            let where_clause = if is_generic {
                quote!(
                    #where_prelude #type_path: core::marker::Copy
                    + core::ops::Div<Output=T>
                    + core::ops::Mul<Output=T>
                    + core::ops::MulAssign
                    + num_traits::identities::One
                )
            } else {
                quote!()
            };
            let one = if is_generic {
                quote!(#type_path::one())
            } else if is_float {
                quote!(1.)
            } else {
                quote!(1)
            };
            let inner = if is_tuple {
                quote!(#(self.#tuple_indexes *= inv;)*)
            } else {
                quote!(#(self.#fields *= inv;)*)
            };
            quote!(
                impl #generics core::ops::DivAssign<#type_path> for #ident #generics #where_clause
                {
                    #[inline]
                    fn div_assign(&mut self, rhs: #type_path) {
                        let inv = #one / rhs;
                        #inner
                    }
                }
            )
        };
        quote!(
            #impl_mul
            #impl_mul_assign
            #impl_div
            #impl_div_assign
        )
    };

    #[rustfmt::skip]
    let impl_binop_commutative = {
        let impl_binop_commutative = |op_trait: proc_macro2::TokenStream,
        op: proc_macro2::TokenStream,
        primitive: Option<proc_macro2::TokenStream>|
            -> proc_macro2::TokenStream {
                let ident = if is_generic {
                    quote!(#ident<#primitive>)
                } else {
                    quote!(#ident)
                };
                let primitive = if is_generic { quote!(#primitive) } else { quote!(#type_path) };

                quote!(
                    impl core::ops::#op_trait<#ident> for #primitive {
                        type Output = #ident;

                        #[inline]
                        fn #op(self, rhs: #ident) -> Self::Output {
                            let lhs: #ident = self.into();
                            rhs.#op(lhs)
                        }
                    }
                )
            };

        if is_generic {
            let impl_add_f32  = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(f32)));
            let impl_add_f64  = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(f64)));
            let impl_add_i8   = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(i8)));
            let impl_add_u8   = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(u8)));
            let impl_add_i16  = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(i16)));
            let impl_add_u16  = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(u16)));
            let impl_add_i32  = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(i32)));
            let impl_add_u32  = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(u32)));
            let impl_add_i64  = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(i64)));
            let impl_add_u64  = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(u64)));
            let impl_add_i128 = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(i128)));
            let impl_add_u128 = impl_binop_commutative(quote!(Add), quote!(add), Some(quote!(u128)));
            let impl_mul_f32  = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(f32)));
            let impl_mul_f64  = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(f64)));
            let impl_mul_i8   = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(i8)));
            let impl_mul_u8   = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(u8)));
            let impl_mul_i16  = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(i16)));
            let impl_mul_u16  = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(u16)));
            let impl_mul_i32  = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(i32)));
            let impl_mul_u32  = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(u32)));
            let impl_mul_i64  = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(i64)));
            let impl_mul_u64  = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(u64)));
            let impl_mul_i128 = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(i128)));
            let impl_mul_u128 = impl_binop_commutative(quote!(Mul), quote!(mul), Some(quote!(u128)));
            quote!(
                #impl_add_f32 
                #impl_add_f64 
                #impl_add_i8  
                #impl_add_u8  
                #impl_add_i16 
                #impl_add_u16 
                #impl_add_i32 
                #impl_add_u32 
                #impl_add_i64 
                #impl_add_u64 
                #impl_add_i128
                #impl_add_u128
                #impl_mul_f32 
                #impl_mul_f64 
                #impl_mul_i8  
                #impl_mul_u8  
                #impl_mul_i16 
                #impl_mul_u16 
                #impl_mul_i32 
                #impl_mul_u32 
                #impl_mul_i64 
                #impl_mul_u64 
                #impl_mul_i128
                #impl_mul_u128
                )
        } else if is_primitive {
            let impl_add = impl_binop_commutative(quote!(Add), quote!(add), None);
            let impl_mul = impl_binop_commutative(quote!(Mul), quote!(mul), None);
            quote!(
                #impl_add
                #impl_mul
            )
        } else {
            quote!()
        }
    };

    let impl_fn_shared = {
        let impl_consts = {
            // ONLY: is_primitive
            //
            // We have no means to construct a generic const because calls in constants are limited
            // to constant functions, tuple structs and tuple variants.
            let const_zero = if is_primitive {
                let zero = if is_float { quote!(0.) } else { quote!(0) };
                let zeros = repeat(&zero, fields_count);
                let bracketed = if is_tuple {
                    quote!((#(#zeros),*))
                } else {
                    quote!({#(#fields: #zero),*})
                };
                quote!(
                    /// A vector with all its elements set to zero.
                    pub const ZERO: #ident = #ident #bracketed;
                )
            } else {
                quote!()
            };

            // ONLY: is_primitive
            let const_one = if is_primitive {
                let one = if is_float { quote!(1.) } else { quote!(1) };
                let ones = repeat(&one, fields_count);
                let bracketed = if is_tuple {
                    quote!((#(#ones),*))
                } else {
                    quote!({#(#fields: #one),*})
                };
                quote!(
                    /// A vector with all its elements set to one.
                    pub const ONE: #ident = #ident #bracketed;
                )
            } else {
                quote!()
            };

            let const_lanes = {
                quote!(
                    /// The number of lanes in this vector.
                    pub const LANES: usize = #fields_count;
                )
            };

            quote!(
                #const_lanes
                #const_zero
                #const_one
            )
        };

        let impl_fn_new = {
            let body = if is_tuple {
                quote!(#ident(#(#fields),*))
            } else {
                quote!(#ident{#(#fields),*})
            };

            quote!(
                /// Constructs a new vector with the given values.
                #[inline]
                pub fn new(#(#fields: #type_path),*) -> Self {
                    #body
                }
            )
        };

        let impl_fn_splat = {
            let where_clause = if is_generic {
                quote!(where #type_path: core::marker::Copy)
            } else {
                quote!()
            };

            quote!(
                /// Construct a new vector by setting each of its element to the given value.
                #[inline]
                pub fn splat(value: #type_path) -> Self #where_clause {
                    Self::from(value)
                }
            )
        };

        // Should we expose these methods?
        let impl_fn_as_ptr = {
            quote!(
                /// Returns a raw pointer to this vector.
                #[inline]
                fn as_ptr(&self) -> *const #type_path {
                    self as *const _ as *const #type_path
                }

                /// Returns an unsafe mutable pointer to this vector.
                #[inline]
                fn as_mut_ptr(&mut self) -> *mut #type_path {
                    self as *mut _ as *mut #type_path
                }
            )
        };

        let impl_fn_as_slice = {
            quote!(
                /// Views this vector as an immutable slice.
                #[inline]
                pub fn as_slice(&self) -> &[#type_path] {
                    unsafe {
                        core::slice::from_raw_parts(self.as_ptr(), #fields_count)
                    }
                }

                /// Views this vector as a mutable slice.
                #[inline]
                pub fn as_mut_slice(&mut self) -> &mut [#type_path] {
                    unsafe {
                        core::slice::from_raw_parts_mut(self.as_mut_ptr(), #fields_count)
                    }
                }
            )
        };

        let impl_fn_unit_field = if fields_count <= 4 {
            let fn_names = if is_tuple {
                tuple_indexes
                    .clone()
                    .into_iter()
                    .map(|x| Ident::new(&format!("unit_{}", x.index), x.span()))
                    .collect::<Vec<Ident>>()
            } else {
                fields
                    .clone()
                    .into_iter()
                    .map(|x| Ident::new(&format!("unit_{}", x), x.span()))
                    .collect::<Vec<Ident>>()
            };
            let where_clause = if is_generic {
                quote!( where #type_path: core::marker::Copy + num_traits::One + num_traits::Zero)
            } else {
                quote!()
            };
            let (zero, one) = match (is_generic, is_float) {
                (true, _) => (quote!(#type_path::zero()), quote!(#type_path::one())),
                (false, true) => (quote!(0.0), quote!(1.0)),
                (false, false) => (quote!(0), quote!(1)),
            };
            let docs = if is_tuple {
                tuple_indexes
                    .clone()
                    .into_iter()
                    .map(|x| {
                        format!(
                            "Returns a unit vector which points along the `{}` axis.",
                            x.index
                        )
                    })
                    .collect::<Vec<String>>()
            } else {
                fields
                    .clone()
                    .into_iter()
                    .map(|x| format!("Returns a unit vector which points along the `{}` axis.", x))
                    .collect::<Vec<String>>()
            };
            if is_tuple {
                quote!(
                    #(
                        #[doc = #docs]
                        #[inline]
                        pub fn #fn_names() -> #ident #generics #where_clause {
                            let mut unit_field = #ident::splat(#zero);
                            unit_field.#tuple_indexes = #one;
                            unit_field
                        }
                    )*
                )
            } else {
                quote!(
                    #(
                        #[doc = #docs]
                        pub fn #fn_names() -> #ident #generics #where_clause {
                            let mut unit_field = #ident::splat(#zero);
                            unit_field.#fields = #one;
                            unit_field
                        }
                    )*
                )
            }
        } else {
            quote!()
        };

        let impl_fn_sum = {
            let where_clause = if is_generic {
                quote!(where #type_path: core::ops::Add<Output=#type_path>)
            } else {
                quote!()
            };
            let sum = if is_tuple {
                quote!(#(self.#tuple_indexes)+*)
            } else {
                quote!(#(self.#fields)+*)
            };
            quote!(
                /// Returns the sum of all elements in this vector.
                #[inline]
                pub fn sum(self) -> #type_path #where_clause {
                    #sum
                }
            )
        };

        let impl_fn_product = {
            let where_clause = if is_generic {
                quote!(where #type_path: core::ops::Mul<Output=#type_path>)
            } else {
                quote!()
            };

            let mut tuple_indexes = tuple_indexes.to_vec();
            let mut struct_fields = fields.to_vec();
            let last_tuple_field = tuple_indexes.pop();
            let last_struct_field = struct_fields.pop();

            let product = if is_tuple {
                quote!(#(self.#tuple_indexes *)* self.#last_tuple_field)
            } else {
                quote!(#(self.#struct_fields *)* self.#last_struct_field)
            };

            quote!(
                /// Returns the product of all elements in this vector.
                #[inline]
                pub fn product(self) -> #type_path #where_clause {
                    #product
                }
            )
        };

        let impl_fn_dot = {
            let where_clause = if is_generic {
                quote!(
                    where #type_path: core::ops::Add<Output=#type_path>
                    + core::ops::Mul<Output=#type_path>
                )
            } else {
                quote!()
            };

            quote!(
                /// Returns the dot product between `self` and `other`.
                #[inline]
                pub fn dot(self, other: Self) -> #type_path #where_clause {
                    (self * other).sum()
                }
            )
        };

        let impl_fn_length = if fields_count <= 4 {
            // ONLY: fields_count <= 4
            let fn_length_squared = {
                let where_clause = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + core::ops::Add<Output=#type_path>
                        + core::ops::Mul<Output=#type_path>
                    )
                } else {
                    quote!()
                };

                quote!(
                    /// Returns the sqaured length of this vector.
                    #[inline]
                    pub fn length_squared(self) -> #type_path #where_clause {
                        self.dot(self)
                    }
                )
            };

            let fn_length = if is_generic || is_float {
                let where_clause = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + num_traits::real::Real
                        + core::ops::Add<Output=#type_path>
                        + core::ops::Mul<Output=#type_path>
                    )
                } else {
                    quote!()
                };

                quote!(
                    /// Returns the length of this vector.
                    #[inline]
                    pub fn length(self) -> #type_path #where_clause {
                        self.dot(self).sqrt()
                    }
                )
            } else {
                quote!()
            };

            let fn_length_recip = if is_generic || is_float {
                let where_clause_length_recip = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + num_traits::One
                        + num_traits::real::Real
                        + core::ops::Add<Output=#type_path>
                        + core::ops::Mul<Output=#type_path>
                    )
                } else {
                    quote!()
                };

                let one = if is_float {
                    quote!(1.)
                } else {
                    quote!(#type_path::one())
                };

                quote!(
                    /// Returns the reciprocal of its length.
                    ///
                    /// # Panics
                    ///
                    /// Panics if `self` is of length zero.
                    #[inline]
                    pub fn length_recip(self) -> #type_path #where_clause_length_recip {
                        #one / self.length()
                    }
                )
            } else {
                quote!()
            };

            quote!(
                #fn_length_squared
                #fn_length
                #fn_length_recip
            )
        } else {
            quote!()
        };

        let impl_fn_distance = if fields_count <= 4 && (is_float || is_generic) {
            // ONLY: fields_count <= 4 && (is_float || is_generic)
            let fn_distance_squared = {
                let where_clause = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + num_traits::real::Real
                        + core::ops::Add<Output=#type_path>
                        + core::ops::Sub<Output=#type_path>
                        + core::ops::Mul<Output=#type_path>
                    )
                } else {
                    quote!()
                };

                quote!(
                    /// Returns the squared euclidean distance between two points.
                    #[inline]
                    pub fn distance_squared(self, other: Self) -> #type_path #where_clause {
                        (self - other).length_squared()
                    }
                )
            };

            let fn_distance = {
                let where_clause = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + num_traits::real::Real
                        + core::ops::Add<Output=#type_path>
                        + core::ops::Sub<Output=#type_path>
                        + core::ops::Mul<Output=#type_path>
                    )
                } else {
                    quote!()
                };

                quote!(
                    /// Returns the euclidean distance between two points.
                    #[inline]
                    pub fn distance(self, other: Self) -> #type_path #where_clause {
                        (self - other).length()
                    }
                )
            };

            quote!(
                #fn_distance_squared
                #fn_distance
            )
        } else {
            quote!()
        };

        let impl_fn_min_max = {
            let fn_min = {
                let where_clause = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + core::cmp::PartialOrd
                    )
                } else {
                    quote!()
                };
                let inner_new = if is_tuple {
                    quote!(
                        #(if self.#tuple_indexes > other.#tuple_indexes {
                            other.#tuple_indexes
                        } else {
                            self.#tuple_indexes
                        }),*
                    )
                } else {
                    quote!(
                        #(if self.#fields > other.#fields {
                            other.#fields
                        } else {
                            self.#fields
                        }),*
                    )
                };

                quote!(
                    /// Returns a vector containing the mininum values for each elements of `self`
                    /// and `other`.
                    #[inline]
                    pub fn min(self, other: Self) -> Self #where_clause {
                        Self::new(#inner_new)
                    }
                )
            };

            let fn_max = {
                let where_clause = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + core::cmp::PartialOrd
                    )
                } else {
                    quote!()
                };
                let inner_new = if is_tuple {
                    quote!(
                        #(if self.#tuple_indexes < other.#tuple_indexes {
                            other.#tuple_indexes
                        } else {
                            self.#tuple_indexes
                        }),*
                    )
                } else {
                    quote!(
                        #(if self.#fields < other.#fields {
                            other.#fields
                        } else {
                            self.#fields
                        }),*
                    )
                };

                quote!(
                    /// Returns a vector containing the maximun values for each elements of `self`
                    /// and `other`.
                    #[inline]
                    pub fn max(self, other: Self) -> Self #where_clause {
                        Self::new(#inner_new)
                    }
                )
            };

            let fn_min_elem = {
                let where_clause = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + core::cmp::PartialOrd
                    )
                } else {
                    quote!()
                };
                quote!(
                    /// Returns the smallest element in this vector.
                    #[inline]
                    pub fn min_elem(self) -> #type_path #where_clause {
                        let mut iter = self.iter();
                        // SAFETY: Unwrap here is safe bacause vector cannot be empty.
                        let mut min = iter.next().cloned().unwrap();
                        for &x in iter {
                            if x < min {
                                min = x;
                            }
                        }
                        min
                    }
                )
            };

            let fn_max_elem = {
                let where_clause = if is_generic {
                    quote!(
                        where #type_path: core::marker::Copy
                        + core::cmp::PartialOrd
                    )
                } else {
                    quote!()
                };
                quote!(
                    /// Returns the largest element in this vector.
                    #[inline]
                    pub fn max_elem(self) -> #type_path #where_clause {
                        let mut iter = self.iter();
                        // SAFETY: Unwrap here is safe bacause vector cannot be empty.
                        let mut max = iter.next().cloned().unwrap();
                        for &x in iter {
                            if x > max {
                                max = x;
                            }
                        }
                        max
                    }
                )
            };

            quote!(
                #fn_min
                #fn_max
                #fn_min_elem
                #fn_max_elem
            )
        };

        let impl_fn_map = {
            // ONLY: is_generic
            let inner_new = if is_tuple {
                quote!(#(f(self.#tuple_indexes)),*)
            } else {
                quote!(#(f(self.#fields)),*)
            };

            if is_generic {
                let predicates = where_clause.as_ref().map(|w| w.predicates.clone());
                let m_bounds = match &predicates {
                    Some(punct) => match punct.first() {
                        Some(syn::WherePredicate::Type(ty)) => {
                            let bounds = ty.bounds.clone();
                            quote!(M: #bounds, )
                        }
                        Some(_) => unreachable!(),
                        None => quote!(),
                    },
                    _ => quote!(),
                };
                quote!(
                    /// Returns an vector of the same shape as `self`, with function `f` applied
                    /// to each element in order.
                    #[inline]
                    pub fn map<M, F>(self, mut f: F) -> #ident<M>
                    where #m_bounds F: core::ops::FnMut(#type_path) -> M {
                        #ident::new(#inner_new)
                    }
                )
            } else {
                quote!()
            }
        };

        let impl_fn_apply = {
            let where_clause = if is_generic {
                quote!(
                    where #type_path: core::marker::Copy
                    , F: core::ops::FnMut(#type_path) -> #type_path
                )
            } else {
                quote!(
                    where F: core::ops::FnMut(#type_path) -> #type_path
                )
            };
            let body = if is_tuple {
                quote!(#(self.#tuple_indexes = f(self.#tuple_indexes);)*)
            } else {
                quote!(#(self.#fields = f(self.#fields);)*)
            };
            quote!(
                /// Applies the function `f` to each element in order, in-place.
                #[inline]
                pub fn apply<F>(&mut self, mut f: F) #where_clause {
                    #body
                }
            )
        };

        let impl_fn_abs = if is_generic || is_signed_primitive {
            // ONLY: is_generic || is_signed_primitive
            let where_clause = if is_generic {
                quote!(where #type_path: num_traits::sign::Signed)
            } else {
                quote!()
            };
            let inner_new = if is_tuple {
                quote!(#(self.#tuple_indexes.abs()),*)
            } else {
                quote!(#(self.#fields.abs()),*)
            };

            quote!(
                /// Computes the absolute value of self.
                #[inline]
                pub fn abs(self) -> Self #where_clause {
                    Self::new(#inner_new)
                }
            )
        } else {
            quote!()
        };

        let impl_fn_normalize = if fields_count <= 4 && (is_generic || is_float) {
            let where_clause = if is_generic {
                quote!(
                    where #type_path: core::ops::Add<Output=#type_path>
                    + num_traits::real::Real + num_traits::identities::One
                )
            } else {
                quote!()
            };
            let inv = if is_generic {
                quote!(let inv = #type_path::one() / self.length();)
            } else {
                quote!(let inv = 1.0 / self.length();)
            };
            quote!(
                /// Returns a normalized `self` whose length is equal to 1.
                #[inline]
                pub fn normalize(self) -> Self #where_clause {
                    #inv
                    self * inv
                }
            )
        } else {
            quote!()
        };

        let impl_fn_clamp = {
            let where_clause = if is_generic {
                quote!(
                    where #type_path: core::cmp::PartialOrd
                )
            } else {
                quote!()
            };
            let inner_new = if is_tuple {
                quote!(#(_clamp(self.#tuple_indexes, min.#tuple_indexes, max.#tuple_indexes)),*)
            } else {
                quote!(#(_clamp(self.#fields, min.#fields, max.#fields)),*)
            };
            quote!(
                /// Restricts each element in `self` to a certain interval given by the
                /// corresponing element in `min` and `max`.
                ///
                /// # Panics
                ///
                /// For each corresponing element in `min` and `max`, panics if `min > max`, `min`
                /// is NaN, or `max` is NaN.
                pub fn clamp(self, min: Self, max: Self) -> Self #where_clause {
                    fn _clamp<T>(x: T, min: T, max: T) -> T where T: core::cmp::PartialOrd {
                        assert!(min <= max);
                        let mut x = x;
                        if x < min {
                            x = min;
                        }
                        if x > max {
                            x = max;
                        }
                        x
                    }

                    Self::new(#inner_new)
                }
            )
        };

        let impl_fn_cross = if fields_count == 3 {
            // ONLY: fields_count == 3
            let where_clause = if is_generic {
                quote!(where #type_path: Copy + core::ops::Mul<Output=T> + core::ops::Sub<Output=T>)
            } else {
                quote!()
            };
            quote!(
                /// Returns the cross product between `self` and `other`.
                #[inline]
                pub fn cross(self, other: Self) -> Self #where_clause {
                    let ((x, y, z), (u, v, w)) = (self.into(), other.into());
                    (y * w - z * v, z * u - x * w, x * v - y * u).into()
                }
            )
        } else {
            quote!()
        };

        let impl_fn_is_nan = if is_generic || is_float {
            // ONLY: is_generic || is_float
            let where_clause = if is_generic {
                quote!(where #type_path: num_traits::float::Float)
            } else {
                quote!()
            };

            quote!(
                /// Returns `true` if any elements are `NaN`.
                #[inline]
                pub fn is_nan(self) -> bool #where_clause {
                    self.iter().any(|&x| x.is_nan())
                }
            )
        } else {
            quote!()
        };

        quote!(
            impl #generics #ident #generics #where_clause {
                #impl_consts
                #impl_fn_new
                #impl_fn_splat
                #impl_fn_unit_field
                #impl_fn_as_ptr
                #impl_fn_as_slice
                #impl_fn_sum
                #impl_fn_product
                #impl_fn_dot
                #impl_fn_length
                #impl_fn_distance
                #impl_fn_min_max
                #impl_fn_map
                #impl_fn_apply
                #impl_fn_abs
                #impl_fn_normalize
                #impl_fn_clamp
                #impl_fn_cross
                #impl_fn_is_nan
            }
        )
    };

    let expanded = quote! {
        #item_repr_c

        // Cheap traits
        #impl_as_ref
        #impl_de_ref
        #impl_borrow

        // Costly traits
        #impl_from_inner
        #impl_from_tuple_array
        #impl_into_tuple_array

        // Unops and Binops
        #impl_unop
        #impl_binop
        #impl_binop_inner
        #impl_binop_commutative

        // Shared methods
        #impl_fn_shared
    };

    // eprintln!("{}", expanded);

    Ok(expanded.into())
}

#[derive(Debug)]
struct ParseResult {
    ident: Ident,
    generics: Option<Generics>,
    where_clause: Option<WhereClause>,
    generic_ident: Option<Ident>,
    type_path: Option<Path>,
    fields: Vec<Ident>,
    fields_count: usize,
    tuple_indexes: Vec<Index>,
    compile_error: Option<TokenStream>,
    is_tuple: bool,
    is_generic: bool,
    is_primitive: bool,
    is_signed_primitive: bool,
    is_float: bool,
}

impl ParseResult {
    fn dummy() -> Self {
        ParseResult {
            ident: Ident::new("dummy", Span::call_site()),
            generics: None,
            where_clause: None,
            generic_ident: None,
            type_path: None,
            fields: Vec::new(),
            fields_count: 0,
            tuple_indexes: Vec::new(),
            compile_error: None,
            is_tuple: false,
            is_generic: false,
            is_primitive: false,
            is_signed_primitive: false,
            is_float: false,
        }
    }
}

// Parses the given struct and checks if the given struct is suitable to represent a vector.
//
// - Unit struct is definitely not a valid choice.
// - All fields of the struct must be of the same type.
// - The struct can only contain at most one generic type T.
// - If the struct is generic over type T, all types of its fields must be of type T.
fn parse_struct(item: &ItemStruct) -> ParseResult {
    let mut ret = ParseResult::dummy();

    ret.ident = item.ident.clone();
    ret.generics = Some(item.generics.clone());

    // Removes trailing punct from where_clause if any.
    let mut where_clause = item.generics.where_clause.clone();
    if let Some(v) = where_clause.as_mut() {
        if !v.predicates.is_empty() && !v.predicates.trailing_punct() {
            (*v).predicates.push_punct(Default::default());
        }
    }
    ret.where_clause = where_clause;

    macro_rules! err_if_none {
        ($span:expr, $fmt:literal $(,)? $($arg:expr),*) => {{
            if ret.compile_error.is_none() {
                ret.compile_error = Some(err!($span, $fmt $(, $arg),*));
                return ret;
            }
        }}
    }

    match &item.generics.params.first() {
        Some(generic) => {
            use syn::GenericParam::*;
            match generic {
                Type(ty) => {
                    ret.is_generic = true;
                    ret.generic_ident = Some(ty.ident.clone());
                }
                Lifetime(lifetime) => {
                    err_if_none!(
                        lifetime.span(),
                        "expected generic type, but found generic lifetime"
                    );
                }
                Const(cons) => {
                    err_if_none!(
                        cons.span(),
                        "expected generic type, but found generic const"
                    );
                }
            }
        }
        // Reaching here means this struct contains no generics.
        None => (),
    }

    match &item.fields {
        // Rejects unit struct.
        Fields::Unit => {
            err_if_none!(item.span(), "unit struct cannot be made into a vector");
        }
        Fields::Named(fields) => {
            // Errors if this struct contains no fields.
            if fields.named.is_empty() {
                err_if_none!(
                    fields.span(),
                    "expected at least one field, but found nothing"
                );
            } else {
                let iter = fields.named.iter();

                // SAFETY: Reaching this branch means fields cannot be empty, thus the unwrap here
                // is safe.
                let mut prev = fields.named.first().unwrap();
                for x in iter {
                    match &x.ty {
                        Type::Path(type_path) => {
                            if !x.ty.eq(&prev.ty) {
                                err_if_none!(
                                    fields.span(),
                                    "all fields of this struct should be of the same type"
                                );
                            } else {
                                ret.fields.push(x.ident.as_ref().unwrap().clone());
                                ret.fields_count += 1;
                                ret.type_path = Some(type_path.path.clone());
                            }
                        }
                        _ => {
                            if ret.is_generic {
                                err_if_none!(
                                    fields.span(),
                                    "expected at most one generic type, but found others",
                                );
                            } else {
                                err_if_none!(
                                    fields.span(),
                                    "expected owned primitive type, but found others"
                                );
                            }
                        }
                    }
                    prev = x;
                }
            }
        }
        Fields::Unnamed(fields) => {
            ret.is_tuple = true;

            // Errors if this struct contains no fields.
            if fields.unnamed.is_empty() {
                err_if_none!(
                    fields.span(),
                    "expected at least one field, but found nothing"
                );
            } else {
                let iter = fields.unnamed.iter();

                // SAFETY: Reaching this branch means fields cannot be empty, thus the unwrap here
                // is safe.
                let mut prev = fields.unnamed.first().unwrap();
                for x in iter {
                    match &x.ty {
                        Type::Path(type_path) => {
                            if !x.ty.eq(&prev.ty) {
                                err_if_none!(
                                    fields.span(),
                                    "all fields of this struct should be of the same type"
                                );
                            } else {
                                ret.tuple_indexes.push(Index::from(ret.fields_count));
                                ret.fields.push(format_ident!("a{}", ret.fields_count));
                                ret.fields_count += 1;
                                ret.type_path = Some(type_path.path.clone());
                            }
                        }
                        _ => {
                            // err_if_none!(fields.span(), "expected owned type, but found others");
                            if ret.is_generic {
                                err_if_none!(
                                    fields.span(),
                                    "expected at most one generic type, but found others",
                                );
                            } else {
                                err_if_none!(
                                    fields.span(),
                                    "expected owned primitive type, but found others"
                                );
                            }
                        }
                    }
                    prev = x;
                }
            }
        }
    }

    if ret.fields_count == 1 {
        err_if_none!(
            item.fields.span(),
            "create a vector with only one field is not allowed"
        )
    }

    if !ret.is_generic {
        let checked = check_primitive(&ret.type_path);
        ret.is_primitive = checked.0;
        ret.is_signed_primitive = checked.1;
        ret.is_float = checked.2;

        if !ret.is_primitive {
            err_if_none!(
                item.fields.span(),
                "expected numeric primitive type, but found others"
            );
        }
    }

    ret
}
