pub fn resize_no_init<T: POD>(vec: &mut Vec<T>, new_len: usize) {
    let len = vec.len();

    if new_len > len {
        vec.reserve_exact(new_len - len);
    }

    let ptr = vec.as_mut_ptr();
    let cap = vec.capacity();

    unsafe { (vec as *mut Vec<T>).write(Vec::from_raw_parts(ptr, new_len, cap)) };
}

/// # Safety
///
/// POD stands for Plain Old Data.
///
/// That is, a type that does not need explicit initialization and
/// does not have `Drop` impl for itself and any type it contains.
///
/// Namely, a type that only contains integers.
pub unsafe trait POD {}

macro_rules! impl_POD {
    ($type:ident) => {
        unsafe impl POD for $type {}
    };
}

impl_POD!(u8);
impl_POD!(u16);
impl_POD!(u32);
impl_POD!(u64);
impl_POD!(u128);
impl_POD!(usize);

impl_POD!(i8);
impl_POD!(i16);
impl_POD!(i32);
impl_POD!(i64);
impl_POD!(i128);
impl_POD!(isize);

#[cfg(test)]
mod tests {
    use super::resize_no_init;

    #[test]
    fn test() {
        let mut vec = Vec::<u8>::new();
        vec.push(0);
        vec.push(1);
        vec.push(2);

        resize_no_init(&mut vec, 100);

        assert_eq!(vec.len(), 100);
        assert_eq!(vec[0], 0);
        assert_eq!(vec[1], 1);
        assert_eq!(vec[2], 2);

        // ensure that the newly allocated memory is usuable
        for (i, each) in vec[3..].iter_mut().enumerate() {
            *each = (i + 3).try_into().unwrap();
        }

        for (i, each) in vec.iter().enumerate() {
            assert_eq!(*each as usize, i);
        }

        // ensure that cap is set correctly so that
        // the vec can still be extended.
        for i in 100..200 {
            vec.push(i);
        }

        for (i, each) in vec.iter().enumerate() {
            assert_eq!(*each as usize, i);
        }

        // Try to reserve to ensure that cap is set correctly.
        vec.reserve(1000);
    }
}
