// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Imported from hg/rust/hg-core/src/testing.rs@168041fa6d5f
//
// Copyright 2018  Georges RACINET <georges.racinet@octobus.net>
//           2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::ancestors::{merge_rank, rank};
use crate::graph::{
    Graph, GraphReadError, Parents, Rank, Revision, NULL_REVISION,
};
use std::result::Result;
use std::result::Result::{Err, Ok};

/// A stub `Graph`, same as the one from `test-ancestor.py`
///
/// o  13
/// |
/// | o  12
/// | |
/// | | o    11
/// | | |\
/// | | | | o  10
/// | | | | |
/// | o---+ |  9
/// | | | | |
/// o | | | |  8
///  / / / /
/// | | o |  7
/// | | | |
/// o---+ |  6
///  / / /
/// | | o  5
/// | |/
/// | o  4
/// | |
/// o |  3
/// | |
/// | o  2
/// |/
/// o  1
/// |
/// o  0
#[derive(Clone, Debug)]
pub struct SampleGraph;

impl Graph for SampleGraph {
    fn parents(&self, rev: Revision) -> Result<Parents, GraphReadError> {
        match rev {
            0 => Ok(Parents([NULL_REVISION, NULL_REVISION])),
            1 => Ok(Parents([0, NULL_REVISION])),
            2 => Ok(Parents([1, NULL_REVISION])),
            3 => Ok(Parents([1, NULL_REVISION])),
            4 => Ok(Parents([2, NULL_REVISION])),
            5 => Ok(Parents([4, NULL_REVISION])),
            6 => Ok(Parents([4, NULL_REVISION])),
            7 => Ok(Parents([4, NULL_REVISION])),
            8 => Ok(Parents([NULL_REVISION, NULL_REVISION])),
            9 => Ok(Parents([6, 7])),
            10 => Ok(Parents([5, NULL_REVISION])),
            11 => Ok(Parents([3, 7])),
            12 => Ok(Parents([9, NULL_REVISION])),
            13 => Ok(Parents([8, NULL_REVISION])),
            _ => Err(GraphReadError::InvalidKey),
        }
    }

    fn rank(&self, rev: Revision) -> Result<Rank, GraphReadError> {
        rank(self, rev)
    }

    fn merge_rank(&self, rev: Revision) -> Result<Rank, GraphReadError> {
        merge_rank(self, rev)
    }
}

/// A corrupted Graph, supporting error handling tests
#[derive(Clone, Debug)]
pub struct Corrupted;

impl Graph for Corrupted {
    fn parents(&self, rev: Revision) -> Result<Parents, GraphReadError> {
        match rev {
            1 => Ok(Parents([0, -1])),
            _ => Err(GraphReadError::InvalidKey),
        }
    }

    fn rank(&self, rev: Revision) -> Result<Rank, GraphReadError> {
        match rev {
            0 => Ok(0),
            _ => Err(GraphReadError::InvalidKey),
        }
    }

    fn merge_rank(&self, rev: Revision) -> Result<Rank, GraphReadError> {
        match rev {
            0 => Ok(0),
            _ => Err(GraphReadError::InvalidKey),
        }
    }
}
