// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::graph::{
    Graph, GraphReadError, LabelledGraph, NodeID, Revision, StableOrderGraph,
    NODE_ID_LEN,
};
use crate::testing::graph_in_mem::InMemoryGraph;
use crypto::digest::Digest;
use crypto::sha1::Sha1;
use std::cmp::Ordering;
use std::result::Result;
use std::result::Result::Ok;

/// Seed to use when hashing NodeIDs for tests randomisation purposes.
/// 0 disables hashing (for compatibility with existing tests with already
/// fixed outputs)
pub static mut HASH_SEED: u8 = 0;

/// Hashes a NodeID with the given seed.
fn hash_node_id_with_seed(seed: u8, node_id: NodeID) -> NodeID {
    let mut hasher = Sha1::new();
    hasher.input(&[seed]);
    hasher.input(&node_id.0);

    let mut result = [0; NODE_ID_LEN];
    hasher.result(&mut result);
    NodeID(result)
}

/// Hashes a NodeID using the seed of the global context.
fn hash_node_id(node_id: NodeID) -> NodeID {
    match unsafe { HASH_SEED } {
        0 => node_id,
        seed => hash_node_id_with_seed(seed, node_id),
    }
}

/// A strategy to use to compare nodes, defining a stable order.
pub trait NodeComparator: Sized {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError>;
}

fn compare_node_id(
    graph: &LabelledGraph,
    l: Revision,
    r: Revision,
) -> Result<Ordering, GraphReadError> {
    let lh = hash_node_id(graph.node_id(l)?);
    let rh = hash_node_id(graph.node_id(r)?);
    Ok(lh.cmp(&rh))
}

/// NodeComparator implementation that uses the stable NodeID.
/// For testing purpopses.
pub struct NodeIDComparator {}

impl NodeComparator for NodeIDComparator {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError> {
        compare_node_id(graph, l, r)
    }
}

/// NodeComparator implementation that uses the stable NodeID.
/// Reverses the order.
/// For testing purpopses.
pub struct NodeIDComparatorReverse {}

impl NodeComparator for NodeIDComparatorReverse {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError> {
        compare_node_id(graph, r, l)
    }
}

pub struct MinRankComparator {}

impl NodeComparator for MinRankComparator {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError> {
        match graph.rank(r)?.cmp(&graph.rank(l)?) {
            Ordering::Equal => compare_node_id(graph, l, r),
            neq => Ok(neq),
        }
    }
}

pub struct MaxRankComparator {}

impl NodeComparator for MaxRankComparator {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError> {
        match graph.rank(l)?.cmp(&graph.rank(r)?) {
            Ordering::Equal => compare_node_id(graph, r, l),
            neq => Ok(neq),
        }
    }
}

pub struct MinMergeRankComparator {}

impl NodeComparator for MinMergeRankComparator {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError> {
        match graph.merge_rank(r)?.cmp(&graph.merge_rank(l)?) {
            Ordering::Equal => compare_node_id(graph, l, r),
            neq => Ok(neq),
        }
    }
}

pub struct MaxMergeRankComparator {}

impl NodeComparator for MaxMergeRankComparator {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError> {
        match graph.merge_rank(l)?.cmp(&graph.merge_rank(r)?) {
            Ordering::Equal => compare_node_id(graph, r, l),
            neq => Ok(neq),
        }
    }
}

pub struct MinHardLeapsComparator {}

impl NodeComparator for MinHardLeapsComparator {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError> {
        let l_leaps = graph.leap_counts(l)?.hard_leaps;
        let r_leaps = graph.leap_counts(r)?.hard_leaps;
        match r_leaps.cmp(&l_leaps) {
            Ordering::Equal => compare_node_id(graph, l, r),
            neq => Ok(neq),
        }
    }
}

pub struct MaxHardLeapsComparator {}

impl NodeComparator for MaxHardLeapsComparator {
    fn compare(
        graph: &InMemoryGraph<Self>,
        l: Revision,
        r: Revision,
    ) -> Result<Ordering, GraphReadError> {
        let l_leaps = graph.leap_counts(l)?.hard_leaps;
        let r_leaps = graph.leap_counts(r)?.hard_leaps;
        match l_leaps.cmp(&r_leaps) {
            Ordering::Equal => compare_node_id(graph, r, l),
            neq => Ok(neq),
        }
    }
}
