// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use std::io::{stdin, BufRead};
use std::iter::IntoIterator;
use structopt::*;
use vcsgraph::ancestors::{
    ancestor_set, is_ancestor, ordered_ancestors, rank,
    OrderedAncestorIterator,
};
use vcsgraph::graph::{Graph, LabelledGraph, NodeID, Revision, SizedGraph};
use vcsgraph::testing::graph_io::read_graph;
use vcsgraph::testing::ordering::NodeIDComparator;

#[derive(Debug, StructOpt)]
#[structopt(
    name = "vcsgraph-testing",
    about = r#"
    Runs various functions on graphs piped to the standard input from:
    hg log --rev 'sort(all())' --template '{node} {p1.node} {p2.node}\n'
    "#
)]
struct CliArgs {
    #[structopt(subcommand)]
    command: Command,
}

#[derive(Debug, StructOpt)]
enum Command {
    #[structopt(
        about = "Prints the ranks (number of ancestors including itself) of \
        all changsets. This uses the naive implementation."
    )]
    NaiveRanks,

    #[structopt(
        about = "Prints the ranks (number of ancestors including itself) of \
        all changsets. This uses the rank computed from the leap information."
    )]
    LeapRanks,

    #[structopt(
        about = "Prints the set of ancestors of a given changeset, including \
        itself."
    )]
    AncestorSet(AncestorSetArguments),

    #[structopt(
        about = "Prints whether some changeset is the ancestor of some head."
    )]
    IsAncestor(IsAncestorArguments),

    #[structopt(
        about = "Prints the ordered ancestors of a given changeset, including \
        itself. Merge parents are ordered by id. This uses the naive \
        implementation."
    )]
    NaiveOrderedAncestors(OrderedAncestorArguments),

    #[structopt(
        about = "Prints the ordered ancestors of a given changeset, including \
        itself. Merge parents are ordered by id. This uses the leap iterator \
        implementation."
    )]
    LeapOrderedAncestors(OrderedAncestorArguments),
}

#[derive(Debug, StructOpt)]
struct AncestorSetArguments {
    node_id: NodeID,
}

#[derive(Debug, StructOpt)]
struct IsAncestorArguments {
    head: NodeID,
    node_id: NodeID,
}

#[derive(Debug, StructOpt)]
struct OrderedAncestorArguments {
    node_id: NodeID,
    // TODO: add order func choice
}

fn print_node_ids(
    graph: &impl LabelledGraph,
    revs: impl IntoIterator<Item = Revision>,
) {
    for rev in revs {
        println!("{}", graph.node_id(rev).unwrap());
    }
}

fn main() {
    let args: CliArgs = CliArgs::from_args();
    let graph =
        read_graph::<_, NodeIDComparator>(stdin().lock().lines()).unwrap();

    match args.command {
        Command::NaiveRanks => {
            for rev in 0..graph.nb_nodes() as Revision {
                let node_id = graph.node_id(rev).unwrap();
                let rank = rank(&graph, rev).unwrap();
                println!("{} {}", node_id, rank);
            }
        }

        Command::LeapRanks => {
            for rev in 0..graph.nb_nodes() as Revision {
                let node_id = graph.node_id(rev).unwrap();
                let rank = graph.rank(rev).unwrap();
                println!("{} {}", node_id, rank);
            }
        }

        Command::AncestorSet(AncestorSetArguments { node_id }) => {
            let rev = graph.rev_of_node_id(node_id).unwrap();
            let ancestors = ancestor_set(&graph, &[rev]).unwrap();
            print_node_ids(&graph, ancestors);
        }

        Command::IsAncestor(IsAncestorArguments { head, node_id }) => {
            let head_rev = graph.rev_of_node_id(head).unwrap();
            let node_rev = graph.rev_of_node_id(node_id).unwrap();
            let result = is_ancestor(&graph, head_rev, node_rev).unwrap();
            println!("{}", result);
        }

        Command::NaiveOrderedAncestors(OrderedAncestorArguments {
            node_id,
        }) => {
            let node_rev = graph.rev_of_node_id(node_id).unwrap();
            let ancestors = ordered_ancestors(&graph, node_rev).unwrap();
            print_node_ids(&graph, ancestors);
        }

        Command::LeapOrderedAncestors(OrderedAncestorArguments {
            node_id,
        }) => {
            let node_rev = graph.rev_of_node_id(node_id).unwrap();
            let iterator =
                OrderedAncestorIterator::new(&graph, node_rev).unwrap();
            print_node_ids(&graph, iterator.map(|res| res.unwrap()));
        }
    }
}
