// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Laurent Bulteau <laurent.bulteau@u-pem.fr>
//                 Pierre-Yves David <pierre-yves.david@octobus.net>
//                 Georges Racinet <georges.racinet@octobus.net>
//                 Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::ancestors::node_leaps;
use crate::graph::{
    Graph, GraphReadError, GraphWriteError, LabelledGraph, Leap, LeapCounts,
    LeapType, MutableGraph, NodeID, Parents, Rank, Revision, SizedGraph,
    StableOrderGraph, NULL_ID, NULL_REVISION,
};
use crate::testing::ordering::NodeComparator;
use std::cmp::Ordering;
use std::collections::HashMap;
use std::marker::PhantomData;
use std::option::Option::Some;
use std::result::Result;
use std::result::Result::{Err, Ok};
use std::vec::Vec;

/// Node
///
/// A single revision node.
#[derive(Clone, Debug, PartialEq, Eq)]
struct Node {
    rev: Revision,
    id: NodeID,
    parents: Parents,
    rank: Rank,
    merge_rank: Rank,
    leap_counts: LeapCounts,
    leaps: Vec<Leap>,
}

/// Fully in-memory representation of a graph
#[derive(Clone, Debug)]
pub struct InMemoryGraph<C: NodeComparator> {
    _node_comparator: PhantomData<C>,
    _empty_leap_vec: Vec<Leap>,
    /// Nodes indexed by Revision
    nodes: Vec<Node>,
    rev_map: HashMap<NodeID, Revision>,
}

impl<C: NodeComparator> InMemoryGraph<C> {
    pub fn new() -> Self {
        let mut graph = Self {
            _node_comparator: PhantomData,
            _empty_leap_vec: Vec::with_capacity(0),
            nodes: Vec::new(),
            rev_map: HashMap::new(),
        };
        graph.rev_map.insert(NULL_ID, NULL_REVISION);
        graph
    }

    fn get_by_revision(&self, rev: Revision) -> Result<Node, GraphReadError> {
        match self.nodes.get(rev as usize) {
            Some(node) => Ok(node.clone()),
            None => Err(GraphReadError::InvalidKey),
        }
    }
}

impl<C: NodeComparator> Graph for InMemoryGraph<C> {
    fn parents(&self, rev: Revision) -> Result<Parents, GraphReadError> {
        Ok(match rev {
            NULL_REVISION => Parents([NULL_REVISION, NULL_REVISION]),
            r => self.get_by_revision(r)?.parents,
        })
    }

    fn rank(&self, rev: Revision) -> Result<Rank, GraphReadError> {
        Ok(match rev {
            NULL_REVISION => 0,
            r => self.get_by_revision(r)?.rank,
        })
    }

    fn merge_rank(&self, rev: Revision) -> Result<Rank, GraphReadError> {
        Ok(match rev {
            NULL_REVISION => 0,
            r => self.get_by_revision(r)?.merge_rank,
        })
    }
}

impl<C: NodeComparator> LabelledGraph for InMemoryGraph<C> {
    fn rev_of_node_id(
        &self,
        node_id: NodeID,
    ) -> Result<Revision, GraphReadError> {
        self.rev_map
            .get(&node_id)
            .cloned()
            .ok_or(GraphReadError::InvalidKey)
    }

    fn node_id(&self, rev: Revision) -> Result<NodeID, GraphReadError> {
        Ok(self.get_by_revision(rev)?.id)
    }
}

impl<C: NodeComparator> SizedGraph for InMemoryGraph<C> {
    fn nb_nodes(&self) -> usize {
        self.nodes.len()
    }
}

impl<C: NodeComparator> StableOrderGraph for InMemoryGraph<C> {
    fn sorted_parents(
        &self,
        parents: Parents,
    ) -> Result<Parents, GraphReadError> {
        Ok(Parents(match parents {
            Parents([p, NULL_REVISION]) | Parents([NULL_REVISION, p]) => {
                [p, NULL_REVISION]
            }
            Parents([p1, p2]) => {
                if C::compare(self, p1, p2)? == Ordering::Less {
                    [p1, p2]
                } else {
                    [p2, p1]
                }
            }
        }))
    }

    fn ordered_parents(
        &self,
        rev: Revision,
    ) -> Result<Parents, GraphReadError> {
        self.sorted_parents(self.parents(rev)?)
    }

    fn leaps(&self, rev: Revision) -> Result<&Vec<Leap>, GraphReadError> {
        match rev {
            NULL_REVISION => Ok(&self._empty_leap_vec),
            r => Ok(&self.nodes[r as usize].leaps),
        }
    }

    fn leap_counts(
        &self,
        rev: Revision,
    ) -> Result<LeapCounts, GraphReadError> {
        Ok(match rev {
            NULL_REVISION => LeapCounts {
                soft_leaps: 0,
                hard_leaps: 0,
            },
            r => self.nodes[r as usize].leap_counts,
        })
    }
}

fn count_leaps(leaps: &[Leap]) -> LeapCounts {
    let mut leap_counts = LeapCounts {
        soft_leaps: 0,
        hard_leaps: 0,
    };

    for leap in leaps {
        match leap.leap_type {
            LeapType::Soft => leap_counts.soft_leaps += 1,
            LeapType::Hard | LeapType::Last => leap_counts.hard_leaps += 1,
        }
    }

    leap_counts
}

fn make_node<C: NodeComparator>(
    graph: &InMemoryGraph<C>,
    parents: Parents,
    node_id: NodeID,
    rev: Revision,
) -> Result<Node, GraphReadError> {
    let Parents([p_min, p_max]) = graph.sorted_parents(parents)?;
    let leap_info = node_leaps(graph, rev, &parents)?;
    let exclusive_merge_rank = if p_max == NULL_REVISION {
        0
    } else {
        1 + leap_info.exclusive_merge_count
    };

    Ok(Node {
        rev,
        id: node_id,
        parents,
        rank: 1 + leap_info.exclusive_part_size + graph.rank(p_min)?,
        merge_rank: exclusive_merge_rank + graph.merge_rank(p_min)?,
        leap_counts: count_leaps(&leap_info.leaps)
            + graph.leap_counts(p_min)?,
        leaps: leap_info.leaps,
    })
}

impl<C: NodeComparator> MutableGraph for InMemoryGraph<C> {
    fn push(
        &mut self,
        node_id: NodeID,
        p1: NodeID,
        p2: NodeID,
    ) -> Result<Revision, GraphWriteError> {
        let parents = Parents([self.rev_map[&p1], self.rev_map[&p2]]);
        let rev = self.nodes.len() as Revision;
        let node = make_node(self, parents, node_id, rev)
            .map_err(|_| GraphWriteError::InvalidParents(node_id))?;

        self.rev_map.insert(node_id, rev);
        self.nodes.push(node);
        Ok(rev)
    }
}

#[cfg(test)]
mod tests {
    use crate::graph::{
        Graph, MutableGraph, NodeID, Parents, SizedGraph, NODE_ID_LEN,
        NULL_ID, NULL_REVISION,
    };
    use crate::testing::graph_in_mem::InMemoryGraph;
    use crate::testing::ordering::NodeIDComparator;
    use std::iter::Iterator;
    use std::vec::Vec;

    #[test]
    fn test_simple_in_memory_graph() {
        let node: Vec<NodeID> =
            (0..=6).map(|i| NodeID([i + 1; NODE_ID_LEN])).collect();

        let mut graph = InMemoryGraph::<NodeIDComparator>::new();
        graph.push(node[0], NULL_ID, NULL_ID).unwrap();
        graph.push(node[1], node[0], NULL_ID).unwrap();
        graph.push(node[2], node[0], NULL_ID).unwrap();
        graph.push(node[3], node[1], NULL_ID).unwrap();
        graph.push(node[4], node[3], node[2]).unwrap();
        graph.push(node[5], node[4], NULL_ID).unwrap();
        graph.push(node[6], node[3], NULL_ID).unwrap();

        assert_eq!(graph.nb_nodes(), 7);

        assert_eq!(
            graph.parents(NULL_REVISION).unwrap(),
            Parents([NULL_REVISION, NULL_REVISION])
        );
        assert_eq!(graph.rank(NULL_REVISION).unwrap(), 0);

        let r0 = graph.get_by_revision(0).unwrap();
        assert_eq!(r0.rev, 0);
        assert_eq!(r0.id, node[0]);
        assert_eq!(r0.parents, Parents([NULL_REVISION, NULL_REVISION]));
        assert_eq!(r0.rank, 1);
        assert_eq!(r0.merge_rank, 0);
        assert_eq!(r0.leap_counts.soft_leaps, 0);
        assert_eq!(r0.leap_counts.hard_leaps, 0);

        let r4 = graph.get_by_revision(4).unwrap();
        assert_eq!(r4.rev, 4);
        assert_eq!(r4.id, node[4]);
        assert_eq!(r4.parents, Parents([3, 2]));
        assert_eq!(r4.rank, 5);
        assert_eq!(r4.merge_rank, 1);
        assert_eq!(r4.leap_counts.soft_leaps, 0);
        assert_eq!(r4.leap_counts.hard_leaps, 1);

        let r6 = graph.get_by_revision(6).unwrap();
        assert_eq!(r6.rev, 6);
        assert_eq!(r6.id, node[6]);
        assert_eq!(r6.parents, Parents([3, NULL_REVISION]));
        assert_eq!(graph.parents(6).unwrap(), Parents([3, NULL_REVISION]));
        assert_eq!(r6.rank, 4);
        assert_eq!(r6.merge_rank, 0);
        assert_eq!(r6.leap_counts.soft_leaps, 0);
        assert_eq!(r6.leap_counts.hard_leaps, 0);
    }
}
