// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use crate::ancestors::OrderedAncestorIterator;
use crate::graph::{
    Graph, GraphReadError, Rank, Revision, SizedGraph, StableOrderGraph,
};
use std::collections::{HashMap, HashSet, VecDeque};
use std::iter::FromIterator;
use std::result::Result;
use std::result::Result::Ok;

///// BASE DEFS

/// Sorting/permutation of the Revision-s of a graph.
pub type NodeSort = Vec<Revision>;

/// Returns a {Revision -> sort-index} mapping from a given NodeSort.
/// This loosely corresponds to the inverse of a sorting permutation.
pub fn index_map(node_sort: &NodeSort) -> HashMap<Revision, usize> {
    let mut inverse = HashMap::with_capacity(node_sort.len());
    for (i, rev) in node_sort.iter().enumerate() {
        inverse.insert(*rev, i);
    }
    inverse
}

///// SORTERS

/// Stable sort.
pub fn stable_sort(
    graph: &(impl Graph + SizedGraph + StableOrderGraph),
    head: Revision,
) -> Result<NodeSort, GraphReadError> {
    let iter = OrderedAncestorIterator::new(graph, head)?;
    let mut list = iter.collect::<Result<NodeSort, GraphReadError>>()?;
    list.reverse();
    Ok(list)
}

// TODO: move to graph.rs?
// TODO: accumulate revision children at insertion in InMemoryGraph
// TODO: define some order in branchpoints children
trait OrderedChildrenGraph: Graph {
    fn ordered_children(
        &self,
        rev: Revision,
    ) -> Result<&[Revision], GraphReadError>;
}

/// Returns the node sort given by a postfix DFS walk of the Revision-s of a
/// subgraph, starting at each of the given roots and excluding the already
/// visited nodes.
///
/// General idea of the non-recursive implementation: we do a DFS walk on the
/// graph and we record the traversed nodes in a stack as we go deeper. We
/// simulate the postfix appending by unwinding the stack when we are done
/// exploring a branch, down to the latest active branchpoint. This is detected
/// and determined by using the rank of the Revision currently considered in
/// the DFS walk, which increases in the downward exploration, and jumps back
/// to a lower value after completely exploring a branch. This lower value
/// corresponds to the rank of the branchpoint + 1.
///
/// TODO: prove correctness
/// TODO: add test
fn postfix_depth_first_node_sort(
    graph: &(impl Graph + OrderedChildrenGraph),
    node_restriction: &HashSet<Revision>,
    roots: &[Revision],
) -> Result<NodeSort, GraphReadError> {
    fn unwind_down_to_rank(
        graph: &impl Graph,
        down_to_rank_excl: Rank,
        traversed_stack: &mut Vec<Revision>,
        node_sort: &mut NodeSort,
    ) -> Result<(), GraphReadError> {
        while let Some(rev) = traversed_stack.last() {
            if graph.rank(*rev)? <= down_to_rank_excl {
                break;
            }

            node_sort.push(*rev);
            traversed_stack.pop();
        }

        Ok(())
    }

    let mut node_sort = Vec::with_capacity(node_restriction.len());
    let mut visited = HashSet::with_capacity(node_restriction.len());
    let mut traversed = Vec::new();
    let mut to_visit = roots.to_vec();
    to_visit.reverse();

    while let Some(rev) = to_visit.pop() {
        if !node_restriction.contains(&rev) || !visited.insert(rev) {
            continue;
        }

        unwind_down_to_rank(
            graph,
            graph.rank(rev)? - 1, // parent of current rev
            &mut traversed,
            &mut node_sort,
        )?;

        traversed.push(rev);
        to_visit.extend(graph.ordered_children(rev)?);
    }

    unwind_down_to_rank(graph, 0, &mut traversed, &mut node_sort)?;
    node_sort.reverse();
    Ok(node_sort)
}

/// Returns the permutation given by a BFS walk of the Revision-s of a
/// subgraph, starting by the given roots and excluding the already visited
/// nodes.
///
/// General idea of the implementation: we do a BFS walk on the subgraph, but
/// we enqueue a new node to visit only once all its parents have been visited.
///
/// TODO: add test
fn breadth_first_head_sort(
    graph: &(impl Graph + OrderedChildrenGraph),
    node_restriction: &HashSet<Revision>,
    roots: &[Revision],
) -> Result<NodeSort, GraphReadError> {
    let mut node_sort = Vec::with_capacity(node_restriction.len());
    let mut visited = HashSet::with_capacity(node_restriction.len());
    let mut to_visit = VecDeque::from_iter(roots.iter().cloned());
    let mut to_visit_set = HashSet::new();
    to_visit_set.extend(&to_visit);

    while let Some(rev) = to_visit.pop_front() {
        to_visit_set.remove(&rev);

        if !node_restriction.contains(&rev) {
            continue;
        }

        visited.insert(rev);
        node_sort.push(rev);

        for child in graph.ordered_children(rev)? {
            if visited.contains(child) || to_visit_set.contains(child) {
                continue;
            }

            if graph.parents(*child)?.into_iter().all(|p| {
                visited.contains(&p) || !node_restriction.contains(&p)
            }) {
                continue;
            }

            to_visit.push_back(*child);
            to_visit_set.insert(*child);
        }
    }

    Ok(node_sort)
}

#[cfg(test)]
mod tests {
    use crate::analytics::sorts::{index_map, stable_sort, NodeSort};
    use crate::graph::{
        Graph, GraphReadError, Revision, SizedGraph, StableOrderGraph,
    };
    use crate::testing::graph_io::read_graph;
    use crate::testing::ordering::NodeIDComparator;
    use std::collections::HashSet;
    use std::io::{BufRead, Cursor};

    #[test]
    fn test_index_map() {
        let res = index_map(&vec![1, 2, 3, 4, 0]);
        assert_eq!(res[&0], 4);
        assert_eq!(res[&1], 0);
        assert_eq!(res[&2], 1);
        assert_eq!(res[&3], 2);
        assert_eq!(res[&4], 3);
    }

    fn make_dummy_graph() -> impl Graph + SizedGraph + StableOrderGraph {
        // rnd2.graph
        let def = "
0000000000000000000000000000000000000100 0000000000000000000000000000000000000000 0000000000000000000000000000000000000000
0000000000000000000000000000000000000101 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000102 0000000000000000000000000000000000000100 0000000000000000000000000000000000000101
0000000000000000000000000000000000000103 0000000000000000000000000000000000000102 0000000000000000000000000000000000000100
0000000000000000000000000000000000000104 0000000000000000000000000000000000000103 0000000000000000000000000000000000000000
0000000000000000000000000000000000000105 0000000000000000000000000000000000000101 0000000000000000000000000000000000000100
0000000000000000000000000000000000000106 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000107 0000000000000000000000000000000000000101 0000000000000000000000000000000000000100
0000000000000000000000000000000000000108 0000000000000000000000000000000000000105 0000000000000000000000000000000000000103
0000000000000000000000000000000000000109 0000000000000000000000000000000000000107 0000000000000000000000000000000000000000
0000000000000000000000000000000000000110 0000000000000000000000000000000000000108 0000000000000000000000000000000000000109
0000000000000000000000000000000000000111 0000000000000000000000000000000000000108 0000000000000000000000000000000000000110
0000000000000000000000000000000000000112 0000000000000000000000000000000000000104 0000000000000000000000000000000000000000
0000000000000000000000000000000000000113 0000000000000000000000000000000000000102 0000000000000000000000000000000000000000
0000000000000000000000000000000000000114 0000000000000000000000000000000000000109 0000000000000000000000000000000000000106
0000000000000000000000000000000000000115 0000000000000000000000000000000000000102 0000000000000000000000000000000000000000
0000000000000000000000000000000000000116 0000000000000000000000000000000000000105 0000000000000000000000000000000000000000
0000000000000000000000000000000000000117 0000000000000000000000000000000000000105 0000000000000000000000000000000000000000
0000000000000000000000000000000000000118 0000000000000000000000000000000000000101 0000000000000000000000000000000000000101
0000000000000000000000000000000000000119 0000000000000000000000000000000000000101 0000000000000000000000000000000000000000
0000000000000000000000000000000000000120 0000000000000000000000000000000000000119 0000000000000000000000000000000000000105
0000000000000000000000000000000000000121 0000000000000000000000000000000000000109 0000000000000000000000000000000000000108
0000000000000000000000000000000000000122 0000000000000000000000000000000000000103 0000000000000000000000000000000000000000
0000000000000000000000000000000000000123 0000000000000000000000000000000000000106 0000000000000000000000000000000000000000
0000000000000000000000000000000000000124 0000000000000000000000000000000000000112 0000000000000000000000000000000000000115
0000000000000000000000000000000000000125 0000000000000000000000000000000000000118 0000000000000000000000000000000000000124
0000000000000000000000000000000000000126 0000000000000000000000000000000000000120 0000000000000000000000000000000000000000
0000000000000000000000000000000000000127 0000000000000000000000000000000000000100 0000000000000000000000000000000000000103
0000000000000000000000000000000000000128 0000000000000000000000000000000000000123 0000000000000000000000000000000000000120
0000000000000000000000000000000000000129 0000000000000000000000000000000000000103 0000000000000000000000000000000000000105
0000000000000000000000000000000000000130 0000000000000000000000000000000000000101 0000000000000000000000000000000000000000
0000000000000000000000000000000000000131 0000000000000000000000000000000000000100 0000000000000000000000000000000000000000
0000000000000000000000000000000000000132 0000000000000000000000000000000000000106 0000000000000000000000000000000000000100
0000000000000000000000000000000000000133 0000000000000000000000000000000000000107 0000000000000000000000000000000000000000
0000000000000000000000000000000000000134 0000000000000000000000000000000000000114 0000000000000000000000000000000000000000
0000000000000000000000000000000000000135 0000000000000000000000000000000000000107 0000000000000000000000000000000000000130
0000000000000000000000000000000000000136 0000000000000000000000000000000000000104 0000000000000000000000000000000000000000
0000000000000000000000000000000000000137 0000000000000000000000000000000000000121 0000000000000000000000000000000000000000
0000000000000000000000000000000000000138 0000000000000000000000000000000000000135 0000000000000000000000000000000000000000
0000000000000000000000000000000000000139 0000000000000000000000000000000000000131 0000000000000000000000000000000000000114
0000000000000000000000000000000000000140 0000000000000000000000000000000000000135 0000000000000000000000000000000000000135
0000000000000000000000000000000000000141 0000000000000000000000000000000000000140 0000000000000000000000000000000000000101
0000000000000000000000000000000000000142 0000000000000000000000000000000000000141 0000000000000000000000000000000000000102
0000000000000000000000000000000000000143 0000000000000000000000000000000000000142 0000000000000000000000000000000000000103
0000000000000000000000000000000000000144 0000000000000000000000000000000000000143 0000000000000000000000000000000000000104
0000000000000000000000000000000000000145 0000000000000000000000000000000000000144 0000000000000000000000000000000000000105
0000000000000000000000000000000000000146 0000000000000000000000000000000000000145 0000000000000000000000000000000000000106
0000000000000000000000000000000000000147 0000000000000000000000000000000000000146 0000000000000000000000000000000000000107
0000000000000000000000000000000000000148 0000000000000000000000000000000000000147 0000000000000000000000000000000000000108
0000000000000000000000000000000000000149 0000000000000000000000000000000000000148 0000000000000000000000000000000000000109
0000000000000000000000000000000000000150 0000000000000000000000000000000000000149 0000000000000000000000000000000000000110
0000000000000000000000000000000000000151 0000000000000000000000000000000000000150 0000000000000000000000000000000000000111
0000000000000000000000000000000000000152 0000000000000000000000000000000000000151 0000000000000000000000000000000000000112
0000000000000000000000000000000000000153 0000000000000000000000000000000000000152 0000000000000000000000000000000000000113
0000000000000000000000000000000000000154 0000000000000000000000000000000000000153 0000000000000000000000000000000000000114
0000000000000000000000000000000000000155 0000000000000000000000000000000000000154 0000000000000000000000000000000000000115
0000000000000000000000000000000000000156 0000000000000000000000000000000000000155 0000000000000000000000000000000000000116
0000000000000000000000000000000000000157 0000000000000000000000000000000000000156 0000000000000000000000000000000000000117
0000000000000000000000000000000000000158 0000000000000000000000000000000000000157 0000000000000000000000000000000000000118
0000000000000000000000000000000000000159 0000000000000000000000000000000000000158 0000000000000000000000000000000000000119
0000000000000000000000000000000000000160 0000000000000000000000000000000000000159 0000000000000000000000000000000000000120
0000000000000000000000000000000000000161 0000000000000000000000000000000000000160 0000000000000000000000000000000000000121
0000000000000000000000000000000000000162 0000000000000000000000000000000000000161 0000000000000000000000000000000000000122
0000000000000000000000000000000000000163 0000000000000000000000000000000000000162 0000000000000000000000000000000000000123
0000000000000000000000000000000000000164 0000000000000000000000000000000000000163 0000000000000000000000000000000000000124
0000000000000000000000000000000000000165 0000000000000000000000000000000000000164 0000000000000000000000000000000000000125
0000000000000000000000000000000000000166 0000000000000000000000000000000000000165 0000000000000000000000000000000000000126
0000000000000000000000000000000000000167 0000000000000000000000000000000000000166 0000000000000000000000000000000000000127
0000000000000000000000000000000000000168 0000000000000000000000000000000000000167 0000000000000000000000000000000000000128
0000000000000000000000000000000000000169 0000000000000000000000000000000000000168 0000000000000000000000000000000000000129
0000000000000000000000000000000000000170 0000000000000000000000000000000000000169 0000000000000000000000000000000000000130
0000000000000000000000000000000000000171 0000000000000000000000000000000000000170 0000000000000000000000000000000000000131
0000000000000000000000000000000000000172 0000000000000000000000000000000000000171 0000000000000000000000000000000000000132
0000000000000000000000000000000000000173 0000000000000000000000000000000000000172 0000000000000000000000000000000000000133
0000000000000000000000000000000000000174 0000000000000000000000000000000000000173 0000000000000000000000000000000000000134
0000000000000000000000000000000000000175 0000000000000000000000000000000000000174 0000000000000000000000000000000000000135
0000000000000000000000000000000000000176 0000000000000000000000000000000000000175 0000000000000000000000000000000000000136
0000000000000000000000000000000000000177 0000000000000000000000000000000000000176 0000000000000000000000000000000000000137
0000000000000000000000000000000000000178 0000000000000000000000000000000000000177 0000000000000000000000000000000000000138
0000000000000000000000000000000000000179 0000000000000000000000000000000000000178 0000000000000000000000000000000000000139
";
        let lines = Cursor::new(def).lines();
        read_graph::<_, NodeIDComparator>(lines).unwrap()
    }

    fn is_topological_order(
        graph: &impl Graph,
        node_sort: NodeSort,
    ) -> Result<bool, GraphReadError> {
        let mut discovered = HashSet::with_capacity(node_sort.len());

        for node in node_sort {
            if !discovered.insert(node) {
                return Ok(false); // found duplicate
            }

            for parent in graph.parents(node)? {
                if !discovered.contains(&parent) {
                    return Ok(false);
                }
            }
        }

        Ok(true)
    }

    #[test]
    fn test_stable_sort() {
        let graph = make_dummy_graph();
        let head = (graph.nb_nodes() - 1) as Revision;
        let node_sort = stable_sort(&graph, head).unwrap();
        assert_eq!(node_sort.len(), graph.rank(head).unwrap());
        assert!(is_topological_order(&graph, node_sort).unwrap());
    }

    /*
    #[test]
    fn test_postfix_depth_first_node_sort() {
        let graph = make_dummy_graph();
        let roots = &[0 as Revision];
        let node_sort = postfix_depth_first_node_sort(&graph, roots).unwrap();
        assert_eq!(node_sort.len(), graph.nb_nodes());
        assert!(is_topological_order(&graph, node_sort).unwrap());
    }
    */
}
