// SPDX-License-Identifier: GPL-2.0-or-later
//
// This software may be used and distributed according to the terms of the
// GNU General Public License version 2 or any later version.
//
// Copyright 2020  Pacien TRAN-GIRARD <pacien.trangirard@pacien.net>

use clap::arg_enum;
use std::fs::File;
use std::io::{self, BufRead};
use std::time::{SystemTime, UNIX_EPOCH};
use structopt::*;
use vcsgraph::analytics::counts::{aggregate_counts, AggregatedCounts};
use vcsgraph::analytics::sorts::{stable_sort, NodeSort};
use vcsgraph::analytics::sortsindicators::{sort_indicators, SortIndicators};
use vcsgraph::graph::{
    Graph, GraphReadError, LabelledGraph, NodeID, Revision, SizedGraph,
    StableOrderGraph,
};
use vcsgraph::testing::graph_io::read_graph;
use vcsgraph::testing::ordering::{
    MaxHardLeapsComparator, MaxMergeRankComparator, MaxRankComparator,
    MinHardLeapsComparator, MinMergeRankComparator, MinRankComparator,
    NodeComparator, NodeIDComparator, NodeIDComparatorReverse, HASH_SEED,
};

#[derive(StructOpt)]
#[structopt(
    name = "vcsgraph-analytics",
    about = r#"
    Runs various analytics functions on graphs piped to the standard input from:
    hg log --rev 'sort(all())' --template '{node} {p1.node} {p2.node}\n'
    "#
)]
struct CliArgs {
    #[structopt(
        long,
        short,
        possible_values = &ParentSorter::variants(),
        case_insensitive = true,
        help = "Primary criterion defining the order on parent revisions of \
merge nodes. The NodeID is used as a tie-breaker. This order is used by the \
stable sort."
    )]
    parent_sorter: ParentSorter,

    #[structopt(
        long,
        short = "s",
        default_value = "0",
        help = "Optional 8-bits unsigned int seed to use for hashing node IDs."
    )]
    hash_seed: u8,

    #[structopt(subcommand)]
    command: Command,
}

arg_enum! {
    #[derive(Debug)]
    enum ParentSorter {
        NodeID,
        NodeIDReverse,
        MinRank,
        MaxRank,
        MinMergeRank,
        MaxMergeRank,
        MinHardLeaps,
        MaxHardLeaps,
    }
}

#[derive(Debug, StructOpt)]
enum Command {
    #[structopt(about = "Prints counts and indicators as CSV. Columns are: \
start_timestamp,import_time_millis,sort_time_millis,parent_sorter,\
topological_sorter,hash_seed,graph_path,head_rev,root_node_count,\
linear_node_count,merge_node_count,hard_leap_count,soft_leap_count,\
exclusive_parts_sum,xc,xp,xa,xf")]
    Stats(StatsArguments),
}

#[derive(Debug, StructOpt)]
struct StatsArguments {
    #[structopt(help = "Path to the input graph.")]
    graph_path: String,

    #[structopt(
        possible_values = &TopologicalSorter::variants(),
        case_insensitive = true,
        help = "Sorter to use to produce a topological order of the nodes, \
on which the indicators will be computed."
    )]
    topological_sorter: TopologicalSorter,

    #[structopt(help = "Optional head revision. \
Defaults to the last revision in the input graph file if not explicitly given. \
The considered graph is restricted to the induced subgraph of the ancestors \
of this head.")]
    head: Option<NodeID>,
}

arg_enum! {
    #[derive(Debug)]
    enum TopologicalSorter {
        StableSort,
    }
}

fn sort_nodes(
    topological_sorter: &TopologicalSorter,
    graph: &(impl Graph + SizedGraph + StableOrderGraph),
    head_rev: Revision,
) -> Result<NodeSort, GraphReadError> {
    match topological_sorter {
        TopologicalSorter::StableSort => stable_sort(graph, head_rev),
    }
}

fn print_stats(
    start_timestamp: u128,
    import_time_millis: u128,
    sort_time_millis: u128,
    parent_sorter: ParentSorter,
    topological_sorter: TopologicalSorter,
    graph_path: String,
    head_rev: Revision,
    counts: AggregatedCounts,
    indicators: SortIndicators,
) {
    // quick and dirty CSV output because the `csv` crate has some conflicting
    // dependencies
    print!("{},", start_timestamp);
    print!("{},", import_time_millis);
    print!("{},", sort_time_millis);
    print!("{:#?},", parent_sorter);
    print!("{:#?},", topological_sorter);
    print!("{},", unsafe { HASH_SEED.to_string() });
    print!("{},", graph_path);
    print!("{},", head_rev);
    print!("{},", counts.root_node_count);
    print!("{},", counts.linear_node_count);
    print!("{},", counts.merge_node_count);
    print!("{},", counts.hard_leap_count);
    print!("{},", counts.soft_leap_count);
    print!("{},", counts.exclusive_parts_sum);
    print!("{},", indicators.xc);
    print!("{},", indicators.xp);
    print!("{},", indicators.xa);
    println!("{},", indicators.xf);
}

fn run_stats<C: NodeComparator>(
    graph_path: String,
    head: Option<NodeID>,
    parent_sorter: ParentSorter,
    topological_sorter: TopologicalSorter,
) {
    let start_time = SystemTime::now();

    let graph_file = File::open(&graph_path).unwrap();
    let graph_input = io::BufReader::new(graph_file).lines();
    let graph = read_graph::<_, C>(graph_input).unwrap();
    let head_rev = head.map_or(graph.nb_nodes() as Revision - 1, |head_id| {
        graph.rev_of_node_id(head_id).unwrap()
    });
    let import_time_millis = start_time.elapsed().unwrap().as_millis();

    let node_sort = sort_nodes(&topological_sorter, &graph, head_rev).unwrap();
    let sort_time_millis =
        start_time.elapsed().unwrap().as_millis() - import_time_millis;

    let counts = aggregate_counts(&graph, &[head_rev]).unwrap();
    let indicators = sort_indicators(&graph, &node_sort).unwrap();

    let start_timestamp =
        start_time.duration_since(UNIX_EPOCH).unwrap().as_millis();

    print_stats(
        start_timestamp,
        import_time_millis,
        sort_time_millis,
        parent_sorter,
        topological_sorter,
        graph_path,
        head_rev,
        counts,
        indicators,
    );
}

fn main() {
    let args: CliArgs = CliArgs::from_args();

    unsafe {
        HASH_SEED = args.hash_seed;
    }

    match args.command {
        Command::Stats(StatsArguments {
            graph_path,
            topological_sorter,
            head,
        }) => match args.parent_sorter {
            ParentSorter::NodeID => run_stats::<NodeIDComparator>(
                graph_path,
                head,
                args.parent_sorter,
                topological_sorter,
            ),
            ParentSorter::NodeIDReverse => {
                run_stats::<NodeIDComparatorReverse>(
                    graph_path,
                    head,
                    args.parent_sorter,
                    topological_sorter,
                )
            }
            ParentSorter::MinRank => run_stats::<MinRankComparator>(
                graph_path,
                head,
                args.parent_sorter,
                topological_sorter,
            ),
            ParentSorter::MaxRank => run_stats::<MaxRankComparator>(
                graph_path,
                head,
                args.parent_sorter,
                topological_sorter,
            ),
            ParentSorter::MinMergeRank => run_stats::<MinMergeRankComparator>(
                graph_path,
                head,
                args.parent_sorter,
                topological_sorter,
            ),
            ParentSorter::MaxMergeRank => run_stats::<MaxMergeRankComparator>(
                graph_path,
                head,
                args.parent_sorter,
                topological_sorter,
            ),
            ParentSorter::MaxHardLeaps => run_stats::<MaxHardLeapsComparator>(
                graph_path,
                head,
                args.parent_sorter,
                topological_sorter,
            ),
            ParentSorter::MinHardLeaps => run_stats::<MinHardLeapsComparator>(
                graph_path,
                head,
                args.parent_sorter,
                topological_sorter,
            ),
        },
    }
}
