//! V Compiler Tools
//! 
//! Useful tools for making simple compilers

/// Common compiler io functions
pub mod io {
    use std::io::prelude::*;

    /// Reads the file specified in the first command-line argument `argv[1]` and returns the source.
    /// 
    /// If the arguments are invalid, the file is not found, or any other file error occurs this function will print the error and exit the program.
    /// 
    /// Example command-line input:
    /// 
    /// `thing.exe input.txt output.txt`
    /// 
    /// This function would then return the source of input.txt as a String.
    /// 
    /// 
    /// # Code Example
    /// 
    /// ```
    /// let source = vcompiler_tools::io::read_file_argv();
    /// let mut compiled = String::new();
    ///     
    /// //do stuff with source and output to compiled
    /// compiled.push_str(source.as_str());
    /// 
    /// vcompiler_tools::io::write_file_argv(compiled);
    /// ```
    pub fn read_file_argv() -> String {
        let argv: Vec<String> = std::env::args().collect();
        
        if argv.len() != 3 {
            println!("Arguments: <input_file> <output_file>");
            std::process::exit(0);
        }
        let filename = &argv[1];

        let file_result = std::fs::read_to_string(filename);
        let source = match file_result {
            Ok(source) => source,
            Err(error) => match error.kind() {
                std::io::ErrorKind::NotFound => {
                    println!("File Not Found: {}", filename);
                    std::process::exit(0);
                },
                _ => {
                    println!("File Error: {}", error);
                    std::process::exit(0);
                }
            }
        };

        return source;
    }

    /// Writes a String to the file specified in the second command-line argument `argv[2]`.
    /// 
    /// This function should be used with `read_file_argv`
    /// 
    /// If a file error occurs this function will print the error and exit the program.
    /// 
    /// Example command-line input:
    /// 
    /// `thing.exe input.txt output.txt`
    /// 
    /// This function would then write the `data` to the file output.txt
    pub fn write_file_argv(data: String) {
        let argv: Vec<String> = std::env::args().collect();

        if argv.len() != 3 {
            panic!("error argv not length 3, you might have forgot to run read_file_argv");
        }
        let filename = &argv[2];

        let file_result = std::fs::File::create(filename);
        let mut file = match file_result {
            Ok(file) => file,
            Err(error) => {
                println!("Error while writing to file: {}", error);
                std::process::exit(0);
            }
        };
        let write_result = file.write_all(data.as_bytes());
        match write_result {
            Ok(_) => {},
            Err(error) => {
                println!("Error while writing to file: {}", error);
                std::process::exit(0);
            }
        }
        
    }

}