use crate::data::*;
use miette::Diagnostic;
use thiserror::Error;

#[derive(Debug, Error, Diagnostic)]
pub enum BspError {
    #[error("unexpected magic numbers or version")]
    #[diagnostic(help("Ensure the loaded file is a valve bsp"))]
    UnexpectedHeader(Header),
    #[error("bsp lump is out of bounds of the bsp file")]
    LumpOutOfBounds(LumpEntry),
    #[error("unexpected length of uncompressed lump, got {got} but expected {expected}")]
    UnexpectedUncompressedLumpSize { got: u32, expected: u32 },
    #[error("error while decompressing lump")]
    LumpDecompressError(lzma_rs::error::Error),
    #[error("io error while reading data: {0}")]
    IO(#[from] std::io::Error),
    #[error(transparent)]
    String(#[from] StringError),
}

impl From<binrw::Error> for BspError {
    fn from(e: binrw::Error) -> Self {
        use binrw::Error;

        // only a few error types should be generated by our code
        match e {
            Error::Io(e) => BspError::IO(e),
            Error::Custom { err, .. } => {
                if let Ok(string_error) = err.downcast::<StringError>() {
                    BspError::String(*string_error)
                } else {
                    panic!("unexpected custom error")
                }
            }
            e => panic!("unexpected no variable match: {:?}", e),
        }
    }
}

impl From<lzma_rs::error::Error> for BspError {
    fn from(e: lzma_rs::error::Error) -> Self {
        use lzma_rs::error::Error;

        match e {
            Error::IoError(e) => BspError::IO(e),
            e => BspError::LumpDecompressError(e),
        }
    }
}

#[derive(Debug, Error, Diagnostic)]
pub enum StringError {
    #[error(transparent)]
    NonUTF8(#[from] std::str::Utf8Error),
    #[error("String is not null-terminated")]
    NotNullTerminated,
}
