//! This crate is probably not what you hoped it was -- in fact, it's probably
//! exactly what you feared.  Rather than integrate against VirtualBox's COM
//! interfaces it will call the command line tools and parse their outputs.
//!
//! Perhaps not surprisingly, this crate originally began as a bash script and
//! slowly morphed into what it is today.
//!
//! # Examples
//!
//! Terminate a virtual machine named _myvm_ and revert it to a snapshot named
//! _mysnap_.
//!
//! ```no_run
//! use std::time::Duration;
//! use vboxhelper::*;
//!
//! let vm = "myvm".parse::<VmId>().unwrap();
//! controlvm::kill(&vm).unwrap();
//!
//! let ten_seconds = Duration::new(10, 0);
//! wait_for_croak(&vm, Some((ten_seconds, TimeoutAction::Error)));
//!
//! // revert to a snapshot
//! let snap = "mysnap".parse::<snapshot::SnapshotId>().unwrap();
//! snapshot::restore(&vm, Some(snap)).unwrap();
//! ```
//!
//! # VirtualBox Versions
//! This crate will generally attempt to track the latest version of
//! VirtualBox.
//!
//!
//! # Environment variables
//! In some cases vboxhelper will execute external commands with no intention
//! of parsing their output.  In these situations vboxhelper will by default
//! consume the child processes' output.  This behavior can be changed by
//! setting the `VBOXHELPER_VERBOSE` environment variable; setting its value to
//! `"1"` will cause the library to output to be directed to stdout and stderr
//! respectively.
//!
//! If the variable `VBOXHELPER_LOGS` is set to a valid directory vboxhelper
//! may use it to store log files.

mod platform;
mod strutils;
mod utils;

pub mod controlvm;
pub mod err;
pub mod nics;
pub mod snapshot;
pub mod storage;
pub mod vmid;

use std::collections::HashMap;
use std::path::PathBuf;
use std::process::Command;
use std::thread;
use std::time::{Duration, Instant};

use regex::Regex;

pub use err::Error;

use strutils::{buf_to_strlines, EmptyLine};

pub use vmid::VmId;


/// Control whether running a VM headless should block until VM terminates or
/// if it should return as soon as the VM has been started in the background.
#[derive(Copy, Clone)]
pub enum Headless {
  /// The VM will start and return immediately while the VM is running in the
  /// background.
  Detached,

  /// If the VM is started successfully the call will block until the VM
  /// terminates.
  Blocking
}


/// Control whether a virtual machine is run with a GUI or in Headless mode.
#[derive(Copy, Clone)]
pub enum RunContext {
  /// The virtual machine will run as a application on a Desktop GUI.
  GUI,

  /// The virtual machine will run without a GUI.
  Headless(Headless)
}


pub enum TimeoutAction {
  /// If the operation times out, then return an error.
  Error,

  /// If the operation times out, then kill the virtual machine
  Kill
}


#[derive(Copy, Clone)]
pub enum Shutdown {
  /// Just power off virtual machine.
  PowerOff,

  /// Send an ACPI power off signal.
  AcpiPowerOff,

  /// Save the virtual machine's state and then power off.
  SaveState
}


pub fn have_vm(vid: &VmId) -> Result<bool, Error> {
  let lst = get_vm_list()?;

  for (name, uuid) in lst {
    match vid {
      VmId::Name(nm) => {
        if name == *nm {
          return Ok(true);
        }
      }
      VmId::Uuid(u) => {
        if uuid == *u {
          return Ok(true);
        }
      }
    }
  }
  Ok(false)
}


pub fn is_vm_running(vid: &VmId) -> Result<bool, Error> {
  let lst = get_running_vms_list()?;

  for (name, uuid) in lst {
    match vid {
      VmId::Name(nm) => {
        if name == *nm {
          return Ok(true);
        }
      }
      VmId::Uuid(u) => {
        if uuid == *u {
          return Ok(true);
        }
      }
    }
  }
  Ok(false)
}


/// Get a list of virtual machines.
pub fn get_vm_list() -> Result<Vec<(String, uuid::Uuid)>, Error> {
  let mut cmd = Command::new(platform::get_cmd("VBoxManage"));
  cmd.args(&["list", "vms"]);

  let output = match cmd.output() {
    Ok(out) => out,
    Err(_) => {
      return Err(Error::FailedToExecute(format!("{:?}", cmd)));
    }
  };

  if !output.status.success() {
    return Err(Error::CommandFailed(format!("{:?}", cmd), output));
  }

  let lines = buf_to_strlines(&output.stdout, EmptyLine::Ignore);

  let mut out = Vec::new();

  for line in lines {
    // Make sure first character is '"'
    match line.find('"') {
      Some(idx) => {
        if idx != 0 {
          continue;
        }
      }
      None => continue
    }

    // Find last '"'
    let idx = match line.rfind('"') {
      Some(idx) => {
        if idx == 0 {
          continue;
        }
        idx
      }
      None => continue
    };


    let idx_ub = match line.rfind('{') {
      Some(idx) => {
        if idx == 0 {
          continue;
        }
        idx
      }
      None => continue
    };

    let idx_ue = match line.rfind('}') {
      Some(idx) => {
        if idx == 0 {
          continue;
        }
        idx
      }
      None => continue
    };

    let name = &line[1..idx];
    let uuidstr = &line[(idx_ub + 1)..idx_ue];
    let u = match uuid::Uuid::parse_str(uuidstr) {
      Ok(u) => u,
      Err(_) => continue
    };
    out.push((name.to_string(), u));
  }

  Ok(out)
}


/// Get a list of all running virtual machines.
pub fn get_running_vms_list() -> Result<Vec<(String, uuid::Uuid)>, Error> {
  let mut cmd = Command::new(platform::get_cmd("VBoxManage"));
  cmd.args(&["list", "runningvms"]);

  let output = match cmd.output() {
    Ok(out) => out,
    Err(_) => {
      return Err(Error::FailedToExecute(format!("{:?}", cmd)));
    }
  };

  if !output.status.success() {
    return Err(Error::CommandFailed(format!("{:?}", cmd), output));
  }

  let lines = buf_to_strlines(&output.stdout, EmptyLine::Ignore);

  let mut out = Vec::new();

  for line in lines {
    // Make sure first character is '"'
    match line.find('"') {
      Some(idx) => {
        if idx != 0 {
          continue;
        }
      }
      None => continue
    }

    // Find last '"'
    let idx = match line.rfind('"') {
      Some(idx) => {
        if idx == 0 {
          continue;
        }
        idx
      }
      None => continue
    };


    let idx_ub = match line.rfind('{') {
      Some(idx) => {
        if idx == 0 {
          continue;
        }
        idx
      }
      None => continue
    };

    let idx_ue = match line.rfind('}') {
      Some(idx) => {
        if idx == 0 {
          continue;
        }
        idx
      }
      None => continue
    };

    let name = &line[1..idx];
    let uuidstr = &line[(idx_ub + 1)..idx_ue];
    let u = match uuid::Uuid::parse_str(uuidstr) {
      Ok(u) => u,
      Err(_) => {
        return Err(Error::BadFormat(
          "Unable to parse output UUID.".to_string()
        ));
      }
    };
    out.push((name.to_string(), u));
  }

  Ok(out)
}


/// Get the process id of the guest virtual machine.
///
/// If the virtual machine doesn't exist or isn't running this function will
/// return [`Error::InvalidState`].
///
/// The PID is determined by searching for a "Process ID:" string in the
/// virtual machine's log (no, I'm not kidding, it really does).  If this
/// entry can not be found in the log the function will return
/// [`Error::Missing`].  Note that this doesn't necessarily mean that
/// something is wrong; it can simply mean that VirtualBox hasn't had time to
/// write the process id log entry yet, though empirically it would seem that
/// it writes it pretty soon into the startup process.  It is recommended that
/// the application wait a few seconds between start of the virtual machine and
/// calling this function, and also that the call be retried (with a delay
/// between call) if `Error::Missing` is returned.
///
/// Use [`try_get_pid()`] to wrap some retry logic around this function.
pub fn get_pid(vid: &VmId) -> Result<u32, Error> {
  let is_running = is_vm_state(vid, VmState::Running)?;
  if !is_running {
    return Err(Error::invalid_state("VM doesn't exist or isn't running"));
  }

  let mut cmd = Command::new(platform::get_cmd("VBoxManage"));
  cmd.arg("showvminfo");
  cmd.arg(vid.to_string());
  cmd.arg("--log");
  cmd.arg("0");

  let output = cmd.output().expect("Failed to execute VBoxManage");
  let lines = strutils::buf_to_strlines(&output.stdout, EmptyLine::Ignore);
  let mut lines = lines.iter();

  //00:00:01.510420 Process ID: 1029
  let re_pid =
    Regex::new(r#"^\d{2}:\d{2}:\d{2}\.\d+ Process ID: (?P<pid>\d+)$"#)
      .unwrap();

  let mut pid = None;
  while let Some(line) = lines.next() {
    let line = line.trim_end();
    if let Some(cap) = re_pid.captures(&line) {
      pid = Some(cap[1].parse::<u32>().unwrap());
      break;
    }
  }

  pid.ok_or(Error::missing("Unable to find Process ID."))
}


/// Wrapper method around [`get_pid()`] to retry getting the guest's PID.
///
/// ```no_run
/// use std::time::Duration;
/// use vboxhelper::{try_get_pid, VmId};
/// fn impatient() {
///   let vmid = VmId::from("myvm");
///   let max_retries = 5;
///   let two_seconds = Duration::new(2, 0);
///   let pid = try_get_pid(&vmid, max_retries, two_seconds).unwrap();
///   println!("Guest's process id in host: {}", pid);
/// }
/// ```
pub fn try_get_pid(
  vid: &VmId,
  max_retries: usize,
  delay: Duration
) -> Result<u32, Error> {
  let mut remain = max_retries;
  let pid = loop {
    match get_pid(vid) {
      Ok(pid) => break pid,
      Err(Error::Missing(s)) => {
        if remain == 0 {
          // Exhaused all tries, abort with the original error.
          return Err(Error::Missing(s));
        }
        // Assume that the the process id will show up later.  Delay for the
        // requested amount of time and then try again.
        remain -= 1;
        thread::sleep(delay);
      }
      Err(e) => {
        return Err(e);
      }
    }
  };

  Ok(pid)
}


/// Get information about a virtual machine as a map.
pub fn get_vm_info_map(vid: &VmId) -> Result<HashMap<String, String>, Error> {
  let mut cmd = Command::new(platform::get_cmd("VBoxManage"));
  cmd.arg("showvminfo");
  cmd.arg(vid.to_string());
  cmd.arg("--machinereadable");

  let output = cmd.output().expect("Failed to execute VBoxManage");

  let lines = strutils::buf_to_strlines(&output.stdout, EmptyLine::Ignore);

  let mut map = HashMap::new();

  // multiline
  //let re_ml1 =
  // Regex::new(r#"^(?P<key>[^"=]+)="(?P<val>[^"=]*)"$"#).unwrap();
  // let re_ml1 =
  // Regex::new(r#"^"(?P<key>[^"=]+)"="(?P<val>[^"=]*)"$"#).unwrap();

  // Capture foo="bar" -> foo=bar
  // This appears to be most common.
  let re1 = Regex::new(r#"^(?P<key>[^"=]+)="(?P<val>[^"=]*)"$"#).unwrap();

  // Capture "foo"="bar" -> foo=bar
  let re2 = Regex::new(r#"^"(?P<key>[^"=]+)"="(?P<val>[^"=]*)"$"#).unwrap();

  // foo=bar -> foo=bar
  let re3 = Regex::new(r#"^(?P<key>[^"=]+)=(?P<val>[^"=]*)$"#).unwrap();

  //let re = Regex::new(r#"^"?(?P<key>[^"=]+)"?="?(?P<val>[^"=]*)"?$"#).
  // unwrap();


  // ToDo: Handle multiline entires, like descriptions
  let mut lines = lines.iter();
  while let Some(line) = lines.next() {
    //println!("line: {}", line);

    let line = line.trim_end();
    let cap = if let Some(cap) = re1.captures(&line) {
      Some(cap)
    } else if let Some(cap) = re2.captures(&line) {
      Some(cap)
    } else if let Some(cap) = re3.captures(&line) {
      Some(cap)
    } else {
      dbg!(format!("Ignored line: {}", line));
      None
    };

    if let Some(cap) = cap {
      map.insert(cap[1].to_string(), cap[2].to_string());
    }
  }

  Ok(map)
}


#[derive(PartialEq, Eq, Copy, Clone)]
/// VirtualBox virtual machine states.
pub enum VmState {
  /// This isn't actually a VirtualBox virtual machine state; it's used as a
  /// placeholder if an unknown state is encountered.
  Unknown,

  /// The virtual machine is powered off.
  PowerOff,

  /// The virtual machine is currently starting up.
  Starting,

  /// The virtual machine is currently up and running.
  Running,

  /// The virtual machine is currently paused.
  Paused,

  /// The virtual machine is currently shutting down.
  Stopping
}

impl From<&str> for VmState {
  fn from(s: &str) -> Self {
    match s {
      "poweroff" => VmState::PowerOff,
      "starting" => VmState::Starting,
      "running" => VmState::Running,
      "paused" => VmState::Paused,
      "stopping" => VmState::Stopping,
      _ => VmState::Unknown
    }
  }
}

impl From<&String> for VmState {
  fn from(s: &String) -> Self {
    match s.as_ref() {
      "poweroff" => VmState::PowerOff,
      "starting" => VmState::Starting,
      "running" => VmState::Running,
      "paused" => VmState::Paused,
      "stopping" => VmState::Stopping,
      _ => VmState::Unknown
    }
  }
}


/// A structured representation of a virtual machine's state and configuration.
pub struct VmInfo {
  pub shares_map: HashMap<String, PathBuf>,
  pub shares_list: Vec<(String, PathBuf)>,
  pub state: VmState,
  pub snapshots: Option<snapshot::Snapshots>,
  pub nics: Vec<nics::NICInfo>
}


/// Get structured information about a virtual machine.
pub fn get_vm_info(vid: &VmId) -> Result<VmInfo, Error> {
  let map = get_vm_info_map(vid)?;

  let mut shares_list = Vec::new();
  let mut shares_map = HashMap::new();

  //
  // Parse shares
  //
  let mut idx = 1;
  loop {
    let name_key = format!("SharedFolderNameMachineMapping{}", idx);
    let path_key = format!("SharedFolderPathMachineMapping{}", idx);

    let name = match map.get(&name_key) {
      Some(nm) => nm.clone(),
      None => break
    };
    let pathname = match map.get(&path_key) {
      Some(pn) => PathBuf::from(pn),
      None => break
    };

    shares_map.insert(name.clone(), pathname.clone());
    shares_list.push((name, pathname));

    idx += 1;
  }

  //
  // Get VM State
  //
  let state = match map.get("VMState") {
    Some(s) => VmState::from(s),
    None => VmState::Unknown
  };

  //
  // Parse snapshots
  //
  let snaps = snapshot::get_from_map(&map)?;

  //
  // Parse NICs
  //
  let nics = nics::get_from_map(&map)?;

  Ok(VmInfo {
    state,
    shares_map,
    shares_list,
    snapshots: snaps,
    nics
  })
}


/// Check whether a virtual machine is currently in a certain state.
pub fn is_vm_state(vid: &VmId, state: VmState) -> Result<bool, Error> {
  let vmi = get_vm_info(vid)?;
  Ok(vmi.state == state)
}


/// Wait for a virtual machine to enter an expected state.
///
/// ```no_run
/// use std::time::Duration;
/// use vboxhelper::{wait_for_state, VmId, VmState};
///
/// let vmid = VmId::from("myvm");
/// let max_rechecks = 4;
/// let four_seconds = Duration::new(4, 0);
/// wait_for_state(&vmid, VmState::Running, max_rechecks, four_seconds);
/// ```
pub fn wait_for_state(
  vid: &VmId,
  state: VmState,
  max_rechecks: usize,
  recheck_delay: Duration
) -> Result<(), Error> {
  //let start = Instant::now();
  let mut remain = max_rechecks;
  loop {
    let is_expected_state = is_vm_state(vid, state)?;
    if is_expected_state {
      break;
    }

    if remain == 0 {
      return Err(Error::Timeout);
    }
    remain -= 1;

    thread::sleep(recheck_delay);
  }
  Ok(())
}


/// Wait for a virtual machine to self-terminate.
///
/// The caller can choose to pass a timeout and what action should be taken if
/// the operation times out.  If the timeout occurs the caller can choose
/// whether to return a timeout error or kill the virtual machine.
///
/// ```no_run
/// use std::time::Duration;
/// use vboxhelper::{TimeoutAction, wait_for_croak, VmId};
/// fn impatient() {
///   let twenty_seconds = Duration::new(20, 0);
///   let vmid = VmId::from("myvm");
///   wait_for_croak(&vmid, Some((twenty_seconds, TimeoutAction::Kill)));
/// }
/// ```
///
/// This function polls `is_vm_state()` which calls `get_vm_info()`.  A very
/// sad state of affairs.  :(
pub fn wait_for_croak(
  vid: &VmId,
  timeout: Option<(Duration, TimeoutAction)>
) -> Result<(), Error> {
  let start = Instant::now();
  loop {
    let poweroff = is_vm_state(vid, VmState::PowerOff)?;
    if poweroff {
      break;
    }
    if let Some((ref max_dur, ref action)) = timeout {
      let duration = start.elapsed();
      if duration > *max_dur {
        match action {
          TimeoutAction::Error => return Err(Error::Timeout),
          TimeoutAction::Kill => {
            controlvm::kill(vid)?;

            // ToDo: Give it some time to croak.  If it doesn't, then return
            //       an "uncroakable vm" error.
            break;
          }
        }
      }
    }

    // Why 11?  Because it's more than 10, and it's a prime.  I don't know why
    // 11 is a prime -- ask the universe.
    let eleven_secs = Duration::from_secs(11);
    thread::sleep(eleven_secs);
  }
  Ok(())
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
