//* # vaultrs
//*
//* > A rust crate for interacting with the Hashicorp Vault API
//*
//* This crate encompasses functions for interacting with the HTTP API available on
//* [Hashicorp Vault](https://www.vaultproject.io/) servers. It uses
//* [rustify](https://github.com/jmgilman/rustify) in order to construct accurate
//* representations of each of the endpoints available with the API. It then wraps
//* these into more usable functions intended to be consumed by users of this crate.
//*
//* The following functionality is currently supported:
//*
//* * [KV Secrets Engine V2](https://www.vaultproject.io/docs/secrets/kv/kv-v2)
//* * [PKI Secrets Engine](https://www.vaultproject.io/docs/secrets/pki)
//* * [Response Wrapping](https://www.vaultproject.io/docs/concepts/response-wrapping)
//*
//* ## Installation
//*
//* ```
//* cargo add vaultrs
//* ```
//*
//* ## Usage
//*
//* ### Basic
//*
//* The client is used to configure the connection to Vault and is required to be
//* passed to all API calls for execution.
//*
//* ```rust
//* use vaultrs::client::{VaultClient, VaultClientSettingsBuilder};
//*
//* // Create a client
//* let client = VaultClient::new(
//*     VaultClientSettingsBuilder::default()
//*         .address("https://127.0.0.1:8200")
//*         .token("TOKEN")
//*         .build()
//*         .unwrap()
//* ).unwrap();
//* ```
//*
//* ### Secrets
//*
//* The library currently supports all operations available for version 2 of the
//* key/value store.
//*
//* ```rust
//* use vaultrs::kv2;
//*
//* // Create and read secrets
//* struct MySecret {
//*     key: String,
//*     password: String,
//* }
//*
//* let secret = MySecret {
//*     key: "super".to_string(),
//*     password: "secret".to_string(),
//* };
//* kv2::set(
//*     &client,
//*     "secret",
//*     "mysecret",
//*     &secret,
//* );
//*
//* let secret = kv2::read::<MySecret>(&client, "secret" "mysecret").unwrap();
//* println!("{}", secret.password) // "secret"
//* ```
//*
//* ### PKI
//*
//* The library currently supports all operations available for the PKI secrets
//* engine.
//*
//* ```rust
//* use vaultrs::pki::cert;
//* use vaultrs::api::pki::requests::GenerateCertificateRequest;
//*
//* // Generate a certificate using the PKI backend
//* let cert = cert::generate(
//*     &client,
//*     "pki",
//*     "my_role",
//*     Some(GenerateCertificateRequest::builder().common_name("test.com")),
//* ).unwrap();
//* println!("{}", cert.certificate) // "{PEM encoded certificate}"
//* ```
//*
//* ### Wrapping
//*
//* All requests implement the ability to be
//* [wrapped](https://www.vaultproject.io/docs/concepts/response-wrapping). These
//* can be passed in your application internally before being unwrapped.
//*
//* ```rust
//* use vaultrs::api::ResponseWrapper;
//* use vaulrs::api::sys::requests::ListMountsRequest;
//*
//* let endpoint = ListMountsRequest::builder().build().unwrap();
//* let wrap_resp = endpoint.wrap(&client); // Wrapped response
//* assert!(wrap_resp.is_ok());
//*
//* let wrap_resp = wrap_resp.unwrap(); // Unwrap Result<>
//* let info = wrap_resp.lookup(&client); // Check status of this wrapped response
//* assert!(info.is_ok());
//*
//* let unwrap_resp = wrap_resp.unwrap(&client); // Unwrap the response
//* assert!(unwrap_resp.is_ok());
//*
//* let info = wrap_resp.lookup(&client); // Error: response already unwrapped
//* assert!(info.is_err());
//* ```
//*
//* ## Error Handling
//*
//* All errors generated by this crate are wrapped in the `ClientError` enum
//* provided by the crate. API warninings are automatically captured via `log` and
//* API errors are captured and returned as their own variant. Connection related
//* errors from `rusify` are wrapped and returned as a single variant.
#[macro_use]
extern crate derive_builder;

pub mod api;
pub mod client;
pub mod error;
pub mod kv2;
pub mod pki;
pub mod sys;
